package com.xforceplus.config;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.core.json.JsonReadFeature;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.json.JsonMapper;
import com.fasterxml.jackson.databind.jsontype.impl.LaissezFaireSubTypeValidator;
import com.fasterxml.jackson.datatype.hibernate5.Hibernate5Module;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.xforceplus.event.publisher.MessagePublisher;
import com.xforceplus.event.publisher.RedisMessagePublisher;
import com.xforceplus.event.subscriber.RefreshingStrategyMessageSubscriber;
import com.xforceplus.security.strategy.event.RefreshingStrategyCacheEvent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.cache.Cache;
import org.springframework.cache.CacheManager;
import org.springframework.cache.annotation.CachingConfigurerSupport;
import org.springframework.cache.annotation.EnableCaching;
import org.springframework.cache.caffeine.CaffeineCacheManager;
import org.springframework.cache.interceptor.CacheErrorHandler;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.ApplicationEventPublisherAware;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.data.redis.cache.RedisCacheConfiguration;
import org.springframework.data.redis.cache.RedisCacheManager;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.connection.lettuce.LettuceConnectionFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.data.redis.listener.ChannelTopic;
import org.springframework.data.redis.listener.RedisMessageListenerContainer;
import org.springframework.data.redis.listener.adapter.MessageListenerAdapter;
import org.springframework.data.redis.serializer.GenericJackson2JsonRedisSerializer;
import org.springframework.data.redis.serializer.RedisSerializationContext;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.data.redis.serializer.StringRedisSerializer;

import java.time.Duration;


/**
 * 缓存配置
 *
 * @author geewit
 */
@SuppressWarnings("all")
@AutoConfigureBefore({CacheInitConfig.class})
@Configuration
@EnableCaching
public class CachingConfig extends CachingConfigurerSupport implements ApplicationEventPublisherAware {
    private final static Logger logger = LoggerFactory.getLogger(CachingConfig.class);

    private final LettuceConnectionFactory lettuceConnectionFactory;

    private final ObjectMapper objectMapper;

    /**
     * redis缓存失效时间, 默认 24小时(1天)
     */
    @Value("${xforce.tenant.cache.redis.time-live-minutes:600}")
    private Long redisTimeLiveMinutes;


    /**
     * caffeine本地缓存失效时间, 默认 3 秒
     */
    @Value("${xforce.tenant.cache.caffeine.time-live-seconds:3}")
    private Long caffeineTimeLiveSeconds;

    /**
     * caffeine本地缓存失效时间, 默认 3 秒
     */
    @Value("${xforce.tenant.cache.caffeine.maximum-size:2000}")
    private Long caffeineMaximumSize;

    public CachingConfig(LettuceConnectionFactory lettuceConnectionFactory, ObjectMapper objectMapper) {
        this.lettuceConnectionFactory = lettuceConnectionFactory;
        this.objectMapper = objectMapper;
    }

    private ApplicationEventPublisher applicationEventPublisher;

    @Override
    public void setApplicationEventPublisher(ApplicationEventPublisher applicationEventPublisher) {
        this.applicationEventPublisher = applicationEventPublisher;
    }

    @Bean("caffeineCacheManager")
    public CaffeineCacheManager caffeineCacheManager() {
        CaffeineCacheManager caffeineCacheManager = new CaffeineCacheManager();
        caffeineCacheManager.setAllowNullValues(true);
        caffeineCacheManager.setCaffeine(Caffeine.newBuilder().recordStats()
                .expireAfterWrite(Duration.ofMinutes(caffeineTimeLiveSeconds))
                .maximumSize(caffeineMaximumSize));
        return caffeineCacheManager;
    }

    @Bean("redisCacheManagerBuilder")
    public RedisCacheManager.RedisCacheManagerBuilder redisCacheManagerBuilder() {
        RedisCacheConfiguration redisCacheConfiguration = RedisCacheConfiguration.defaultCacheConfig()
                .disableCachingNullValues()
                .entryTtl(java.time.Duration.ofMinutes(redisTimeLiveMinutes))
                .computePrefixWith(cacheName -> "uc:" + cacheName + ":")
                .serializeKeysWith(RedisSerializationContext.SerializationPair.fromSerializer(this.keySerializer()))
                .serializeValuesWith(RedisSerializationContext.SerializationPair.fromSerializer(this.redisSerializer()));
        redisCacheConfiguration.usePrefix();

        RedisCacheManager.RedisCacheManagerBuilder redisCacheManagerBuilder = RedisCacheManager.RedisCacheManagerBuilder
                .fromConnectionFactory(lettuceConnectionFactory)
                .cacheDefaults(redisCacheConfiguration);
        return redisCacheManagerBuilder;
    }

    private RedisSerializer<String> keySerializer() {
        return new StringRedisSerializer();
    }


    private GenericJackson2JsonRedisSerializer redisSerializer() {
        ObjectMapper objectMapper = this.objectMapper.copy();
        Hibernate5Module hibernate5Module = new Hibernate5Module();
        hibernate5Module.disable(Hibernate5Module.Feature.USE_TRANSIENT_ANNOTATION);
        hibernate5Module.enable(Hibernate5Module.Feature.REPLACE_PERSISTENT_COLLECTIONS);
        hibernate5Module.enable(Hibernate5Module.Feature.SERIALIZE_IDENTIFIER_FOR_LAZY_NOT_LOADED_OBJECTS);
        hibernate5Module.disable(Hibernate5Module.Feature.FORCE_LAZY_LOADING);
        objectMapper.registerModule(hibernate5Module);
        objectMapper.registerModule(new JavaTimeModule());
        objectMapper.enable(MapperFeature.USE_ANNOTATIONS);
        objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        objectMapper.setSerializationInclusion(JsonInclude.Include.NON_EMPTY);
        objectMapper.setDefaultPropertyInclusion(JsonInclude.Include.NON_NULL);
        objectMapper.setDefaultPropertyInclusion(JsonInclude.Include.NON_EMPTY);
        objectMapper.disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
        objectMapper.setVisibility(PropertyAccessor.ALL, JsonAutoDetect.Visibility.ANY);
        objectMapper.activateDefaultTyping(
                LaissezFaireSubTypeValidator.instance,
                ObjectMapper.DefaultTyping.NON_FINAL,
                JsonTypeInfo.As.PROPERTY);
        objectMapper.disable(SerializationFeature.FAIL_ON_EMPTY_BEANS);
        objectMapper.disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
        objectMapper.enable(JsonReadFeature.ALLOW_UNESCAPED_CONTROL_CHARS.mappedFeature());
        GenericJackson2JsonRedisSerializer serializer = new GenericJackson2JsonRedisSerializer(objectMapper);
        return serializer;
    }

    @Bean
    public RedisTemplate<String, Object> redisTemplate(RedisConnectionFactory connectionFactory) {
        RedisTemplate<String, Object> template = new RedisTemplate<>();
        template.setKeySerializer(new StringRedisSerializer());
        template.setValueSerializer(this.redisSerializer());
        template.setConnectionFactory(connectionFactory);
        return template;
    }

    @Override
    @Primary
    @Bean
    public CacheManager cacheManager() {
        CacheManager cacheManager = redisCacheManagerBuilder().build();
        return cacheManager;
    }

    @Override
    public CacheErrorHandler errorHandler() {
        return new CacheErrorHandler() {
            @Override
            public void handleCacheGetError(RuntimeException exception, Cache cache, Object key) {
                logger.error("Cache Get: {} failed", key, exception);
            }

            @Override
            public void handleCachePutError(RuntimeException exception, Cache cache, Object key, Object value) {
                logger.error("Cache Put: {} failed", key, exception);
            }

            @Override
            public void handleCacheEvictError(RuntimeException exception, Cache cache, Object key) {
                logger.error("Cache Evict: {} failed", key, exception);
            }

            @Override
            public void handleCacheClearError(RuntimeException exception, Cache cache) {
                logger.error("Cache Clear failed", exception);
            }
        };
    }

    @Bean("refreshingStrategyTopic")
    public ChannelTopic topic() {
        return new ChannelTopic(RefreshingStrategyCacheEvent.TOPIC);
    }

    @Bean("refreshingStrategyRedisMessageListenerContainer")
    public RedisMessageListenerContainer refreshingStrategyRedisMessageListenerContainer(RedisConnectionFactory connectionFactory,
                                                                                         RedisTemplate<String, Object> redisTemplate,
                                                                                         @Qualifier("refreshingStrategyTopic")ChannelTopic topic) {
        logger.info("RedisMessageListenerContainer add messageListener");
        RedisMessageListenerContainer container = new RedisMessageListenerContainer();
        container.setConnectionFactory(connectionFactory);
        RefreshingStrategyMessageSubscriber messageSubscriber = new RefreshingStrategyMessageSubscriber();
        messageSubscriber.setApplicationEventPublisher(applicationEventPublisher);
        MessageListenerAdapter messageListenerAdapter = new MessageListenerAdapter(messageSubscriber);
        container.addMessageListener(messageListenerAdapter, topic);
        return container;
    }

    @Bean("refreshingStrategyRedisPublisher")
    public MessagePublisher refreshingStrategyRedisPublisher(StringRedisTemplate redisTemplate,
                                                             @Qualifier("refreshingStrategyTopic")ChannelTopic topic) {
        return new RedisMessagePublisher(redisTemplate, topic);
    }
}
