package com.xforceplus.config;

import com.alibaba.ttl.spi.TtlWrapper;
import com.alibaba.ttl.threadpool.TtlExecutors;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.scheduling.annotation.AsyncConfigurer;
import org.springframework.scheduling.annotation.EnableAsync;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;


/**
 * @author chenpengpeng@xforceplus.com
 * @date 2021/8/3
 **/
@EnableAsync
@Configuration
@Slf4j
public class GlobalThreadPoolConfig implements AsyncConfigurer {
    private static final Integer CPU_SIZE = Math.max(Runtime.getRuntime().availableProcessors(), 8);
    private static final Integer QUEUE_SIZE = 100000;
    private static final String THREAD_NAME = "async-thread-";

    @Bean("threadPoolExecutor")
    public ThreadPoolExecutor getAsyncThreadPoolExecutor() {
        TtlWrapper<ThreadPoolExecutor> ttlWrapper = (TtlWrapper<ThreadPoolExecutor>) this.getAsyncExecutor();
        if (ttlWrapper == null) {
            throw new RuntimeException("ttlWrapper is null");
        }
        ThreadPoolExecutor executor = ttlWrapper.unwrap();
        return executor;
    }

    @Override
    public ExecutorService getAsyncExecutor() {
        log.info("global-thread-cpu-size={}", CPU_SIZE);

        ThreadPoolTaskExecutor executor = new ThreadPoolTaskExecutor();
        // 配置核心线程数
        executor.setCorePoolSize(CPU_SIZE);
        // 配置最大线程数
        executor.setMaxPoolSize(CPU_SIZE * 2);
        // 配置队列大小
        executor.setQueueCapacity(QUEUE_SIZE);
        //配置线程池中的线程的名称前缀
        executor.setThreadNamePrefix(THREAD_NAME);
        // rejection-policy：当pool已经达到max size的时候，如何处理新任务
        // CALLER_RUNS：不在新线程中执行任务，而是有调用者所在的线程来执行
        executor.setRejectedExecutionHandler(new ThreadPoolExecutor.CallerRunsPolicy());
        //执行初始化
        executor.initialize();
        //TtlExecutors使用这个包装，这个在前置知识第四点的博客中可以明白其原理
        return TtlExecutors.getTtlExecutorService(executor.getThreadPoolExecutor());
    }


    @Bean(name = "threadPoolTaskExecutor")
    public ThreadPoolTaskExecutor threadPoolTaskExecutor() {
        ThreadPoolTaskExecutor executor = new ThreadPoolTaskExecutor();
        executor.setCorePoolSize(CPU_SIZE);
        executor.setMaxPoolSize(CPU_SIZE * 2);
        executor.setQueueCapacity(QUEUE_SIZE);
        executor.setTaskDecorator(new MdcTaskDecorator());
        // 线程池对拒绝任务(无线程可用)的处理策略
        executor.setRejectedExecutionHandler(new ThreadPoolExecutor.CallerRunsPolicy());
        return executor;
    }

    @Scheduled(fixedRate = 30, timeUnit = TimeUnit.SECONDS)
    public void threadInfo() {
        ThreadPoolExecutor executor = this.getAsyncThreadPoolExecutor();
        log.info("global thread:current active thread size={},max thread size={},current queue size={},task count={},completed task count={}",
                executor.getActiveCount(), executor.getLargestPoolSize(), executor.getQueue().size(), executor.getTaskCount(), executor.getCompletedTaskCount());
    }
}
