package com.xforceplus.config;

import com.fasterxml.classmate.TypeResolver;
import com.xforceplus.api.common.Uri;
import com.xforceplus.tenant.security.token.domain.UserType;
import io.swagger.models.auth.In;
import org.springframework.boot.autoconfigure.data.web.SpringDataWebProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Profile;
import org.springframework.core.Ordered;
import org.springframework.core.env.Environment;
import org.springframework.data.domain.Pageable;
import org.springframework.web.bind.annotation.RequestMapping;
import springfox.documentation.builders.*;
import springfox.documentation.schema.AlternateTypeRule;
import springfox.documentation.schema.AlternateTypeRuleConvention;
import springfox.documentation.service.*;
import springfox.documentation.spi.DocumentationType;
import springfox.documentation.spi.service.contexts.SecurityContext;
import springfox.documentation.spring.web.plugins.Docket;
import springfox.documentation.swagger2.annotations.EnableSwagger2WebMvc;

import java.lang.reflect.Type;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZonedDateTime;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static springfox.documentation.schema.AlternateTypeRules.newRule;

/**
 * @author geewit
 */
@Profile({"!prod"})
@Configuration
@EnableConfigurationProperties({SpringDataWebProperties.class})
@EnableSwagger2WebMvc
public class Swagger2Config {

    private final Environment environment;
    private final SpringDataWebProperties springDataWebProperties;

    public Swagger2Config(Environment environment, SpringDataWebProperties springDataWebProperties) {
        this.environment = environment;
        this.springDataWebProperties = springDataWebProperties;
    }

    @Bean
    public Docket docket() {
        return new Docket(DocumentationType.SWAGGER_2)
                .directModelSubstitute(LocalDateTime.class, Date.class)
                .directModelSubstitute(LocalDate.class, String.class)
                .directModelSubstitute(LocalTime.class, String.class)
                .directModelSubstitute(ZonedDateTime.class, String.class)
                .securitySchemes(this.securitySchemes())
                .securityContexts(this.securityContexts())
                .apiInfo(this.apiInfo())
                .select()
                .apis(RequestHandlerSelectors.withMethodAnnotation(RequestMapping.class))
                .paths(input -> !PathSelectors.ant("/api/**/" + Uri.API_VERSION).test(input))
                .paths(input -> !PathSelectors.ant("/api/**/list").test(input))
                .paths(input -> !PathSelectors.ant("/api/**/query").test(input))
                .paths(input -> !PathSelectors.ant("/error/**").test(input))
                .build()
                .groupName("v2")
                .tags(new Tag("resources", "资源服务"),
                        new Tag("role", "角色相关接口"),
                        new Tag("resource", "资源码相关接口"),
                        new Tag("resourceSet", "功能集相关接口"),
                        new Tag("serviceApi", "服务接口相关接口"),
                        new Tag("servicePackage", "服务包相关接口"),
                        new Tag("client", "客户端服务"),
                        new Tag("ExcelFileSotre", "ExcelFile导入导出"));
    }

    /**
     * 在Swagger2的securityContexts中通过正则表达式，设置需要使用参数的接口（或者说，是去除掉不需要使用参数的接口），
     * 如下列代码所示，通过PathSelectors.regex("^(?!auth).*$")，
     * 所有包含"auth"的接口不需要使用securitySchemes。即不需要使用上文中设置的名为“Authorization”，
     * type为“header”的参数。
     */
    private List<SecurityContext> securityContexts() {
        SecurityContext build = SecurityContext.builder()
                .securityReferences(this.defaultAuth())
                .forPaths(PathSelectors.any())
                .build();
        return Stream.of(build).collect(Collectors.toList());
    }

    private ApiInfo apiInfo() {
        return new ApiInfoBuilder()
                .title(environment.getProperty("spring.application.name", "tenant-service") + " RESTful APIs")
                .description(environment.getProperty("spring.application.name", "tenant-service"))
                .version(environment.getProperty("spring.application.version", "1.0.0"))
                .build();
    }

    /**
     * swagger加入全局Authorization header 将在ui界面右上角新增token输入界面
     *
     * @return
     */
    private List<SecurityScheme> securitySchemes() {
        return Stream.of(
                new ApiKey(UserType.USER.tokenKey(), UserType.USER.tokenKey(), In.HEADER.toValue()),
                new ApiKey(UserType.APPID.tokenKey(), UserType.APPID.tokenKey(), In.HEADER.toValue())
        ).collect(Collectors.toList());
    }

    List<SecurityReference> defaultAuth() {
        AuthorizationScope[] authorizationScopes = new AuthorizationScope[]{new AuthorizationScope("v2", "accessEverything")};
        return Stream.of(
                new SecurityReference(UserType.APPID.tokenKey(), authorizationScopes),
                new SecurityReference(UserType.USER.tokenKey(), authorizationScopes)
        ).collect(Collectors.toList());
    }

    @Bean
    public AlternateTypeRuleConvention pageableConvention(final TypeResolver resolver) {
        return new AlternateTypeRuleConvention() {
            @Override
            public int getOrder() {
                return Ordered.LOWEST_PRECEDENCE;
            }

            @Override
            public List<AlternateTypeRule> rules() {
                return Stream.of(newRule(
                        resolver.resolve(Pageable.class),
                        resolver.resolve(Swagger2Config.this.pageableMixin()))
                ).collect(Collectors.toList());
            }
        };
    }

    private Type pageableMixin() {
        return new AlternateTypeBuilder()
                .fullyQualifiedClassName(
                        String.format("%s.generated.%s",
                                Pageable.class.getPackage().getName(),
                                Pageable.class.getSimpleName()))
                .withProperties(Stream.of(
                        property(Integer.class, springDataWebProperties.getPageable().getPageParameter()),
                        property(Integer.class, springDataWebProperties.getPageable().getSizeParameter()),
                        property(String.class, "sort")
                ).collect(Collectors.toList()))
                .build();
    }

    private AlternateTypePropertyBuilder property(Class<?> type, String name) {
        return new AlternateTypePropertyBuilder()
                .withName(name)
                .withType(type)
                .withCanRead(true)
                .withCanWrite(true);
    }

}
