/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: ThreadPoolFactory.java   2020-10-15 15-40-47
 * Author: Evan
 */
package com.xforceplus.config;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.util.concurrent.ThreadPoolExecutor.CallerRunsPolicy;

import org.apache.commons.lang3.RandomUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.concurrent.*;


/**
 * <p>
 * Title: ThreadPoolConfig 创建定义
 * </p>
 * <p>
 * Description: ThreadPoolConfig 创建定义 默认核心线数5，队例大小：10，
 * corePoolSize：表示线程池保有的最小线程数。有些项目很闲，但是也不能把人都撤了，至少要留 corePoolSize 个人坚守阵地。
 * <br/>
 * maximumPoolSize：表示线程池创建的最大线程数。当项目很忙时，就需要加人，但是也不能无限制地加，最多就加到 maximumPoolSize 个人。当项目闲下来时，就要撤人了，
 * <br/>
 * 最多能撤到 corePoolSize 个人。keepAliveTime & unit：上面提到项目根据忙闲来增减人员，那在编程世界里，如何定义忙和闲呢？很简单，一个线程如果在一段时间内，都没有执行任务，说明很闲，
 * <br/>
 * keepAliveTime 和 unit 就是用来定义这个“一段时间”的参数。也就是说，如果一个线程空闲了keepAliveTime & unit这么久，
 * <br/>而且线程池的线程数大于 corePoolSize ，那么这个空闲的线程就要被回收了。
 * workQueue：工作队列，和上面示例代码的工作队列同义。
 * <br/>
 * threadFactory：通过这个参数你可以自定义如何创建线程，例如你可以给线程指定一个有意义的名字。
 * <br/>
 * handler：通过这个参数你可以自定义任务的拒绝策略。如果线程池中所有的线程都在忙碌，并且工作队列也满了（前提是工作队列是有界队列），
 * <br/>那么此时提交任务，线程池就会拒绝接收。至于拒绝的策略，你可以通过 handler 这个参数来指定。ThreadPoolExecutor 已经提供了以下 4 种策略。
 * <br/>CallerRunsPolicy：提交任务的线程自己去执行该任务。
 * <br/>AbortPolicy：默认的拒绝策略，会 throws RejectedExecutionException。
 * <br/>DiscardPolicy：直接丢弃任务，没有任何异常抛出。
 * <br/> DiscardOldestPolicy：丢弃最老的任务，其实就是把最早进入工作队列的任务丢弃，然后把新任务加入到工作队列。
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-10-15 15-40-47
 * @since V1.0
 */
public final class ThreadPoolConfig {

    /**
     * 线程池配置
     *
     * @return Config
     */
    public static Config config() {
        return new Config();
    }

    /**
     * 线程池配置
     */
    public static class Config {

        /**
         * 核心线程数 {@value}
         */
        public static final int DEFAULT_CORE_POOL_SIZE = 8;
        /**
         * 队例大小 {@value}
         */
        public static final int DEFAULT_QUEUE_SIZE = 20000;

        /**
         * 线程池大小 {@value}
         */
        public static final int DEFAULT_MAXIMUM_POOL_SIZE = 40;
        /**
         * 当线程数大于核心时，此为终止前多余的空闲线程等待新任务的最长时间。 keepAliveTime=0时，表示超过core线程数的线程在空闲时立即结束
         */
        public static final long DEFAULT_KEEPALIVE_TIME = 60000L;

        /**
         * 线程工厂
         */
        private ThreadFactory threadFactory;
        /**
         * 当线程池中的资源已经全部使用，添加新线程被拒绝时
         */
        private RejectedExecutionHandler handler;
        /**
         * 线程名称
         */
        private String name;
        /**
         * 非核心线程的闲置超时时间，超过这个时间就会被回收。
         */
        private TimeUnit timeUnit;
        /**
         * 核心线程数，默认情况下核心线程会一直存活，即使处于闲置状态也不会受存keepAliveTime限制。
         */
        private Integer corePoolSize;
        /**
         * 线程池所能容纳的最大线程数。超过这个数的线程将被阻塞。当任务队列为没有设置大小的LinkedBlockingDeque时，这个值无效。
         */
        private Integer maximumPoolSize;
        /**
         * 非核心线程的闲置超时时间，超过这个时间就会被回收。
         */
        private Long keepAliveTime;

        /**
         * queueSize
         */
        private Integer queueSize;

        private BlockingQueue<Runnable> queue;

        /**
         * 线程工厂，提供创建新线程的功能。ThreadFactory是一个接口，只有一个方法
         * @param threadFactory ThreadFactory
         * @return Config Config
         */
        public Config threadFactory(ThreadFactory threadFactory) {
            this.threadFactory = threadFactory;
            return this;
        }

        /**
         * Set RejectedExecutionHandler
         *
         * @param handler RejectedExecutionHandler
         * @return Config
         */
        public Config handler(RejectedExecutionHandler handler) {
            this.handler = handler;
            return this;
        }

        /**
         * 设置TimeUnit
         *
         * @param timeUnit TimeUnit
         * @return Config
         */
        public Config timeUnit(TimeUnit timeUnit) {
            this.timeUnit = timeUnit;
            return this;
        }

        /**
         * 核心线程数
         *
         * @param corePoolSize Config
         * @return Config
         */
        public Config corePoolSize(int corePoolSize) {
            this.corePoolSize = corePoolSize;
            return this;
        }

        /**
         * 有界队列。当使用有限的 maximumPoolSizes 时，有界队列（如 ArrayBlockingQueue）
         * <br/>有助于防止资源耗尽，但是可能较难调整和控制。队列大小和最大池大小可能需要相互折衷：
         * <br/>使用大型队列和小型池可以最大限度地降低 CPU 使用率、操作系统资源和上下文切换开销，
         * <br/>但是可能导致人工降低吞吐量。如果任务频繁阻塞（例如，如果它们是 I/O 边界），
         * <br/>则系统可能为超过您许可的更多线程安排时间。使用小型队列通常要求较大的池大小，
         * <br/>CPU 使用率较高，但是可能遇到不可接受的调度开销，这样也会降低吞吐量。
         *
         * @param maximumPoolSize 有界队列
         * @return Config
         */
        public Config maximumPoolSize(int maximumPoolSize) {
            this.maximumPoolSize = maximumPoolSize;
            return this;
        }

        /**
         * ThreadPoolExecutor的keepAliveTime=0时，表示超过core线程数的线程在空闲时立即结束
         *
         * @param keepAliveTime 默认值为0
         * @return Config
         */
        public Config keepAliveTime(long keepAliveTime) {
            this.keepAliveTime = keepAliveTime;
            return this;
        }

        /**
         * 设置线程名称
         *
         * @param name 线程名称
         * @return Config
         */
        public Config name(String name) {
            this.name = name;
            return this;
        }

        /**
         * 设置队列
         *
         * @param queue BlockingQueue<Runnable>
         * @return Config
         */
        public Config queue(BlockingQueue<Runnable> queue) {
            this.queue = queue;
            return this;
        }

        /**
         * 排队的等待数量
         *
         * @param queueSize
         * @return
         */
        public Config queueSize(int queueSize) {
            this.queueSize = queueSize;
            return this;
        }


        /**
         * 创建默认线程工厂类
         *
         * @param name 线程名称 ，如果线程池名称为空，会生成随时数
         * @return ThreadFactory
         */
        protected ThreadFactory getThreadFactory(String name) {
            if (StringUtils.isBlank(name)) {
                //6位随机数做为线程名称
                this.name = String.format("%1$06d", RandomUtils.nextInt(0, 100000));
            } else {
                this.name = name;
            }
            return new ThreadFactoryBuilder().setNameFormat(this.name).build();
        }

        /**
         * 创建默认队列
         *
         * @param size 队列
         * @return BlockingQueue<Runnable>
         */
        protected BlockingQueue<Runnable> createQueue(Integer size) {
            return new LinkedBlockingQueue<>(size);
        }

        /**
         * 创建线程池 判断handler是否为空
         * @return ThreadPoolExecutor
         */
        protected ThreadPoolExecutor newThreadPool() {
            if (this.handler == null) {
                return new ThreadPoolExecutor(this.corePoolSize, this.maximumPoolSize,
                        this.keepAliveTime, this.timeUnit,
                        this.queue, this.threadFactory, new CallerRunsPolicy());
            }
            return new ThreadPoolExecutor(this.corePoolSize, this.maximumPoolSize,
                    this.keepAliveTime, this.timeUnit,
                    this.queue, this.threadFactory, this.handler);
        }

        protected void checkArgument() {
            if (this.corePoolSize < 0 ||
                    this.maximumPoolSize <= 0 ||
                    this.maximumPoolSize < this.corePoolSize ||
                    this.keepAliveTime < 0) {
                throw new IllegalArgumentException();
            }
        }


        public ThreadPoolExecutor build() {
            //线程工厂内
            if (this.threadFactory == null) {
                this.threadFactory = this.getThreadFactory(name);
            }
            if (this.maximumPoolSize == null) {
                this.maximumPoolSize = DEFAULT_MAXIMUM_POOL_SIZE;
            }
            if (corePoolSize == null) {
                corePoolSize = DEFAULT_CORE_POOL_SIZE;
            }
            if (keepAliveTime == null) {
                keepAliveTime = DEFAULT_KEEPALIVE_TIME;
            }
            //检查是否正确
            this.checkArgument();
            if (this.queueSize == null) {
                this.queueSize = DEFAULT_QUEUE_SIZE;
            }
            if (this.timeUnit == null) {
                this.timeUnit = TimeUnit.MILLISECONDS;
            }
            if (this.queue == null || this.queue.size() == 0) {
                this.queue = this.createQueue(this.queueSize);
            }
            return this.newThreadPool();
        }
    }
}