package com.xforceplus.config;

import com.fasterxml.jackson.core.json.JsonReadFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.ser.std.ToStringSerializer;
import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.domain.account.AccountType;
import com.xforceplus.domain.account.deserializer.AccountTypeDeserializer;
import com.xforceplus.jackson.databind.serializer.ResponseEntitySerializer;
import com.xforceplus.tenant.security.core.domain.OrgType;
import com.xforceplus.tenant.security.core.domain.deserializer.OrgTypeDeserializer;
import io.geewit.core.jackson.databind.serializer.BigDecimalSerializer;
import io.geewit.core.jackson.databind.serializer.EnumNameSerializer;
import io.geewit.core.jackson.databind.serializer.EnumValueSerializer;
import io.geewit.core.jackson.databind.serializer.JsonPageSerializer;
import io.geewit.web.convert.converter.DateConverter;
import io.geewit.web.convert.converter.EnumNameConverter;
import io.geewit.web.convert.converter.EnumValueConverter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.data.web.SpringDataWebProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.web.PageableHandlerMethodArgumentResolver;
import org.springframework.data.web.SortHandlerMethodArgumentResolver;
import org.springframework.data.web.config.EnableSpringDataWebSupport;
import org.springframework.format.FormatterRegistry;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.web.cors.CorsConfiguration;
import org.springframework.web.method.support.HandlerMethodArgumentResolver;
import org.springframework.web.servlet.config.annotation.CorsRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;


/**
 * spring mvc 的配置
 *
 * @author geewit
 */
@EnableSpringDataWebSupport
@EnableConfigurationProperties({GlobalProperties.class, SpringDataWebProperties.class})
@Configuration
public class WebConfig implements WebMvcConfigurer {
    private final static Logger logger = LoggerFactory.getLogger(WebConfig.class);

    private final GlobalProperties globalProperties;
    private final SpringDataWebProperties springDataWebProperties;

    public WebConfig(GlobalProperties globalProperties, SpringDataWebProperties springDataWebProperties) {
        logger.info("WebConfig init");
        this.globalProperties = globalProperties;
        this.springDataWebProperties = springDataWebProperties;
    }

    @Override
    public void addCorsMappings(CorsRegistry registry) {
        registry.addMapping("/**")
                .allowedOriginPatterns(CorsConfiguration.ALL)
                .allowedMethods(CorsConfiguration.ALL)
                .allowedHeaders(CorsConfiguration.ALL)
                .allowCredentials(Boolean.FALSE)
                .maxAge(3600L);
    }

    @Override
    public void addFormatters(FormatterRegistry registry) {
        registry.addConverter(String.class, OrgType.class, new EnumValueConverter<>());
        registry.addConverter(Integer.class, OrgType.class, new EnumValueConverter<>());
        registry.addConverter(int.class, OrgType.class, new EnumValueConverter<>());
        registry.addConverter(String.class, AccountType.class, new EnumNameConverter<>());
        registry.addConverter(String.class, Date.class, new DateConverter());
        WebMvcConfigurer.super.addFormatters(registry);
    }


    @Override
    public void extendMessageConverters(List<HttpMessageConverter<?>> converters) {
        for (HttpMessageConverter<?> converter : converters) {
            if (converter instanceof MappingJackson2HttpMessageConverter) {
                MappingJackson2HttpMessageConverter jacksonMessageConverter = (MappingJackson2HttpMessageConverter) converter;
                ObjectMapper objectMapper = jacksonMessageConverter.getObjectMapper();
                objectMapper.enable(JsonReadFeature.ALLOW_UNESCAPED_CONTROL_CHARS.mappedFeature());
                SimpleModule simpleModule = new SimpleModule();
                simpleModule.addSerializer(OrgType.class, EnumValueSerializer.instance);
                simpleModule.addSerializer(AccountType.class, EnumNameSerializer.instance);
                simpleModule.addSerializer(Long.class, ToStringSerializer.instance);
                simpleModule.addSerializer(Long.TYPE, ToStringSerializer.instance);
                simpleModule.addSerializer(BigDecimal.class, BigDecimalSerializer.instance);
                simpleModule.addSerializer(Page.class, new JsonPageSerializer(objectMapper));
                simpleModule.addSerializer(ResponseEntity.class, new ResponseEntitySerializer(objectMapper, globalProperties));
                simpleModule.addDeserializer(OrgType.class, OrgTypeDeserializer.instance);
                simpleModule.addDeserializer(AccountType.class, AccountTypeDeserializer.instance);
                objectMapper.registerModule(simpleModule);
                jacksonMessageConverter.setObjectMapper(objectMapper);
                break;
            }
        }
        WebMvcConfigurer.super.extendMessageConverters(converters);
    }


    @Override
    public void addArgumentResolvers(List<HandlerMethodArgumentResolver> resolvers) {
        SortHandlerMethodArgumentResolver sortResolver = new SortHandlerMethodArgumentResolver();
        sortResolver.setPropertyDelimiter(",");
        PageableHandlerMethodArgumentResolver pageableResolver = new PageableHandlerMethodArgumentResolver(sortResolver);
        pageableResolver.setPageParameterName(springDataWebProperties.getPageable().getPageParameter());
        pageableResolver.setSizeParameterName(springDataWebProperties.getPageable().getSizeParameter());
        pageableResolver.setMaxPageSize(springDataWebProperties.getPageable().getMaxPageSize());
        pageableResolver.setFallbackPageable(PageRequest.of(0, springDataWebProperties.getPageable().getDefaultPageSize()));
        pageableResolver.setOneIndexedParameters(true);
        resolvers.add(sortResolver);
        resolvers.add(pageableResolver);
        WebMvcConfigurer.super.addArgumentResolvers(resolvers);
    }
}
