package com.xforceplus.event.listener;

import com.google.common.collect.Lists;
import com.xforceplus.client.ActionTrailClient;
import com.xforceplus.client.vo.MetricEvent;
import com.xforceplus.client.vo.Response;
import com.xforceplus.event.model.EntityPostSaveEvent;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import io.geewit.web.utils.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.event.EventListener;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import javax.persistence.Column;
import java.lang.reflect.Method;
import java.util.List;

import static com.xforceplus.constants.ActionTrailConstants.*;


/**
 * 通用实体审计
 *
 * @author duanhy
 */
@Slf4j
@Component
public class ActionTrailEventListener {

    private final ActionTrailClient actionTrailClient;

    public ActionTrailEventListener(ActionTrailClient actionTrailClient) {
        this.actionTrailClient = actionTrailClient;
    }


    /**
     * 属地不审计
     */
    @Value("${app.isPaas:true}")
    private boolean isPaas;


    /**
     * 实体审计
     *
     * @param event
     */
    @Async("threadPoolTaskExecutor")
    @EventListener(classes = {EntityPostSaveEvent.class}
            , condition = "event.source instanceof T(com.xforceplus.entity.Company) " +
            "|| event.source instanceof T(com.xforceplus.entity.Tenant) " +
            "|| event.source instanceof T(com.xforceplus.entity.OrgStruct) " +
            "|| event.source instanceof T(com.xforceplus.entity.RoleUserRel)")
    public <T> void actionTrailListener(EntityPostSaveEvent<T> event) {
        if (!isPaas) {
            return;
        }
        try {
            Long tenantId = getTenant(event.getSource());
            String traceId = MDC.get(TRACE_ID);
            String url = MDC.get(URL);
            String ip = MDC.get(IP);
            String method = MDC.get(METHOD);
            String path = MDC.get(PATH);
            IAuthorizedUser iAuthorizedUser = UserInfoHolder.get();
            MetricEvent metricEvent = (new MetricEvent.Builder())
                    .eventId(traceId)
                    .tenantId(tenantId)
                    .userInfo(iAuthorizedUser)
                    .eventType(event.getRevisionType().name())
                    .build();
            String body = JsonUtils.toJson(event.getSource());
            metricEvent.setRequestBody(body);
            metricEvent.setHost(ip);
            metricEvent.setHttpMethod(method);
            metricEvent.setUri(url);
            metricEvent.setPath(path);
            this.saveAction(metricEvent);
        } catch (Exception e) {
            log.error("actionTrailListenerEvent error", e);
        }

    }

    private void saveAction(MetricEvent metricEvent) {
        try {
            log.debug("saveAction request body:{}", JsonUtils.toJson(metricEvent));

            List<MetricEvent> list = Lists.newArrayList(metricEvent);
            Response response = actionTrailClient.saveAction(list);
            log.debug("saveAction response,traceId:{},body:{}", metricEvent.getEventId(), JsonUtils.toJson(response));
        } catch (Exception e) {
            log.error("SaveAction error", e);
        }

    }

    private Long getTenant(Object obj) {
        Long tenantId = 0L;
        Class<?> doClass = obj.getClass();
        try {
            for (Method method : doClass.getMethods()) {
                Column column = method.getAnnotation(Column.class);
                if (null != column && "tenant_id".equals(column.name()) && method.getName().startsWith("get")) {
                    method.setAccessible(true);
                    tenantId = (Long) method.invoke(obj);
                    break;
                }
            }
        } catch (Exception e) {
            log.error("actionTrailListenerEvent error getTenant", e);
        }
        return tenantId;
    }
}
