package com.xforceplus.event.listener;

import com.xforceplus.dao.OrgStructDao;
import com.xforceplus.entity.OrgStruct;
import com.xforceplus.event.dto.OrgCodeChanged;
import com.xforceplus.event.dto.OrgParentIdsBuild;
import com.xforceplus.event.dto.OrgParentIdsRebuild;
import com.xforceplus.event.model.EntityPostSaveEvent;
import com.xforceplus.event.model.EntityPreSaveEvent;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.event.EventListener;
import org.springframework.data.history.RevisionMetadata;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Isolation;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

/**
 * 组织保存事件监听器
 * @author geewit
 */
@Slf4j
@Component
public class OrgSaveEventListener {

    private final OrgStructDao orgStructDao;

    public OrgSaveEventListener(OrgStructDao orgStructDao) {
        this.orgStructDao = orgStructDao;
    }

    @Transactional(propagation = Propagation.REQUIRES_NEW, isolation = Isolation.READ_COMMITTED, readOnly = true)
    @EventListener(classes = {EntityPreSaveEvent.class}, condition = "event.source instanceof T(com.xforceplus.event.dto.OrgCodeChanged)")
    public void validListener(EntityPreSaveEvent<OrgCodeChanged> event) {
        OrgCodeChanged orgCodeValidation = (OrgCodeChanged) event.getSource();
        OrgStruct org = orgCodeValidation.getEntity();
        long count = orgStructDao.countByTenantIdAndOrgCode(org.getTenantId(), org.getOrgCode());
        if (count > 0) {
            throw new IllegalArgumentException("租户(" + org.getTenantId() + ")下已存在重复的组织code(" + org.getOrgCode() + ")");
        }
    }

    @Transactional(readOnly = true)
    @EventListener(classes = {EntityPreSaveEvent.class}, condition = "event.source instanceof T(com.xforceplus.event.dto.OrgParentIdsBuild)")
    public void buildParentIdsListener(EntityPreSaveEvent<OrgParentIdsBuild> event) {
        OrgParentIdsBuild orgParentIdsBuild = (OrgParentIdsBuild) event.getSource();
        OrgStruct org = orgParentIdsBuild.getEntity();
        StringBuilder builder = new StringBuilder();
        if (org.getParentId() != null && org.getParentId() > 0) {
            String parentIds;
            if (RevisionMetadata.RevisionType.INSERT.equals(event.getRevisionType())) {
                parentIds = orgStructDao.findUcommittedParentIdsByOrgId(org.getParentId());
            } else {
                parentIds = orgStructDao.findCommittedParentIdsByOrgId(org.getParentId());
            }

            if (parentIds != null) {
                builder.append(parentIds);
            } else {
                String message = "不存在的组织parentId(" + org.getParentId() + ")";
                log.warn(message);
                throw new IllegalArgumentException(message);
            }
        }
        if (org.getOrgId() != null) {
            builder.append(org.getOrgId());
        }
        if (org.getOrgType() != null) {
            builder.append(org.getOrgType().separator());
        }
        org.setParentIds(builder.toString());
    }



    @Async("threadPoolExecutor")
    @EventListener(classes = {EntityPostSaveEvent.class}, condition = "event.source instanceof T(com.xforceplus.event.dto.OrgParentIdsRebuild)")
    public void rebuildParentIdsListener(EntityPostSaveEvent<OrgParentIdsRebuild> event) {
        OrgParentIdsRebuild parentIdsRebuild = (OrgParentIdsRebuild) event.getSource();
        String previousParentIds = parentIdsRebuild.getPreviousParentIds();
        String nextParentIds = parentIdsRebuild.getNextParentIds();
        orgStructDao.updateParentIds(previousParentIds, nextParentIds);
    }
}
