package com.xforceplus.event.listener;

import com.xforceplus.dao.RoleDao;
import com.xforceplus.entity.Role;
import com.xforceplus.event.dto.RoleCodeChanged;
import com.xforceplus.event.dto.RoleNameChanged;
import com.xforceplus.event.model.EntityPreSaveEvent;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Isolation;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;


/**
 * Role 保存事件监听器
 *
 * @author geewit
 */
@Slf4j
@Component
public class RoleSaveEventListener {

    private final RoleDao roleDao;

    public RoleSaveEventListener(RoleDao roleDao) {
        this.roleDao = roleDao;
    }


    /**
     * 保存前校验事件监听器
     * @param event 保存前校验事件
     */
    @Transactional(readOnly = true, propagation = Propagation.REQUIRES_NEW, isolation = Isolation.READ_COMMITTED)
    @EventListener(classes = {EntityPreSaveEvent.class}, condition = "event.source instanceof T(com.xforceplus.event.dto.RoleCodeChanged)")
    public void validCodeListener(EntityPreSaveEvent<RoleCodeChanged> event) {
        RoleCodeChanged roleValidation = (RoleCodeChanged) event.getSource();
        Role role = roleValidation.getEntity();
        Long tenantId = role.getTenantId();
        if (tenantId == null) {
            tenantId = 0L;
        }
        long count = roleDao.countByTenantIdAndCode(tenantId, role.getCode());
        if (count > 0) {
            throw new IllegalArgumentException("重复的角色代码(" + role.getCode() + ")");
        }
    }

    /**
     * 保存前校验事件监听器
     * @param event 保存前校验事件
     */
    @Transactional(readOnly = true, propagation = Propagation.REQUIRES_NEW, isolation = Isolation.READ_COMMITTED)
    @EventListener(classes = {EntityPreSaveEvent.class}, condition = "event.source instanceof T(com.xforceplus.event.dto.RoleNameChanged)")
    public void validNameListener(EntityPreSaveEvent<RoleNameChanged> event) {
        RoleNameChanged roleValidation = (RoleNameChanged) event.getSource();
        Role role = roleValidation.getEntity();
        Long tenantId = role.getTenantId();
        if (tenantId == null) {
            tenantId = 0L;
        }
        long count = roleDao.countByTenantIdAndName(tenantId, role.getName());
        if (count > 0) {
            throw new IllegalArgumentException("重复的角色名称(" + role.getCode() + ")");
        }
    }
}
