package com.xforceplus.event.listener;

import com.xforceplus.api.model.ServicePackageModel;
import com.xforceplus.dao.ServicePackageDao;
import com.xforceplus.entity.ServicePackage;
import com.xforceplus.event.dto.ServicePackageCodeChanged;
import com.xforceplus.event.dto.ServicePackageNameChanged;
import com.xforceplus.event.model.EntityPreSaveEvent;
import com.xforceplus.query.ServicePackageQueryHelper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Isolation;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;


/**
 * ServicePackage 保存事件监听器
 *
 * @author geewit
 */
@Slf4j
@Component
public class ServicePackageSaveEventListener {

    private final ServicePackageDao servicePackageDao;

    public ServicePackageSaveEventListener(ServicePackageDao servicePackageDao) {
        this.servicePackageDao = servicePackageDao;
    }


    /**
     * 保存前校验事件监听器
     * @param event 保存前校验事件
     */
    @Transactional(readOnly = true, propagation = Propagation.REQUIRES_NEW, isolation = Isolation.READ_COMMITTED)
    @EventListener(classes = {EntityPreSaveEvent.class}, condition = "event.source instanceof T(com.xforceplus.event.dto.ServicePackageCodeChanged)")
    public void validCodeListener(EntityPreSaveEvent<ServicePackageCodeChanged> event) {
        ServicePackageCodeChanged codeChanged = (ServicePackageCodeChanged) event.getSource();
        ServicePackage entity = codeChanged.getEntity();
        long count = servicePackageDao.countByServicePackageCode(entity.getServicePackageCode());
        if (count > 0) {
            throw new IllegalArgumentException("已存在该服务包代码(" + entity.getServicePackageCode() + ")");
        }
    }

    /**
     * 保存前校验事件监听器
     * @param event 保存前校验事件
     */
    @Transactional(readOnly = true, propagation = Propagation.REQUIRES_NEW, isolation = Isolation.READ_COMMITTED)
    @EventListener(classes = {EntityPreSaveEvent.class}, condition = "event.source instanceof T(com.xforceplus.event.dto.ServicePackageNameChanged)")
    public void validNameListener(EntityPreSaveEvent<ServicePackageNameChanged> event) {
        ServicePackageNameChanged nameChanged = (ServicePackageNameChanged) event.getSource();
        ServicePackage entity = nameChanged.getEntity();
        long count = servicePackageDao.countByServicePackageName(entity.getAppId(), entity.getServicePackageName());
        if (count > 0) {
            throw new IllegalArgumentException("已存在该服务包名称(appId:" + entity.getAppId() + ",名称:" + entity.getServicePackageName() + ")");
        }
    }
}
