package com.xforceplus.event.listener;

import com.xforceplus.api.model.TenantModel;
import com.xforceplus.business.tenant.service.OrgService;
import com.xforceplus.dao.TenantDao;
import com.xforceplus.entity.Tenant;
import com.xforceplus.event.dto.TenantCodeChanged;
import com.xforceplus.event.dto.TenantNameChanged;
import com.xforceplus.event.model.EntityPostSaveEvent;
import com.xforceplus.event.model.EntityPreSaveEvent;
import com.xforceplus.query.TenantQueryHelper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.event.EventListener;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Isolation;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

/**
 * 租户保存事件监听器
 * @author geewit
 */
@Slf4j
@Component
public class TenantSaveEventListener {

    private final TenantDao tenantDao;
    private final OrgService orgService;

    public TenantSaveEventListener(TenantDao tenantDao, OrgService orgService) {
        this.tenantDao = tenantDao;
        this.orgService = orgService;
    }


    @Transactional(propagation = Propagation.REQUIRES_NEW, isolation = Isolation.READ_COMMITTED,rollbackFor = Exception.class)
    @EventListener(classes = {EntityPreSaveEvent.class}, condition = "event.source instanceof T(com.xforceplus.event.dto.TenantCodeChanged)")
    public void validCodeListener(EntityPreSaveEvent<TenantCodeChanged> event) {
        TenantCodeChanged codeChanged = (TenantCodeChanged) event.getSource();
        Tenant tenant = codeChanged.getEntity();
        TenantModel.Request.Query query = new TenantModel.Request.Query();
        query.setTenantCode(tenant.getTenantCode());
        long count = tenantDao.count(TenantQueryHelper.queryOneSpecification(query));
        if (count > 0) {
            throw new IllegalArgumentException("租户编码(" + tenant.getTenantCode() + ")已存在");
        }
    }

    @Transactional(propagation = Propagation.REQUIRES_NEW, isolation = Isolation.READ_COMMITTED,rollbackFor = Exception.class)
    @EventListener(classes = {EntityPreSaveEvent.class}, condition = "event.source instanceof T(com.xforceplus.event.dto.TenantNameChanged)")
    public void validNameListener(EntityPreSaveEvent<TenantNameChanged> event) {
        TenantNameChanged nameChanged = (TenantNameChanged) event.getSource();
        Tenant tenant = nameChanged.getEntity();
        TenantModel.Request.Query query = new TenantModel.Request.Query();
        query.setTenantName(tenant.getTenantName());
        long count = tenantDao.count(TenantQueryHelper.queryOneSpecification(query));
        if (count > 0) {
            throw new IllegalArgumentException("租户名称(" + tenant.getTenantName() + ")已存在");
        }
    }

    @Async("threadPoolExecutor")
    @EventListener(classes = {EntityPostSaveEvent.class}, condition = "event.source instanceof T(com.xforceplus.event.dto.TenantNameChanged)")
    public void tenantNameChangedListener(EntityPostSaveEvent<TenantNameChanged> event) {
        TenantNameChanged nameChanged = (TenantNameChanged) event.getSource();
        Tenant tenant = nameChanged.getEntity();
        long tenantId = tenant.getTenantId();
        String tenantName = tenant.getTenantName();
        orgService.changeRootName(tenantId, tenantName);
    }
}
