package com.xforceplus.exception;

import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.exc.InvalidFormatException;
import com.xforceplus.api.utils.Separator;
import com.xforceplus.business.reponse.code.Rep;
import com.xforceplus.security.login.exception.AuthenticationException;
import com.xforceplus.security.login.response.LoginResponse;
import com.xforceplus.tenant.core.exception.CodeException;
import com.xforceplus.tenant.core.exception.response.ErrorResponse;
import io.geewit.core.exception.ErrorCode;
import io.geewit.core.exception.HttpStatus;
import io.geewit.core.exception.NotFoundException;
import io.geewit.core.exception.ProcessedException;
import org.hibernate.PropertyValueException;
import org.hibernate.exception.ConstraintViolationException;
import org.hibernate.exception.DataException;
import org.hibernate.exception.LockAcquisitionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.*;
import org.springframework.http.ResponseEntity;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.jdbc.BadSqlGrammarException;
import org.springframework.orm.jpa.JpaObjectRetrievalFailureException;
import org.springframework.stereotype.Controller;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.web.HttpMediaTypeNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.method.annotation.MethodArgumentTypeMismatchException;
import org.springframework.web.servlet.NoHandlerFoundException;

import javax.persistence.EntityNotFoundException;
import javax.servlet.http.HttpServletResponse;
import javax.validation.ConstraintViolation;
import java.security.InvalidParameterException;
import java.sql.BatchUpdateException;
import java.util.*;
import java.util.stream.Collectors;


/**
 * 截取Exception 返回json {@link ControllerAdvice}
 *
 * @author geewit
 * @since 2019-11-04
 */
@SuppressWarnings("all")
@ControllerAdvice(annotations = {RestController.class, Controller.class})
public class ExceptionControllerAdvice {
    private final static Logger logger = LoggerFactory.getLogger(ExceptionControllerAdvice.class);

    @ResponseBody
    @ExceptionHandler(CodeException.class)
    public ResponseEntity<ErrorResponse> processException(CodeException e) {
        String message = e.getMessage();
        logger.warn("CodeException:" + message);
        int httpStatus = e.getHttpStatus();
        //返回exception.getMessage()
        return ResponseEntity.status(httpStatus).body(ErrorResponse.fail(e.getCode(), message));
    }

    @ResponseBody
    @ExceptionHandler({ProcessedException.class,})
    public ResponseEntity<ErrorResponse> processExcepton(ProcessedException e) {
        String message = e.getMessage();
        logger.warn("ProcessedException:" + message);
        int httpStatus = e.getHttpStatus();
        //返回exception.getMessage()
        return ResponseEntity.status(httpStatus).body(ErrorResponse.fail(e.getCode(), message));
    }

    @ResponseBody
    @ExceptionHandler(NotFoundException.class)
    public ResponseEntity<ErrorResponse> processExcepton(NotFoundException e) {
        String message = e.getMessage();
        logger.warn("NotFoundException:" + message);
        int httpStatus = e.getHttpStatus();
        //返回exception.getMessage()
        return ResponseEntity.status(httpStatus).body(ErrorResponse.fail(e.getCode(), message));
    }

    @ResponseBody
    @ExceptionHandler(NoHandlerFoundException.class)
    public ResponseEntity<ErrorResponse> processException(NoHandlerFoundException e) {
        String code = Rep.CommonCode.NOT_EXIST;
        String message = e.getMessage();
        logger.warn("NoHandlerFoundException:" + message);
        int httpStatus = HttpStatus.NOT_FOUND;
        return ResponseEntity.status(httpStatus).body(ErrorResponse.fail(code, message));
    }

    @ResponseBody
    @ExceptionHandler(AuthenticationException.class)
    public ResponseEntity<LoginResponse> processException(AuthenticationException e) {
        int code = e.getCode();
        String message = e.getMessage();
        logger.warn("AuthenticationException:" + message);
        int httpStatus = e.getHttpStatus();
        return ResponseEntity.status(httpStatus).body(LoginResponse.builder().code(code).message(message).data(e.getResponse()).build());
    }

    @ResponseBody
    @ExceptionHandler(AccessDeniedException.class)
    public ResponseEntity<ErrorResponse> processException(AccessDeniedException e) {
        String code = Rep.CommonCode.FAIL;
        String message = e.getMessage();
        logger.warn("AccessDeniedException:" + message);
        int httpStatus = HttpStatus.FORBIDDEN;
        return ResponseEntity.status(httpStatus).body(ErrorResponse.fail(code, message));
    }

    @ResponseBody
    @ExceptionHandler(MethodArgumentNotValidException.class)
    public ResponseEntity<ErrorResponse> processException(MethodArgumentNotValidException e) {
        String code = Rep.CommonCode.PARAM_FORMAT_WRONG;
        String message;
        int httpStatus;
        if (e != null) {
            BindingResult bindingResult = e.getBindingResult();
            List<FieldError> errors = new ArrayList<>(bindingResult.getFieldErrors());
            //对错误以一定顺序显示
            Collections.sort(errors, Comparator.comparing(FieldError::getDefaultMessage).thenComparing(FieldError::getField));
            StringBuilder messageBuilder = new StringBuilder();
            for (Iterator<FieldError> iterator = errors.iterator(); iterator.hasNext(); ) {
                FieldError error = iterator.next();
                messageBuilder.append(error.getField())
                        .append(Separator.COLON)
                        .append(error.getDefaultMessage());
                if (iterator.hasNext()) {
                    messageBuilder.append(",");
                }
            }
            if (messageBuilder.length() == 0) {
                message = "参数格式错误";
            } else {
                message = messageBuilder.toString();
            }
            httpStatus = HttpStatus.BAD_REQUEST;
        } else {
            message = ErrorCode.UNKNOWN_ERROR;
            httpStatus = HttpStatus.INTERNAL_SERVER_ERROR;
        }
        logger.warn("message: {}, httpStatus: {}", message, httpStatus);
        //返回exception.getMessage()
        return ResponseEntity.status(httpStatus).body(ErrorResponse.fail(code, message));
    }

    @ResponseBody
    @ExceptionHandler(MissingServletRequestParameterException.class)
    public ResponseEntity<ErrorResponse> processExeption(MissingServletRequestParameterException e) {
        String code = Rep.CommonCode.PARAM_FORMAT_WRONG;
        String message = e.getMessage();
        logger.warn("MissingServletRequestParameterException: " + message);
        int httpStatus = HttpStatus.BAD_REQUEST;
        return ResponseEntity.status(httpStatus).body(ErrorResponse.fail(code, message));
    }


    @ResponseBody
    @ExceptionHandler(InvalidParameterException.class)
    public ResponseEntity<ErrorResponse> processException(InvalidParameterException e) {
        String message = e.getMessage();
        logger.warn("InvalidParameterException: " + message);
        String code = Rep.CommonCode.PARAM_UNKOWN;
        //返回exception.getMessage()
        return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(ErrorResponse.fail(code, message));
    }

    @ResponseBody
    @ExceptionHandler(IllegalArgumentException.class)
    public ResponseEntity<ErrorResponse> processException(IllegalArgumentException e) {
        String message = e.getMessage();
        logger.warn("IllegalArgumentException:" + message);
        String code = Rep.CommonCode.PARAM_FORMAT_WRONG;
        int httpStatus = HttpServletResponse.SC_BAD_REQUEST;
        //返回exception.getMessage()
        return ResponseEntity.status(httpStatus).body(ErrorResponse.fail(code, message));
    }

    @ResponseBody
    @ExceptionHandler(MethodArgumentTypeMismatchException.class)
    public ResponseEntity<ErrorResponse> processException(MethodArgumentTypeMismatchException e) {
        String code = Rep.CommonCode.PARAM_UNKOWN;
        logger.error("参数转换失败，方法：" + Objects.requireNonNull(e.getParameter().getMethod()).getName() + ",参数：" +
                e.getName() + "，信息：" + e.getLocalizedMessage());
        String message = "参数错误:" + e.getName();
        int httpStatus = HttpServletResponse.SC_BAD_REQUEST;
        //返回exception.getMessage()
        return ResponseEntity.status(httpStatus).body(ErrorResponse.fail(code, message));
    }

    @ResponseBody
    @ExceptionHandler(HttpMediaTypeNotSupportedException.class)
    public ResponseEntity<ErrorResponse> processException(HttpMediaTypeNotSupportedException e) {
        String code = Rep.CommonCode.PARAM_FORMAT_WRONG;
        String message = "不支持的MediaType";
        logger.warn(message + ":" + e.getMessage());
        int httpStatus = HttpServletResponse.SC_BAD_REQUEST;
        //返回exception.getMessage()
        return ResponseEntity.status(httpStatus).body(ErrorResponse.fail(code, message));
    }

    @ResponseBody
    @ExceptionHandler(javax.validation.ConstraintViolationException.class)
    public ResponseEntity<ErrorResponse> processException(javax.validation.ConstraintViolationException e) {
        Set<ConstraintViolation<?>> constraintViolations = e.getConstraintViolations();
        String message = constraintViolations.stream().map(violation -> violation.getPropertyPath() + Separator.COLON + violation.getMessage()).collect(Collectors.joining(","));
        logger.warn("ConstraintViolationException: " + message);
        String code = Rep.CommonCode.PARAM_FORMAT_WRONG;
        int httpStatus = HttpServletResponse.SC_BAD_REQUEST;
        //返回exception.getMessage()
        return ResponseEntity.status(httpStatus).body(ErrorResponse.fail(code, message));
    }

    @ResponseBody
    @ExceptionHandler(DataIntegrityViolationException.class)
    public ResponseEntity<ErrorResponse> processException(DataIntegrityViolationException e) {
        logger.warn("触发数据库校验错误:" + e.getMessage());
        if (e.getCause() instanceof ConstraintViolationException) {
            return processException((ConstraintViolationException) e.getCause());
        } else if (e.getCause() instanceof DataException) {
            return processException((DataException) e.getCause());
        } else if (e.getCause() instanceof PropertyValueException) {
            return processException((PropertyValueException) e.getCause());
        } else {
            return processException((Exception) e);
        }
    }


    @ResponseBody
    @ExceptionHandler(PropertyValueException.class)
    public ResponseEntity<ErrorResponse> processException(PropertyValueException e) {
        String code = Rep.CommonCode.PARAM_FORMAT_WRONG;
        String message = "缺少必填参数(" + e.getEntityName() + "." + e.getPropertyName() + ")错误";
        logger.warn(message + ":" + e.getMessage());
        int httpStatus = HttpServletResponse.SC_BAD_REQUEST;
        return ResponseEntity.status(httpStatus).body(ErrorResponse.fail(code, message)); //返回exception.getMessage()
    }

    @ResponseBody
    @ExceptionHandler(DataException.class)
    public ResponseEntity<ErrorResponse> processException(DataException e) {
        String code = Rep.CommonCode.PARAM_FORMAT_WRONG;

        String message = "sql参数错误";
        logger.warn(message + ", sql:" + e.getSQL() + e.getMessage());
        int httpStatus = HttpServletResponse.SC_BAD_REQUEST;
        return ResponseEntity.status(httpStatus).body(ErrorResponse.fail(code, message)); //返回exception.getMessage()
    }

    @ResponseBody
    @ExceptionHandler(ConstraintViolationException.class)
    public ResponseEntity<ErrorResponse> processException(ConstraintViolationException e) {
        String code = Rep.CommonCode.EXIST_RESOURCE;
        String message = e.getConstraintName() + "参数唯一错误";
        logger.warn(message + ":" + e.getMessage());
        int httpStatus = HttpServletResponse.SC_BAD_REQUEST;
        return ResponseEntity.status(httpStatus).body(ErrorResponse.fail(code, message)); //返回exception.getMessage()
    }

    @ResponseBody
    @ExceptionHandler(EmptyResultDataAccessException.class)
    public ResponseEntity<ErrorResponse> processException(EmptyResultDataAccessException e) {
        String code = Rep.CommonCode.NOT_EXIST;
        String message = "数据返回数量0错误";
        logger.warn(message + ":" + e.getMessage());
        int httpStatus = HttpServletResponse.SC_BAD_REQUEST;
        return ResponseEntity.status(httpStatus).body(ErrorResponse.fail(code, message)); //返回exception.getMessage()
    }

    @ResponseBody
    @ExceptionHandler(IncorrectResultSizeDataAccessException.class)
    public ResponseEntity<ErrorResponse> processException(IncorrectResultSizeDataAccessException e) {
        String code = Rep.CommonCode.PARAM_FORMAT_WRONG;
        String message = "数据期望返回数量:" + e.getExpectedSize() + ", 实际返回数量:" + e.getActualSize() + "错误";
        logger.warn(message + ":" + e.getMessage());
        int httpStatus = HttpServletResponse.SC_BAD_REQUEST;
        return ResponseEntity.status(httpStatus).body(ErrorResponse.fail(code, message)); //返回exception.getMessage()
    }

    @ResponseBody
    @ExceptionHandler(JpaObjectRetrievalFailureException.class)
    public ResponseEntity<ErrorResponse> processException(JpaObjectRetrievalFailureException e) {
        String message;
        String code;
        if (e.getCause() != null && e.getCause() instanceof EntityNotFoundException) {
            message = "找不到实体";
            code = Rep.CommonCode.NOT_EXIST;
        } else {
            message = "参数错误";
            code = Rep.CommonCode.PARAM_LOGIC_WRONG;
        }
        logger.warn(message + ":" + e.getMessage(), e);
        int httpStatus = HttpServletResponse.SC_BAD_REQUEST;
        //返回exception.getMessage()
        return ResponseEntity.status(httpStatus).body(ErrorResponse.fail(code, message));
    }

    @ResponseBody
    @ExceptionHandler(BatchUpdateException.class)
    public ResponseEntity<ErrorResponse> processException(BatchUpdateException e) {
        if (e.getCause() instanceof LockAcquisitionException) {
            return this.processException((LockAcquisitionException) e.getCause());
        }
        if (e.getCause() instanceof DuplicateKeyException) {
            return this.processException((DuplicateKeyException) e.getCause());
        }
        String code = Rep.CommonCode.EXIST_RESOURCE;
        String message = "sql执行错误";
        logger.warn(message + ":" + e.getMessage(), e);
        int httpStatus = HttpServletResponse.SC_BAD_REQUEST;
        //返回exception.getMessage()
        return ResponseEntity.status(httpStatus).body(ErrorResponse.fail(code, message));
    }

    @ResponseBody
    @ExceptionHandler(BadSqlGrammarException.class)
    public ResponseEntity<ErrorResponse> processException(BadSqlGrammarException e) {
        if (e.getCause() instanceof IllegalArgumentException) {
            return this.processException((IllegalArgumentException) e.getCause());
        }
        String code = Rep.CommonCode.EXIST_RESOURCE;
        String message = "sql语法错误";
        logger.warn(message + ":" + e.getMessage(), e);
        int httpStatus = HttpServletResponse.SC_BAD_REQUEST;
        //返回exception.getMessage()
        return ResponseEntity.status(httpStatus).body(ErrorResponse.fail(code, message));
    }

    @ResponseBody
    @ExceptionHandler(DuplicateKeyException.class)
    public ResponseEntity<ErrorResponse> processException(DuplicateKeyException e) {
        if (e.getCause() instanceof IllegalArgumentException) {
            return this.processException((IllegalArgumentException) e.getCause());
        }
        String code = Rep.CommonCode.EXIST_RESOURCE;
        String message = "重复的主键";
        logger.warn(message + ":" + e.getMessage(), e);
        int httpStatus = HttpServletResponse.SC_BAD_REQUEST;
        //返回exception.getMessage()
        return ResponseEntity.status(httpStatus).body(ErrorResponse.fail(code, message));
    }

    @ResponseBody
    @ExceptionHandler(LockAcquisitionException.class)
    public ResponseEntity<ErrorResponse> processException(LockAcquisitionException e) {
        if (e.getCause() instanceof IllegalArgumentException) {
            return this.processException((IllegalArgumentException) e.getCause());
        }
        String code = Rep.CommonCode.EXIST_RESOURCE;
        String message = "数据库死锁";
        logger.warn(message + ":" + e.getMessage(), e);
        int httpStatus = HttpServletResponse.SC_BAD_REQUEST;
        //返回exception.getMessage()
        return ResponseEntity.status(httpStatus).body(ErrorResponse.fail(code, message));
    }

    @ResponseBody
    @ExceptionHandler(CannotAcquireLockException.class)
    public ResponseEntity<ErrorResponse> processException(CannotAcquireLockException e) {
        if (e.getCause() instanceof IllegalArgumentException) {
            return this.processException((IllegalArgumentException) e.getCause());
        }
        String code = Rep.CommonCode.EXIST_RESOURCE;
        String message = "数据库死锁";
        logger.warn(message + ":" + e.getMessage(), e);
        int httpStatus = HttpServletResponse.SC_BAD_REQUEST;
        //返回exception.getMessage()
        return ResponseEntity.status(httpStatus).body(ErrorResponse.fail(code, message));
    }

    @ResponseBody
    @ExceptionHandler(DataAccessException.class)
    public ResponseEntity<ErrorResponse> processException(DataAccessException e) {
        if (e.getCause() instanceof IllegalArgumentException) {
            return this.processException((IllegalArgumentException) e.getCause());
        }
        String code = Rep.CommonCode.EXIST_RESOURCE;
        String message = "数据获取错误";
        logger.warn(message + ":" + e.getMessage(), e);
        int httpStatus = HttpServletResponse.SC_BAD_REQUEST;
        //返回exception.getMessage()
        return ResponseEntity.status(httpStatus).body(ErrorResponse.fail(code, message));
    }


    @ResponseBody
    @ExceptionHandler(HttpMessageNotReadableException.class)
    public ResponseEntity<ErrorResponse> processException(HttpMessageNotReadableException e) {
        String code = Rep.CommonCode.PARAM_FORMAT_WRONG;
        String message = "参数格式错误!";
        logger.warn(message + ":" + e.getMessage());
        int httpStatus = HttpServletResponse.SC_BAD_REQUEST;

        if (e.getCause() instanceof InvalidFormatException) {
            InvalidFormatException e1 = (InvalidFormatException) e.getCause();
            List<JsonMappingException.Reference> path = e1.getPath();
            for (JsonMappingException.Reference reference : path) {
                message += "参数名：" + reference.getFieldName() + " 输入不合法，需要的是 " + e1.getTargetType().getName() + " 类型，提交的值是：" + e1.getValue().toString();
            }
        }
        //返回exception.getMessage()
        return ResponseEntity.status(httpStatus).body(ErrorResponse.fail(code, message));
    }

    @ResponseBody
    @ExceptionHandler(BindException.class)
    public ResponseEntity<ErrorResponse> processException(BindException e) {
        String code = Rep.CommonCode.PARAM_FORMAT_WRONG;
        String message = "参数格式错误";
        if (e != null) {
            logger.error(e.getMessage());
            BindingResult bindingResult = e.getBindingResult();
            List<FieldError> fieldErrors = bindingResult.getFieldErrors();
            if (!fieldErrors.isEmpty()) {
                StringBuilder messageBuilder = new StringBuilder();
                for (Iterator<FieldError> iterator = fieldErrors.iterator(); iterator.hasNext(); ) {
                    FieldError fieldError = iterator.next();
                    messageBuilder.append(fieldError.getDefaultMessage());
                    if (iterator.hasNext()) {
                        messageBuilder.append(",");
                    }
                }
                message = messageBuilder.toString();
            }
        }
        //返回exception.getMessage()
        return ResponseEntity.status(HttpServletResponse.SC_BAD_REQUEST).body(ErrorResponse.fail(code, message));
    }

    @ResponseBody
    @ExceptionHandler(Exception.class)
    public ResponseEntity<ErrorResponse> processException(Exception e) {
        String code = Rep.CommonCode.FAIL;
        String message;
        if (e != null) {
            logger.error(e.getMessage(), e);
            message = e.getMessage();
        } else {
            message = ErrorCode.UNKNOWN_ERROR;
        }
        //返回exception.getMessage()
        return ResponseEntity.status(HttpServletResponse.SC_INTERNAL_SERVER_ERROR).body(ErrorResponse.fail(code, message));
    }
}
