package com.xforceplus.security.login.controller;

import com.xforceplus.business.reponse.code.Rep;
import com.xforceplus.business.tenant.service.UserService;
import com.xforceplus.config.MessageTemplateProperties;
import com.xforceplus.entity.User;
import com.xforceplus.feign.tenant.message.SmsAuthCodeFeignClient;
import com.xforceplus.security.login.context.LoginContext;
import com.xforceplus.security.login.exception.AuthenticationException;
import com.xforceplus.security.login.request.*;
import com.xforceplus.security.login.response.LoginTokenResponse;
import com.xforceplus.security.login.response.LoginResponse;
import com.xforceplus.security.login.service.LoginService;
import com.xforceplus.tenant.security.autoscan.annotation.AuthorizedDefinition;
import com.xforceplus.utils.RegExUtil;
import com.xforececlound.message.model.AuthCodeResp;
import com.xforececlound.message.model.SmsCodeReq;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.http.HttpHeaders;
import org.springframework.http.ResponseCookie;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;
import java.util.Set;

/**
 * 登录接口 Controller
 *
 * @author geewit
 */
@Slf4j
@Validated
@Controller
public class LoginController {

    private final LoginService loginService;

    private final UserService userService;

    private final SmsAuthCodeFeignClient smsAuthCodeFeignClient;

    private final MessageTemplateProperties messageTemplateProperties;

    public LoginController(LoginService loginService,
                           UserService userService,
                           SmsAuthCodeFeignClient smsAuthCodeFeignClient,
                           MessageTemplateProperties messageTemplateProperties) {
        this.loginService = loginService;
        this.userService = userService;
        this.smsAuthCodeFeignClient = smsAuthCodeFeignClient;
        this.messageTemplateProperties = messageTemplateProperties;
    }

    /**
     * 对应原来的 'api/normal/login'
     *
     * @param loginRequest
     * @param request
     * @return
     */
    @ApiOperation(value = "对应原来的 'api/normal/login'")
    @AuthorizedDefinition(authentication = false, authorization = false)
    @ResponseBody
    @PostMapping({"/api/security/normal/login"})
    public ResponseEntity<LoginResponse<LoginTokenResponse>> login(@RequestBody @Valid PasswordLoginRequest loginRequest, HttpServletRequest request) {
        LoginContext<PasswordLoginRequest> loginContext = loginService.login(loginRequest, request);
        return this.buildResponse(loginContext);
    }

    /**
     * 对应原来的 'api/sms/login/v2'
     *
     * @param loginRequest
     * @param request
     * @return
     */
    @ApiOperation(value = "对应原来的 'api/sms/login/v2'")
    @AuthorizedDefinition(authentication = false, authorization = false)
    @ResponseBody
    @PostMapping({"/api/security/sms/login/v2"})
    public ResponseEntity<LoginResponse<LoginTokenResponse>> smsLoginV2(@RequestBody @Valid SmsLoginRequest loginRequest, HttpServletRequest request) {
        LoginContext<SmsLoginRequest> loginContext = loginService.login(loginRequest, request);
        return this.buildResponse(loginContext);
    }

    /**
     * 对应原来的 'api/sms/login'
     *
     * @param loginRequest
     * @param request
     * @return
     */
    @ApiOperation(value = "对应原来的 'api/sms/login'")
    @AuthorizedDefinition(authentication = false, authorization = false)
    @ResponseBody
    @PostMapping({"/api/security/sms/login"})
    public ResponseEntity<LoginResponse<LoginTokenResponse>> smsLogin(@RequestBody @Valid SmsLoginRequest loginRequest, HttpServletRequest request) {
        LoginContext<SmsLoginRequest> loginContext = loginService.smsLogin(loginRequest, request);
        return this.buildResponse(loginContext);
    }

    /**
     * 对应原来的 'api/login/double-auth/code'
     *
     * @param loginRequest
     * @param request
     * @return
     */
    @ApiOperation(value = "对应原来的 'api/login/double-auth/code'")
    @ResponseBody
    @PostMapping({"/api/security/two-factor/sms/login"})
    public ResponseEntity<LoginResponse<LoginTokenResponse>> login(@RequestBody TwoFactorSmsLoginRequest loginRequest, HttpServletRequest request) {
        LoginContext<TwoFactorSmsLoginRequest> loginContext = loginService.login(loginRequest, request);
        return this.buildResponse(loginContext);
    }

    /**
     * 对应原来的 '/api/login/double-auth/pwd'
     *
     * @param loginRequest
     * @param request
     * @return
     */
    @ApiOperation(value = "对应原来的 'api/login/double-auth/pwd'")
    @ResponseBody
    @PostMapping({"/api/security/two-factor/password/login"})
    public ResponseEntity<LoginResponse<LoginTokenResponse>> login(@RequestBody @Valid TwoFactorPasswordLoginRequest loginRequest, HttpServletRequest request) {
        LoginContext<TwoFactorPasswordLoginRequest> loginContext = loginService.login(loginRequest, request);
        return this.buildResponse(loginContext);
    }

    private <T extends LoginRequest> ResponseEntity<LoginResponse<LoginTokenResponse>> buildResponse(LoginContext<T> loginContext) {
        ResponseEntity.BodyBuilder responseBuilder = ResponseEntity.ok();
        Set<ResponseCookie> cookies = loginContext.getCookies();
        if (cookies != null && !cookies.isEmpty()) {
            HttpHeaders cookieHeaders = new HttpHeaders();
            cookies.forEach(cookie -> cookieHeaders.add(HttpHeaders.SET_COOKIE, cookie.toString()));
            responseBuilder = responseBuilder.headers(cookieHeaders);
        }

        return responseBuilder.body(
                LoginResponse.<LoginTokenResponse>builder()
                        .data(loginContext.getResponse())
                        .build());
    }

    /**
     * 对应原来的 '/api/sms/login/{phone}'
     *
     * @param phone   手机号
     * @return
     */
    @AuthorizedDefinition(authentication = false, authorization = false)
    @ApiOperation(value = "对应原来的 'api/sms/login/{phone}'")
    @ResponseBody
    @GetMapping({"/api/security/sms/login/{phone}"})
    public LoginResponse<String> sendSms(@PathVariable("phone") String phone) {
        if (!RegExUtil.checkMobile(phone)) {
            throw new AuthenticationException(Rep.AccountCode.FAIL, "请输入正确的手机号码！");
        }
        User user = userService.findByUsername(phone, 0);
        if (user == null) {
            throw new AuthenticationException(Rep.AccountCode.INVALID_USERNAME_OR_PASSWORD, "用户名或密码错误。");
        }
        if (1 != user.getAccount().getStatus()) {
            throw new AuthenticationException(Rep.AccountCode.FAIL, "账号未启用！");
        }
        SmsCodeReq smsCodeReq = new SmsCodeReq();
        smsCodeReq.setMobile(phone);

        smsCodeReq.setTemplateCode(messageTemplateProperties.getSmsCode());
        smsCodeReq.setExpireTime(MessageTemplateProperties.EXPIRE_TIME);
        smsCodeReq.setTenantId(user.getTenantId());

        AuthCodeResp authCodeResp = smsAuthCodeFeignClient.sendAuthCode(String.valueOf(ObjectUtils.defaultIfNull(user.getTenantId(), 0)), smsCodeReq);
        if (authCodeResp == null) {
            throw new AuthenticationException(Rep.AccountCode.FAIL, "发送短信验证码失败。");
        } else if (authCodeResp.getCode() != 1) {
            throw new AuthenticationException(Rep.AccountCode.FAIL, authCodeResp.getDesc());
        }
        String msgId = authCodeResp.getMsgId();
        if (StringUtils.isBlank(msgId)) {
            throw new AuthenticationException(Rep.AccountCode.FAIL, "发送验证码失败");
        }
        return LoginResponse.<String>builder().data(msgId).build();
    }
}
