package com.xforceplus.security.login.service;

import com.google.common.net.HttpHeaders;
import com.xforceplus.security.login.request.*;
import com.xforceplus.security.login.context.LoginContext;
import com.xforceplus.security.strategy.filter.StrategyFilterChain;
import com.xforceplus.security.strategy.filter.impl.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Service;
import org.springframework.web.util.UriComponentsBuilder;

import javax.servlet.http.HttpServletRequest;

/**
 * 登录服务
 * @author geewit
 */
@Slf4j
@Service
public class LoginService implements ApplicationContextAware {

    public static final String TOKEN_COOKIE_HOST = "token-cookie-host";

    private ApplicationContext applicationContext;

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    /**
     * 执行密码登录
     * @param loginRequest
     * @return LoginContext
     */
    public LoginContext<PasswordLoginRequest> login(PasswordLoginRequest loginRequest, HttpServletRequest request) {
        String host = this.parseHost(request);
        LoginContext<PasswordLoginRequest> loginContext = LoginContext.<PasswordLoginRequest>builder()
                .loginRequest(loginRequest)
                .loginName(loginRequest.getUsername())
                .host(host)
                .build();
        StrategyFilterChain<PasswordLoginRequest> filterChain = StrategyFilterChain.<PasswordLoginRequest>builder().loginContext(loginContext).build();
        return this.doStrategyFilterChain(filterChain);
    }

    /**
     * 执行短信登录
     * @param loginRequest
     * @return LoginContext
     */
    public LoginContext<SmsLoginRequest> login(SmsLoginRequest loginRequest, HttpServletRequest request) {
        String host = this.parseHost(request);
        LoginContext<SmsLoginRequest> loginContext = LoginContext.<SmsLoginRequest>builder()
                .loginRequest(loginRequest)
                .loginName(loginRequest.getPhone())
                .host(host).build();
        StrategyFilterChain<SmsLoginRequest> filterChain = StrategyFilterChain.<SmsLoginRequest>builder().loginContext(loginContext).build();
        return this.doStrategyFilterChain(filterChain);
    }

    /**
     * 执行短信登录
     * @param loginRequest
     * @return LoginContext
     */
    public LoginContext<SmsLoginRequest> smsLogin(SmsLoginRequest loginRequest, HttpServletRequest request) {
        String host = this.parseHost(request);
        LoginContext<SmsLoginRequest> loginContext = LoginContext.<SmsLoginRequest>builder()
                .loginRequest(loginRequest)
                .loginName(loginRequest.getPhone())
                .host(host).build();
        StrategyFilterChain<SmsLoginRequest> filterChain = StrategyFilterChain.<SmsLoginRequest>builder().loginContext(loginContext).build();
        return this.doSmsStrategyFilterChain(filterChain);
    }

    /**
     * 双因子密码登录请求报文
     * @param loginRequest
     * @return LoginContext
     */
    public LoginContext<TwoFactorPasswordLoginRequest> login(TwoFactorPasswordLoginRequest loginRequest, HttpServletRequest request) {
        String host = this.parseHost(request);
        LoginContext<TwoFactorPasswordLoginRequest> loginContext = LoginContext.<TwoFactorPasswordLoginRequest>builder()
                .loginRequest(loginRequest)
                .host(host)
                .build();
        StrategyFilterChain<TwoFactorPasswordLoginRequest> filterChain = StrategyFilterChain.<TwoFactorPasswordLoginRequest>builder().loginContext(loginContext).build();
        return this.doStrategyFilterChain(filterChain);
    }

    /**
     * 双因子短信/邮箱验证码登录请求报文
     * @param loginRequest
     * @return LoginContext
     */
    public LoginContext<TwoFactorSmsLoginRequest> login(TwoFactorSmsLoginRequest loginRequest, HttpServletRequest request) {
        String host = this.parseHost(request);
        LoginContext<TwoFactorSmsLoginRequest> loginContext = LoginContext.<TwoFactorSmsLoginRequest>builder()
                .loginRequest(loginRequest)
                .host(host)
                .build();
        StrategyFilterChain<TwoFactorSmsLoginRequest> filterChain = StrategyFilterChain.<TwoFactorSmsLoginRequest>builder().loginContext(loginContext).build();
        return this.doStrategyFilterChain(filterChain);
    }

    private String parseHost(HttpServletRequest request) {
        String host = request.getHeader(TOKEN_COOKIE_HOST);
        log.info("{} : {}", TOKEN_COOKIE_HOST, host);
        if (host == null) {
            host = request.getHeader(HttpHeaders.X_FORWARDED_HOST);
            log.info("{} : {}", HttpHeaders.X_FORWARDED_HOST, host);
        }
        if (host == null) {
            String originRequest = request.getHeader(HttpHeaders.X_FORWARDED_FOR);
            log.info("{} : {}", HttpHeaders.X_FORWARDED_FOR, originRequest);
            if (originRequest != null) {
                host = UriComponentsBuilder.fromHttpUrl(originRequest).build().getHost();
            } else {
                host = StringUtils.defaultString(request.getHeader(HttpHeaders.HOST), request.getServerName());
            }
        }

        return host;
    }

    /**
     * 执行策略处理器链
     * @param filterChain
     * @param <T>
     * @return
     */
    private <T extends LoginRequest> LoginContext<T> doStrategyFilterChain(StrategyFilterChain<T> filterChain) {
        filterChain.addFilter(LoadUserByPasswordStrategyFilter.builder().priority(1).applicationContext(applicationContext).build());
        filterChain.addFilter(LoadUserBySmsStrategyFilter.builder().priority(2).applicationContext(applicationContext).build());
        filterChain.addFilter(TwoFactorPasswordLoadUserStrategyFilter.builder().priority(3).applicationContext(applicationContext).build());
        filterChain.addFilter(TwoFactorSmsLoadUserStrategyFilter.builder().priority(4).applicationContext(applicationContext).build());
        filterChain.addFilter(FailTimesPredicationStrategyFilter.builder().priority(5).applicationContext(applicationContext).build());
        filterChain.addFilter(PasswordValidationStrategyFilter.builder().priority(6).applicationContext(applicationContext).build());
        filterChain.addFilter(PostLoadUserStrategyFilter.builder().priority(7).applicationContext(applicationContext).build());
        filterChain.addFilter(PasswordExpiredStrategyFilter.builder().priority(8).applicationContext(applicationContext).build());
        filterChain.addFilter(TwoFactorValidStrategyFilter.builder().priority(9).applicationContext(applicationContext).build());
        filterChain.addFilter(GenerateTokenStrategyFilter.builder().priority(10).applicationContext(applicationContext).build());
        filterChain.addFilter(ResponseCookieStrategyFilter.builder().priority(11).applicationContext(applicationContext).build());
        UpdateLoginTimeStrategyFilter updateLoginTimeStrategyFilter = UpdateLoginTimeStrategyFilter.builder().priority(12).applicationContext(applicationContext).build();
        updateLoginTimeStrategyFilter.init();
        filterChain.addFilter(updateLoginTimeStrategyFilter);
        filterChain.doFilters();
        return filterChain.getLoginContext();
    }

    /**
     * 执行策略处理器链
     * @param filterChain
     * @param <T>
     * @return
     */
    private <T extends LoginRequest> LoginContext<T> doSmsStrategyFilterChain(StrategyFilterChain<T> filterChain) {
        filterChain.addFilter(LoadUserByPasswordStrategyFilter.builder().priority(1).applicationContext(applicationContext).build());
        filterChain.addFilter(LoadUserBySmsStrategyFilter.builder().priority(2).applicationContext(applicationContext).build());
        filterChain.addFilter(FailTimesPredicationStrategyFilter.builder().priority(3).applicationContext(applicationContext).build());
        filterChain.addFilter(PasswordValidationStrategyFilter.builder().priority(4).applicationContext(applicationContext).build());
        filterChain.addFilter(PostLoadUserStrategyFilter.builder().priority(5).applicationContext(applicationContext).build());
        filterChain.addFilter(PasswordExpiredStrategyFilter.builder().priority(6).applicationContext(applicationContext).build());
        filterChain.addFilter(GenerateTokenStrategyFilter.builder().priority(7).applicationContext(applicationContext).build());
        filterChain.addFilter(ResponseCookieStrategyFilter.builder().priority(8).applicationContext(applicationContext).build());
        UpdateLoginTimeStrategyFilter updateLoginTimeStrategyFilter = UpdateLoginTimeStrategyFilter.builder().priority(9).applicationContext(applicationContext).build();
        updateLoginTimeStrategyFilter.init();
        filterChain.addFilter(updateLoginTimeStrategyFilter);
        filterChain.doFilters();
        return filterChain.getLoginContext();
    }
}
