package com.xforceplus.security.strategy.controller;

import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.model.AccountModel;
import com.xforceplus.business.account.service.AccountService;
import com.xforceplus.security.strategy.model.*;
import com.xforceplus.security.strategy.service.StrategyService;
import com.xforceplus.tenant.security.autoscan.annotation.AuthorizedDefinition;
import io.geewit.core.utils.reflection.BeanUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

import static com.xforceplus.api.common.Uri.PATH_GLOBAL_PREFIX;

@Controller
public class StrategyController {

    private final StrategyService strategyService;

    private final AccountService accountService;

    public StrategyController(StrategyService strategyService, AccountService accountService) {
        this.strategyService = strategyService;
        this.accountService = accountService;
    }

    @AuthorizedDefinition(authorization = false)
    @ResponseBody
    @RequestMapping(name = "刷新租户策略缓存", value = PATH_GLOBAL_PREFIX + "/strategies/refresh", method = { RequestMethod.POST })
    public String refresh() {
        return strategyService.refresh();
    }

    @AuthorizedDefinition(authorization = false)
    @ResponseBody
    @RequestMapping(name = "默认租户策略列表", value = PATH_GLOBAL_PREFIX + "/strategies", method = { RequestMethod.GET })
    public ResponseEntity<List<? super Strategy>> strategies() {
        List<? super Strategy> strategies = strategyService.allStrategies();
        return ResponseEntity.ok(strategies);
    }

    @AuthorizedDefinition(authentication = false, authorization = false)
    @ResponseBody
    @RequestMapping(name = "默认租户密码策略列表", value = PATH_GLOBAL_PREFIX + "/strategies/password-patterns", method = { RequestMethod.GET })
    public ResponseEntity<Set<String>> passwordPatterns() {
        return ResponseEntity.ok(PasswordPatternStrategy.PATTERN_MAP.keySet());
    }

    @AuthorizedDefinition(authentication = false, authorization = false)
    @ResponseBody
    @RequestMapping(name = "登录账号最佳租户密码策略列表", value = PATH_GLOBAL_PREFIX + "/accounts/{loginName}/password-pattern", method = { RequestMethod.GET })
    public ResponseEntity<String> findPasswordPatternByLoginName(@PathVariable("loginName") String loginName) {
        AccountModel.Request.Login login = new AccountModel.Request.Login();
        login.setUsername(loginName);
        List<Long> tenantIds = accountService.findTenantIdsByLogin(login);
        Map<Long, PasswordPatternStrategy> strategyMap = strategyService.loadStrategiesMap(new HashSet<>(tenantIds), PasswordPatternStrategy.class);
        PasswordPatternStrategy defaultStrategy = new PasswordPatternStrategy();
        strategyMap.values().stream().max((o1, o2) -> ((Comparable) o1).compareTo(o2)).ifPresent(strategy -> BeanUtils.copyProperties(strategy, defaultStrategy));
        Pair<Integer, Pattern> pair = PasswordPatternStrategy.PATTERN_MAP.get(defaultStrategy.getPattern());
        if (pair == null) {
            throw new IllegalArgumentException("错误的参数");
        }
        return ResponseEntity.ok(pair.getValue().pattern());
    }
}
