package com.xforceplus.security.strategy.filter;

import com.xforceplus.entity.Account;
import com.xforceplus.entity.User;
import com.xforceplus.security.login.request.LoginRequest;
import com.xforceplus.security.login.response.LoginTokenResponse;
import com.xforceplus.security.strategy.model.Strategy;
import com.xforceplus.security.login.context.LoginContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 加载User信息, 且登录前安全策略处理器
 * @param <T> 安全策略配置类型
 * @author geewit
 * @since  2022-02-28
 */
public interface LoadUserFilter<T extends Strategy> extends StrategyFilter<T> {
    Logger logger = LoggerFactory.getLogger(LoadUserFilter.class);

    @Override
    default boolean support(LoginContext<? extends LoginRequest> loginContext) {
        if (!StrategyFilter.super.support(loginContext)) {
            return false;
        }
        if (this.getApplicationContext() == null) {
            logger.debug("this {}Filter.applicationContext == null, do nothing", strategyClass().getSimpleName());
            return false;
        }
        LoginRequest request = loginContext.getLoginRequest();
        if (request == null) {
            logger.info("this {}Filter.loginContext.request = null, do nothing", strategyClass().getSimpleName());
            return false;
        }
        return true;
    }

    /**
     * 处理安全策略
     * @param loginContext 登录上下文
     */
    default void executeLoadUser(LoginContext<? extends LoginRequest> loginContext) {
        logger.debug("this {}Filter.executeLoadUser is not override, do nothing", strategyClass().getSimpleName());
    }

    default void loadUser(LoginContext<? extends LoginRequest> loginContext) {
        logger.debug("this {}Filter.loadUser is not override, do nothing", this.strategyClass().getSimpleName());

        this.executeLoadUser(loginContext);

        this.wrapUser(loginContext);
    }

    default void wrapUser(LoginContext<? extends LoginRequest> loginContext) {
        logger.debug("this {}Filter.wrapUser is not override, do default", this.strategyClass().getSimpleName());
        User user = loginContext.getUser();
        if (user == null) {
            throw new IllegalArgumentException("无法加载用户信息");
        }
        Account account = user.getAccount();
        LoginTokenResponse response;
        if (account == null) {
            response = LoginTokenResponse.builder().modifyPasswordTip(false).doubleAuthFlag(false).build();
        } else {
            loginContext.setAccountId(account.getAccountId());
            response = LoginTokenResponse.builder().email(account.getEmail()).telPhone(account.getTelPhone()).build();
            response.setDoubleAuthFlag(account.getDoubleAuthFlag());
            response.setUsername(user.getUsername());
            response.setPasswdLength(account.getPasswdLength());
            response.setModifyPasswordTip(account.getChangePasswordFlag());
        }
        loginContext.setUser(user);
        loginContext.setCurrentTenantId(user.getTenantId());
        loginContext.setResponse(response);
    }
}
