package com.xforceplus.security.strategy.filter;

import com.xforceplus.security.login.context.LoginContext;
import com.xforceplus.security.login.request.LoginRequest;
import com.xforceplus.security.strategy.model.Strategy;
import com.xforceplus.security.strategy.service.StrategyService;
import io.geewit.core.utils.reflection.BeanUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationContext;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Arrays;
import java.util.Map;

/**
 * 安全策略处理器
 *
 * @param <T> 安全策略配置类型
 * @author geewit
 * @since 2022-02-28
 */
public interface StrategyFilter<T extends Strategy> {
    Logger logger = LoggerFactory.getLogger(StrategyFilter.class);

    ApplicationContext getApplicationContext();

    int priority();

    default Class<T> strategyClass() {
        Type[] types = this.getClass().getGenericInterfaces();

        Class<T> clazz = Arrays.stream(types).filter(type -> type instanceof ParameterizedType).filter(type -> StrategyFilter.class.isAssignableFrom((Class<?>) ((ParameterizedType) type).getRawType())).findFirst().map(type -> (Class<T>) ((ParameterizedType) type).getActualTypeArguments()[0]).orElse(null);
        return clazz;
    }

    Class<T> getStrategyClass();

    T defaultStrategy();

    default <S extends Strategy> Map<Long, S> loadStrategyMap(LoginContext<? extends LoginRequest> loginContext, Class<S> strategyClass) {
        StrategyService strategyService = this.getApplicationContext().getBean(StrategyService.class);

        Map<Long, S> strategiesMap = strategyService.loadStrategiesMap(loginContext, strategyClass);
        return strategiesMap;
    }

    default T loadCurrentStrategy(LoginContext<? extends LoginRequest> loginContext) {
        T defaultStrategy = this.defaultStrategy();
        logger.debug("defaultStrategy = {}", defaultStrategy);
        T strategy = this.loadCurrentStrategy(loginContext, this.strategyClass());
        if (strategy != null) {
            logger.debug("strategy = {}", strategy);
            BeanUtils.copyProperties(strategy, defaultStrategy);
        }
        return defaultStrategy;
    }

    default <S extends Strategy> S loadCurrentStrategy(LoginContext<? extends LoginRequest> loginContext, Class<S> strategyClass) {
        Map<Long, S> strategyMap = this.loadStrategyMap(loginContext, strategyClass);
        S strategy = null;
        if (strategyMap != null && !strategyMap.isEmpty()) {
            //region 如果是可排序的,则排序后取第一个策略
            if (Comparable.class.isAssignableFrom(this.strategyClass())) {
                strategy = strategyMap.values().stream().max((o1, o2) -> ((Comparable) o1).compareTo(o2)).orElse(null);
            }
            //endregion
            else if (loginContext.getCurrentTenantId() != null && loginContext.getCurrentTenantId() > 0) {
                S currentStrategy = strategyMap.get(loginContext.getCurrentTenantId());
                logger.debug("currentStrategy = {}", currentStrategy);
                if (currentStrategy != null) {
                    strategy = currentStrategy;
                }
            }
            if (strategy == null) {
                strategy = strategyMap.get(0L);
            }
        }
        return strategy;
    }

    /**
     * 检查登录上下文是否需要执行当前策略处理器
     *
     * @param loginContext
     * @return
     */
    default boolean support(LoginContext<? extends LoginRequest> loginContext) {
        logger.info("this {}Filter.support is not override, do nothing", strategyClass().getSimpleName());
        if (loginContext == null) {
            logger.info("this {}Filter.loginContext = null, do nothing", strategyClass().getSimpleName());
            return false;
        }
        return true;
    }
}
