package com.xforceplus.security.strategy.filter;

import com.xforceplus.business.reponse.code.Rep;
import com.xforceplus.security.login.request.LoginRequest;
import com.xforceplus.security.strategy.model.Strategy;
import com.xforceplus.security.login.context.LoginContext;
import com.xforceplus.security.login.exception.AuthenticationException;
import lombok.Builder;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 安全策略处理器执行链
 *
 * @author geewit
 */
@Slf4j
@Builder
@Getter
public class StrategyFilterChain<T extends LoginRequest> {
    /**
     * 安全策略处理器列表
     */
    private List<StrategyFilter<? extends Strategy>> filters;

    /**
     * 登录上下文
     */
    private LoginContext<T> loginContext;

    /**
     * 添加安全策略处理器列表
     *
     * @param filter 安全策略处理器
     * @return 安全策略处理器执行链
     */
    public StrategyFilterChain<T> addFilter(StrategyFilter<? extends Strategy> filter) {
        if (this.filters == null) {
            this.filters = Stream.of(filter).collect(Collectors.toList());
        } else {
            this.filters.add(filter);
        }
        return this;
    }

    /**
     * 链式执行 安全策略处理器列表
     */
    public void doFilters() {
        this.filters.sort(Comparator.comparing(StrategyFilter::priority));

        List<ValidationFilter<? extends Strategy>> validationFilters = new ArrayList<>();
        List<LoadUserFilter<? extends Strategy>> loadUserFilters = new ArrayList<>();
        List<PostLoadPredicationFilter<? extends Strategy>> postLoadPredicationFilters = new ArrayList<>();
        List<PostLoadValidationFilter<? extends Strategy>> postLoadValidationFilters = new ArrayList<>();
        List<GenerateTokenFilter> generateTokenFilters = new ArrayList<>();
        List<PostGenerateTokenFilter<? extends Strategy>> postGenerateTokenFilters = new ArrayList<>();
        List<PostLoginSuccessFilter<? extends Strategy>> postLoginSuccessFilters = new ArrayList<>();
        List<PostLoginFailFilter<? extends Strategy>> postLoginFailFilters = new ArrayList<>();

        if (CollectionUtils.isNotEmpty(this.filters)) {
            for (StrategyFilter<? extends Strategy> strategyFilter : this.filters) {
                if (strategyFilter instanceof ValidationFilter) {
                    validationFilters.add((ValidationFilter<? extends Strategy>) strategyFilter);
                }
                if (strategyFilter instanceof LoadUserFilter) {
                    loadUserFilters.add((LoadUserFilter<? extends Strategy>) strategyFilter);
                }
                if (strategyFilter instanceof PostLoadPredicationFilter) {
                    postLoadPredicationFilters.add((PostLoadPredicationFilter<? extends Strategy>) strategyFilter);
                }
                if (strategyFilter instanceof PostLoadValidationFilter) {
                    postLoadValidationFilters.add((PostLoadValidationFilter<? extends Strategy>) strategyFilter);
                }
                if (strategyFilter instanceof GenerateTokenFilter) {
                    generateTokenFilters.add((GenerateTokenFilter) strategyFilter);
                }
                if (strategyFilter instanceof PostGenerateTokenFilter) {
                    postGenerateTokenFilters.add((PostGenerateTokenFilter<? extends Strategy>) strategyFilter);
                }
                if (strategyFilter instanceof PostLoginSuccessFilter) {
                    postLoginSuccessFilters.add((PostLoginSuccessFilter<? extends Strategy>) strategyFilter);
                }
                if (strategyFilter instanceof PostLoginFailFilter) {
                    postLoginFailFilters.add((PostLoginFailFilter<? extends Strategy>) strategyFilter);
                }
            }
        }

        try {
            //region 执行校验报文的处理器
            if (CollectionUtils.isNotEmpty(validationFilters)) {
                validationFilters.forEach(filter -> {
                    if (filter.support(this.loginContext)) {
                        filter.valid(this.loginContext);
                    }
                });
            }
            //endregion

            //region 执行加载User的处理器
            if (CollectionUtils.isNotEmpty(loadUserFilters)) {
                loadUserFilters.forEach(filter -> {
                    if (filter.support(this.loginContext)) {
                        filter.loadUser(this.loginContext);
                    }
                });
            }
            //endregion

            //region 执行加载完User的预判处理器
            if (CollectionUtils.isNotEmpty(postLoadPredicationFilters)) {
                postLoadPredicationFilters.forEach(filter -> {
                    if (filter.support(this.loginContext)) {
                        filter.postLoadPredicate(this.loginContext);
                    }
                });
            }
            //endregion

            //region 执行加载完User的处理器
            if (CollectionUtils.isNotEmpty(postLoadValidationFilters)) {
                postLoadValidationFilters.forEach(filter -> {
                    if (filter.support(this.loginContext)) {
                        filter.postLoadValid(this.loginContext);
                    }
                });
            }
            //endregion
        } catch (IllegalArgumentException e) {
            try {
                log.info(e.getMessage());
                //region 执行加载完User的处理器
                if (CollectionUtils.isNotEmpty(postLoginFailFilters)) {
                    postLoginFailFilters.forEach(filter -> {
                        if (filter.support(this.loginContext)) {
                            filter.postLoginFail(this.loginContext);
                        }
                    });
                }
                //endregion
                throw new AuthenticationException(Rep.AccountCode.FAIL, e.getMessage());
            } catch (AuthenticationException ae) {
                throw ae;
            }
        } catch (AuthenticationException e) {
            log.info(e.getMessage());
            throw e;
        }

        //region 执行生成token处理器
        if (CollectionUtils.isNotEmpty(generateTokenFilters)) {
            generateTokenFilters.forEach(filter -> {
                if (filter.support(this.loginContext)) {
                    filter.generateToken(this.loginContext);
                }
            });
        }
        //endregion

        //region 执行生成完Token的处理器
        if (CollectionUtils.isNotEmpty(postGenerateTokenFilters)) {
            postGenerateTokenFilters.forEach(filter -> {
                if (filter.support(this.loginContext)) {
                    filter.postGenerateToken(this.loginContext);
                }
            });
        }
        //endregion

        //region 执行登录完成后的处理器
        if (CollectionUtils.isNotEmpty(postLoginSuccessFilters)) {
            postLoginSuccessFilters.forEach(filter -> {
                if (filter.support(this.loginContext)) {
                    filter.postLogin(this.loginContext);
                }
            });
        }
        //endregion
    }
}
