package com.xforceplus.security.strategy.filter.impl;

import com.xforceplus.domain.account.AccountType;
import com.xforceplus.entity.Account;
import com.xforceplus.entity.User;
import com.xforceplus.security.strategy.filter.GenerateTokenFilter;
import com.xforceplus.security.login.request.LoginRequest;
import com.xforceplus.security.login.response.LoginTokenResponse;
import com.xforceplus.security.strategy.filter.AbstractStrategyFilter;
import com.xforceplus.security.strategy.filter.PostLoginSuccessFilter;
import com.xforceplus.security.strategy.model.GenerateTokenStrategy;
import com.xforceplus.security.login.context.LoginContext;
import com.xforceplus.security.login.service.TenantSecretService;
import com.xforceplus.security.strategy.service.StrategyService;
import com.xforceplus.tenant.security.token.encoder.JwtEncoder;
import com.xforceplus.utils.RegExUtil;
import io.geewit.utils.uuid.UUID;
import lombok.experimental.SuperBuilder;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.digest.Md5Crypt;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.redis.core.StringRedisTemplate;

import java.util.concurrent.TimeUnit;

/**
 * token 生成策略执行器
 * @author geewit
 */
@Slf4j
@SuperBuilder
public class GenerateTokenStrategyFilter extends AbstractStrategyFilter<GenerateTokenStrategy>
        implements GenerateTokenFilter, PostLoginSuccessFilter<GenerateTokenStrategy> {

    /**
     * 租户策略定义需维护用户token状态的redis前缀
     */
    public static final String TENANT_USER_TOKEN_PREFIX = "tenant:user:token:";

    /**
     * 租户策略定义需要拉黑用户token的redis前缀
     */
    public static final String TENANT_USER_TOKEN_BLACK_LIST_PREFIX = "tenant:user:token:blacklist:";

    public static final String MD5_SALT = "$1$/v6QjxiQ";

    @Override
    public GenerateTokenStrategy defaultStrategy() {
        return new GenerateTokenStrategy();
    }

    @Override
    public boolean support(LoginContext<? extends LoginRequest> loginContext) {
        log.debug("execute {}Filter.support", this.strategyClass().getSimpleName());
        if (!GenerateTokenFilter.super.support(loginContext)) {
            return false;
        }
        if (loginContext.getUser() == null) {
            log.debug("this {}Filter.loginContext.user = null, do nothing", this.strategyClass().getSimpleName());
            return false;
        }
        return true;
    }

    @Override
    public void executeGenerateToken(LoginContext<? extends LoginRequest> loginContext) {
        log.debug("execute {}Filter.executeGenerateToken", this.strategyClass().getSimpleName());
        User tokenUser = loginContext.getUser();

        if (StringUtils.isBlank(tokenUser.getLoginId())) {
            tokenUser.setLoginId(UUID.randomUUID().toString());
        }
        if (StringUtils.isNotBlank(tokenUser.getUserName())) {
            tokenUser.setUsername(tokenUser.getUserName());
        } else if (StringUtils.isBlank(tokenUser.getUsername())) {
            if (StringUtils.isNotBlank(tokenUser.getEmail())) {
                tokenUser.setUsername(tokenUser.getEmail());
            } else if (StringUtils.isNotBlank(tokenUser.getMobile())) {
                tokenUser.setUsername(tokenUser.getMobile());
            }
        }
        // username与account.userName一致逻辑先注释掉
//        if(StringUtils.isBlank(tokenUser.getUserName())) {
//            tokenUser.setUserName(tokenUser.getUsername());
//        }
        Account account = tokenUser.getAccount();
        if (StringUtils.isNotBlank(account.getUsername())) {
            tokenUser.setLoginName(account.getUsername());
        } else if (StringUtils.isNotBlank(account.getEmail())) {
            tokenUser.setLoginName(account.getEmail());
        } else if (StringUtils.isNotBlank(account.getTelPhone())) {
            tokenUser.setLoginName(account.getTelPhone());
        }
        String defaultSecret = applicationContext.getEnvironment().getProperty("xforce.tenant.security.jwt.secret", "my_sessionjw_tsecret_xdfdffdsdfdfs");
        String token = JwtEncoder.encode(tokenUser, defaultSecret, TenantSecretService.TENANT_AND_SECRET);
        loginContext.setToken(token);
        LoginTokenResponse response = loginContext.getResponse();
        response.setToken(token);
        response.setPasswdLength(account.getPasswdLength());
        response.setUsername(loginContext.getLoginName());
        response.setTelPhone(account.getTelPhone());
        response.setEmail(account.getEmail());
        boolean isTel = account.getTelPhone() != null && RegExUtil.checkMobile(account.getTelPhone());
        boolean isEmail = account.getEmail() != null && RegExUtil.checkEmail(account.getEmail());
        AccountType accountType = (isTel || isEmail) ? AccountType.PHONE_EMAIL : AccountType.OTHER;
        response.setAccountType(accountType);
        loginContext.setSuccess(true);
    }

    /**
     * 异步更新最近登录时间
     *
     * @param loginContext 登录信息上下文
     */
    @Override
    public void executePostLoginSuccess(LoginContext<? extends LoginRequest> loginContext) {
        log.debug("execute {}Filter.executePostLoginSuccess", this.strategyClass().getSimpleName());
        StringRedisTemplate redisTemplate = applicationContext.getBean(StringRedisTemplate.class);
        StrategyService strategyService = applicationContext.getBean(StrategyService.class);
        Long currentTenantId = loginContext.getCurrentTenantId();
        if (currentTenantId == null) {
            log.debug("execute {}Filter.currentTenantId == null, do nothing", this.strategyClass().getSimpleName());
            return;
        }
        GenerateTokenStrategy strategy = strategyService.loadStrategy(currentTenantId, GenerateTokenStrategy.class);
        if (strategy != null && strategy.isEnabled()) {
            if ((strategy).isSingleSession()) {
                Long userId = loginContext.getUser().getId();
                log.debug("userId = {}", userId);
                String tokenRedisKey = TENANT_USER_TOKEN_PREFIX + currentTenantId + ":" + userId;
                String lastUserToken = redisTemplate.opsForValue().get(tokenRedisKey);
                String token = loginContext.getToken();
                String tokenMd5 = Md5Crypt.md5Crypt(token.getBytes(), MD5_SALT);
                if (null != lastUserToken) {
                    String blackListRedisKey = TENANT_USER_TOKEN_BLACK_LIST_PREFIX + currentTenantId + ":" + userId;
                    log.debug("blackListRedisKey = {}", blackListRedisKey);
                    boolean lastUserTokenBlacked = redisTemplate.opsForSet().isMember(blackListRedisKey, lastUserToken);
                    // 如果用户旧token不在拉黑集合中,则旧token加入黑名单10个小时
                    if (!lastUserTokenBlacked && !lastUserToken.equals(tokenMd5)) {
                        redisTemplate.opsForSet().add(blackListRedisKey, lastUserToken);
                        // 重置用户黑名单token有效期
                        redisTemplate.expire(blackListRedisKey, 10L, TimeUnit.HOURS);
                        log.info("用户id:{}已在其他地方登录，前一次登录token被拉黑,其md5值为: {}", userId, lastUserToken);
                    }
                }

                redisTemplate.opsForValue().set(tokenRedisKey, tokenMd5, 10L, TimeUnit.HOURS);
            }
        }
    }

}
