package com.xforceplus.security.strategy.filter.impl;

import com.xforceplus.business.reponse.code.Rep;
import com.xforceplus.business.tenant.service.UserService;
import com.xforceplus.entity.Account;
import com.xforceplus.entity.User;
import com.xforceplus.security.strategy.filter.AbstractStrategyFilter;
import com.xforceplus.security.strategy.filter.LoadUserFilter;
import com.xforceplus.security.login.request.LoginRequest;
import com.xforceplus.security.login.request.PasswordLoginRequest;
import com.xforceplus.security.strategy.model.LoadUserByPasswordStrategy;
import com.xforceplus.security.login.context.LoginContext;
import com.xforceplus.security.login.exception.AuthenticationException;
import com.xforceplus.utils.DateUtils;
import lombok.experimental.SuperBuilder;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 密码登录安全策略处理器
 * @author geewit
 */
@Slf4j
@SuperBuilder
public class LoadUserByPasswordStrategyFilter extends AbstractStrategyFilter<LoadUserByPasswordStrategy> implements LoadUserFilter<LoadUserByPasswordStrategy> {

    @Override
    public int priority() {
        return this.priority;
    }

    @Override
    public LoadUserByPasswordStrategy defaultStrategy() {
        return new LoadUserByPasswordStrategy();
    }

    @Override
    public boolean support(LoginContext<? extends LoginRequest> loginContext) {
        log.debug("execute {}Filter.support", this.strategyClass().getSimpleName());
        if (!LoadUserFilter.super.support(loginContext)) {
            return false;
        }
        LoginRequest loginRequest = loginContext.getLoginRequest();
        if (loginRequest == null) {
            log.debug("this {}Filter.loginContext.loginRequest = null, do nothing", this.strategyClass().getSimpleName());
            return false;
        }
        if (!(loginRequest instanceof PasswordLoginRequest)) {
            log.debug("this {}Filter.loginContext.loginRequest not instanceof PasswordLoginRequest, do nothing", this.strategyClass().getSimpleName());
            return false;
        }
        return true;
    }

    /**
     * 
     * @param loginContext 登录上下文
     */
    @Override
    public void executeLoadUser(LoginContext<? extends LoginRequest> loginContext) {
        log.debug("execute {}Filter.executeLoadUser", this.strategyClass().getSimpleName());
        PasswordLoginRequest loginRequest = (PasswordLoginRequest)loginContext.getLoginRequest();
        String username = loginRequest.getUsername();
        if (StringUtils.isBlank(username)) {
            throw new AuthenticationException(Rep.AccountCode.FAIL, "登录用户名不能为空");
        }
        if (loginContext.getLoginName() == null) {
            loginContext.setLoginName(username);
        }
        UserService userService = applicationContext.getBean(UserService.class);
        User user;
        if (StringUtils.isNotBlank(loginRequest.getTenantCode())) {
            user = userService.findByTenantCodeAndUsername(loginRequest.getTenantCode(), username, 0);
            if (user != null) {
                Long tenantId = user.getTenantId();
                user.setTenantIds(Stream.of(tenantId).collect(Collectors.toSet()));
            }
        } else {
            user = userService.findByUsername(username, 0);
        }
        if (user == null) {
            throw new IllegalArgumentException("用户名密码错误");
        }
        loginContext.setAccountId(user.getAccountId());
        loginContext.setTenantIds(user.getTenantIds());
        if (user.getExpiredDate() != null && DateUtils.beforeNow(user.getExpiredDate(), false)) {
            throw new AuthenticationException(Rep.AccountCode.FAIL, "用户已过期");
        }
        Account account = user.getAccount();
        user.setModules(loginRequest.getModules());
        loginContext.setUser(user);
        loginContext.setAccount(account);
        loginContext.setCurrentTenantId(user.getTenantId());
    }
}
