package com.xforceplus.security.strategy.filter.impl;

import com.xforceplus.business.reponse.code.Rep;
import com.xforceplus.business.tenant.service.UserService;
import com.xforceplus.entity.User;
import com.xforceplus.security.login.context.LoginContext;
import com.xforceplus.security.login.exception.AuthenticationException;
import com.xforceplus.security.login.request.LoginRequest;
import com.xforceplus.security.login.request.SmsLoginRequest;
import com.xforceplus.security.strategy.filter.AbstractStrategyFilter;
import com.xforceplus.security.strategy.filter.LoadUserFilter;
import com.xforceplus.security.strategy.model.LoadUserBySmsStrategy;
import com.xforceplus.utils.DateUtils;
import com.xforceplus.utils.RegExUtil;
import com.xforececlound.message.api.SmsAuthCodeApi;
import com.xforececlound.message.model.SmsValidateReq;
import com.xforececlound.message.model.SmsValidateResp;
import lombok.experimental.SuperBuilder;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.context.ApplicationContext;

/**
 * 双因子校验规则
 *
 * @author geewit
 */
@Slf4j
@SuperBuilder
public class LoadUserBySmsStrategyFilter extends AbstractStrategyFilter<LoadUserBySmsStrategy>
        implements LoadUserFilter<LoadUserBySmsStrategy> {

    /**
     * 二次认证密码
     */
    private static final String TWO_FACTOR_PWD_PREFIX = "TWO_FACTOR_PWD_";

    /**
     * 二次认证手机/邮箱验证码
     */
    private static final String TWO_FACTOR_CODE_PREFIX = "TWO_FACTOR_CODE_";

    private int priority;

    private ApplicationContext applicationContext;

    @Override
    public ApplicationContext getApplicationContext() {
        return applicationContext;
    }

    @Override
    public int priority() {
        return this.priority;
    }

    @Override
    public LoadUserBySmsStrategy defaultStrategy() {
        return new LoadUserBySmsStrategy();
    }

    @Override
    public boolean support(LoginContext<? extends LoginRequest> loginContext) {
        log.debug("execute {}Filter.support", this.strategyClass().getSimpleName());
        if (!LoadUserFilter.super.support(loginContext)) {
            return false;
        }
        LoginRequest loginRequest = loginContext.getLoginRequest();
        if (loginRequest == null) {
            log.debug("this {}Filter.loginContext.loginRequest = null, do nothing", this.strategyClass().getSimpleName());
            return false;
        }
        if (!(loginRequest instanceof SmsLoginRequest)) {
            log.debug("this {}Handler.loginContext.loginRequest not instanceof SmsLoginRequest, do nothing", this.strategyClass().getSimpleName());
            return false;
        }

        return true;
    }

    /**
     * @param loginContext 登录信息上下文
     */
    @Override
    public void executeLoadUser(LoginContext<? extends LoginRequest> loginContext) {
        log.debug("execute {}Filter.executeLoadUser", this.strategyClass().getSimpleName());
        SmsLoginRequest loginRequest = (SmsLoginRequest) loginContext.getLoginRequest();
        loginContext.setLoginName(loginRequest.getPhone());
        LoadUserBySmsStrategy strategy = this.loadCurrentStrategy(loginContext);
        if (strategy == null || !strategy.isEnabled()) {
            log.debug("execute {}Filter.strategy disabled, do nothing", this.strategyClass().getSimpleName());
            return;
        }
        if (StringUtils.isBlank(loginRequest.getMsgId())) {
            throw new AuthenticationException(Rep.AccountCode.FAIL, "请先发送验证码。");
        }
        String phone = loginRequest.getPhone();
        if (!RegExUtil.checkMobile(phone)) {
            throw new AuthenticationException(Rep.AccountCode.FAIL, "请输入正确的手机号码！");
        }
        loginContext.setLoginName(phone);
        User user;
        try {
            UserService userService = applicationContext.getBean(UserService.class);
            user = userService.findByUsername(phone, 0);
        } catch (Exception e) {
            throw new AuthenticationException(Rep.AccountCode.FAIL, "验证失败，请检查输入信息！");
        }
        if (user == null) {
            throw new AuthenticationException(Rep.AccountCode.FAIL, "请输入正确的手机号码！");
        }
        if (user.getExpiredDate() != null && DateUtils.beforeNow(user.getExpiredDate(), false)) {
            throw new AuthenticationException(Rep.AccountCode.FAIL, "用户已过期");
        }
        SmsValidateReq smsValidateReq = new SmsValidateReq();
        smsValidateReq.setMsgId(loginRequest.getMsgId());
        smsValidateReq.setMobile(loginRequest.getPhone());
        smsValidateReq.setCode(loginRequest.getSmsCode());
        SmsAuthCodeApi smsAuthCodeApi = applicationContext.getBean(SmsAuthCodeApi.class);
        SmsValidateResp smsValidateResp = smsAuthCodeApi.validate(String.valueOf(user.getTenantId()), smsValidateReq);
        boolean checked = smsValidateResp != null && smsValidateResp.isCheck();
        if (!checked) {
            throw new AuthenticationException(Rep.AccountCode.FAIL, "验证码验证失败。");
        }
        loginContext.setUser(user);
        loginContext.setAccount(user.getAccount());
        loginContext.setTenantIds(user.getTenantIds());
        loginContext.setCurrentTenantId(user.getTenantId());
    }

}
