package com.xforceplus.security.strategy.filter.impl;

import com.xforceplus.business.reponse.code.Rep;
import com.xforceplus.entity.Account;
import com.xforceplus.security.login.exception.AuthenticationException;
import com.xforceplus.security.login.request.LoginRequest;
import com.xforceplus.security.login.response.LoginTokenResponse;
import com.xforceplus.security.strategy.filter.AbstractStrategyFilter;
import com.xforceplus.security.strategy.filter.PostLoadValidationFilter;
import com.xforceplus.security.strategy.model.PasswordExpiredStrategy;
import com.xforceplus.security.login.context.LoginContext;
import lombok.experimental.SuperBuilder;
import lombok.extern.slf4j.Slf4j;

import java.sql.Date;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;

/**
 * 密码过期校验规则处理器
 * @author geewit
 */
@Slf4j
@SuperBuilder
public class PasswordExpiredStrategyFilter extends AbstractStrategyFilter<PasswordExpiredStrategy>
        implements PostLoadValidationFilter<PasswordExpiredStrategy> {

    @Override
    public PasswordExpiredStrategy defaultStrategy() {
        return new PasswordExpiredStrategy();
    }

    @Override
    public boolean support(LoginContext<? extends LoginRequest> loginContext) {
        log.debug("execute {}Filter.support", this.strategyClass().getSimpleName());
        if (!PostLoadValidationFilter.super.support(loginContext)) {
            return false;
        }
        LoginRequest loginRequest = loginContext.getLoginRequest();
        if (loginRequest == null) {
            log.debug("this {}Filter.loginContext.loginRequest = null, do nothing", this.strategyClass().getSimpleName());
            return false;
        }
        Account account = loginContext.getAccount();
        if (account == null) {
            log.debug("this {}Filter.loginContext.account = null, do nothing", this.strategyClass().getSimpleName());
            return false;
        }
        return true;
    }

    /**
     * 
     * @param loginContext 登录上下文
     */
    @Override
    public void executePostLoadValid(LoginContext<? extends LoginRequest> loginContext) {
        log.debug("execute {}Filter.executePostLoadValid", this.strategyClass().getSimpleName());
        PasswordExpiredStrategy strategy = this.loadCurrentStrategy(loginContext);
        if (strategy == null || !strategy.isEnabled()) {
            log.debug("execute {}Filter.strategy disabled, do nothing", this.strategyClass().getSimpleName());
            return;
        }
        Account account = loginContext.getAccount();
        LoginTokenResponse response = loginContext.getResponse();
        if (response == null) {
            response = LoginTokenResponse.builder().build();
            loginContext.setResponse(response);
        }

        int expireDays = strategy.getExpireDays() != null && strategy.getExpireDays() > 0 ? strategy.getExpireDays() : PasswordExpiredStrategy.DEFAULT_EXPIRE_DAYS;
        int expireRemindDays = strategy.getExpireRemindDays() != null && strategy.getExpireRemindDays() > 0 ? strategy.getExpireRemindDays() : PasswordExpiredStrategy.DEFAULT_REMIND_DAYS;
        //region 网关原有逻辑 如果密码过期 则返回需要修改密码开关
        if (account.getPwdLastUpdateTime() != null) {
            LocalDate pwdLastUpdateDate = account.getPwdLastUpdateTime().toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
            boolean modifyPasswordTip = pwdLastUpdateDate.plusDays(expireDays).compareTo(LocalDate.now()) <= 0;
            response.setModifyPasswordTip(modifyPasswordTip);
            response.setUsername(loginContext.getLoginName());
            if (modifyPasswordTip) {
                throw new AuthenticationException(Rep.AccountCode.CHANGE_PASSWORD, "请修改密码", response);
            }
            boolean pwdExpireFlag = pwdLastUpdateDate.plusDays(expireDays - expireRemindDays).compareTo(LocalDate.now()) <= 0;
            if (pwdExpireFlag) {
                log.debug("this {}Filter.pwdExpireFlag = true", this.strategyClass().getSimpleName());
                LocalDate expireDate = pwdLastUpdateDate.plusDays(expireDays);
                Instant expireInstant = expireDate.atStartOfDay().atZone(ZoneId.systemDefault()).toInstant();
                response.setPwdExpireDate(Date.from(expireInstant));
            }
        } else {
            log.debug("this {}Filter.account.getPwdLastUpdateTime() != null, modifyPasswordTip = false", strategyClass().getSimpleName());
            response.setModifyPasswordTip(false);
        }
        //endregion

    }
}
