package com.xforceplus.security.strategy.filter.impl;

import com.xforceplus.security.login.response.LoginTokenResponse;
import com.xforceplus.security.strategy.filter.AbstractStrategyFilter;
import com.xforceplus.security.strategy.filter.PostLoadValidationFilter;
import com.xforceplus.security.login.request.LoginPasswordRequest;
import com.xforceplus.security.login.request.LoginRequest;
import com.xforceplus.security.strategy.model.PasswordPatternStrategy;
import com.xforceplus.security.login.context.LoginContext;
import lombok.experimental.SuperBuilder;
import lombok.extern.slf4j.Slf4j;

import java.util.regex.Pattern;

/**
 * 密码格式校验策略处理器
 * @author geewit
 */
@Slf4j
@SuperBuilder
public class PasswordPatternStrategyFilter extends AbstractStrategyFilter<PasswordPatternStrategy>
        implements PostLoadValidationFilter<PasswordPatternStrategy> {

    @Override
    public PasswordPatternStrategy defaultStrategy() {
        return new PasswordPatternStrategy();
    }

    @Override
    public boolean support(LoginContext<? extends LoginRequest> loginContext) {
        log.debug("execute {}Filter.support", this.strategyClass().getSimpleName());
        if (!PostLoadValidationFilter.super.support(loginContext)) {
            return false;
        }
        LoginRequest loginRequest = loginContext.getLoginRequest();
        if (loginRequest == null) {
            log.debug("this {}Filter.loginContext.loginRequest = null, do nothing", this.strategyClass().getSimpleName());
            return false;
        }
        if (!(loginRequest instanceof LoginPasswordRequest)) {
            log.debug("this {}Filter.loginContext.loginRequest not instanceof LoginPasswordRequest, do nothing", this.strategyClass().getSimpleName());
            return false;
        }
        return true;
    }

    /**
     * 
     * @param loginContext 登录上下文
     */
    @Override
    public void executePostLoadValid(LoginContext<? extends LoginRequest> loginContext) {
        log.debug("execute {}Filter.executePostLoadValid", this.strategyClass().getSimpleName());
        LoginPasswordRequest loginRequest = (LoginPasswordRequest)loginContext.getLoginRequest();

        String password = loginRequest.getPassword();
        if (password == null) {
            throw new IllegalArgumentException("登录密码不能为空");
        }
        PasswordPatternStrategy strategy = this.loadCurrentStrategy(loginContext);
        if (strategy == null || !strategy.isEnabled()) {
            log.debug("execute {}Filter.strategy disabled, do nothing", this.strategyClass().getSimpleName());
            return;
        }
        Pattern pattern = PasswordPatternStrategy.findPattern(strategy.getPattern());
        if (!pattern.matcher(password).matches()) {
            throw new IllegalArgumentException(strategy.getMessage());
        }

        LoginTokenResponse response = loginContext.getResponse();
        if (response != null) {
            if (response.getPasswdLength() != null && response.getPasswdLength() > 0) {
                return;
            }
        } else {
            response = LoginTokenResponse.builder().build();
            loginContext.setResponse(response);
        }

        response.setPasswdLength(strategy.minLength());
    }
}
