package com.xforceplus.security.strategy.filter.impl;

import com.xforceplus.entity.Account;
import com.xforceplus.security.login.context.LoginContext;
import com.xforceplus.security.login.request.LoginPasswordRequest;
import com.xforceplus.security.login.request.LoginRequest;
import com.xforceplus.security.strategy.filter.AbstractStrategyFilter;
import com.xforceplus.security.strategy.filter.PostLoadValidationFilter;
import com.xforceplus.security.strategy.model.LoadUserByPasswordStrategy;
import com.xforceplus.tenant.security.core.utils.CryptUtils;
import com.xforceplus.utils.AESHelp;
import lombok.experimental.SuperBuilder;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;


/**
 * 密码 校验规则安全策略处理器
 * @author geewit
 */
@Slf4j
@SuperBuilder
public class PasswordValidationStrategyFilter extends AbstractStrategyFilter<LoadUserByPasswordStrategy> implements PostLoadValidationFilter<LoadUserByPasswordStrategy> {

    @Override
    public LoadUserByPasswordStrategy defaultStrategy() {
        return new LoadUserByPasswordStrategy();
    }

    @Override
    public boolean support(LoginContext<? extends LoginRequest> loginContext) {
        log.debug("execute {}Filter.support", this.getStrategyClass().getSimpleName());
        if (!PostLoadValidationFilter.super.support(loginContext)) {
            return false;
        }
        LoginRequest loginRequest = loginContext.getLoginRequest();
        if (loginRequest == null) {
            log.debug("{}Filter.loginContext.loginRequest = null, do nothing", this.getStrategyClass().getSimpleName());
            return false;
        }
        if (!(loginRequest instanceof LoginPasswordRequest)) {
            log.debug("{}Filter.loginContext.loginRequest not instanceof LoginPasswordRequest, do nothing", this.strategyClass().getSimpleName());
            return false;
        }
        if (loginContext.getAccount() == null) {
            log.debug("{}Filter.loginContext.account = null, do nothing", this.getStrategyClass().getSimpleName());
            return false;
        }
        return true;
    }
    
    /**
     * @param loginContext 待校验的账号id
     */
    @Override
    public void executePostLoadValid(LoginContext<? extends LoginRequest> loginContext) {
        log.debug("{}Filter.executePostLoadValid", this.strategyClass().getSimpleName());
        LoginPasswordRequest loginRequest = (LoginPasswordRequest)loginContext.getLoginRequest();

        Account account = loginContext.getAccount();
        String password;
        if (loginContext.isNeedEncrypt()) {
            try {
                password = AESHelp.desEncrypt(AESHelp.CREDENTIAL_KEY, loginRequest.getPassword());
                loginRequest.setPassword(password);
            } catch (Exception e) {
                throw new IllegalArgumentException("用户名密码错误");
            }
        } else {
            password = loginRequest.getPassword();
        }
        String salt = applicationContext.getEnvironment().getProperty("tenant.security.password.salt", StringUtils.EMPTY);
        String encryptedPassword = CryptUtils.encryptPassword(account.getSalt(), password, salt);
        if (!encryptedPassword.equals(account.getPassword())) {
            throw new IllegalArgumentException("用户名密码错误");
        }
    }
}
