package com.xforceplus.security.strategy.filter.impl;

import com.xforceplus.business.reponse.code.Rep;
import com.xforceplus.domain.account.AccountType;
import com.xforceplus.entity.Account;
import com.xforceplus.entity.User;
import com.xforceplus.security.login.exception.AuthenticationException;
import com.xforceplus.security.login.request.LoginRequest;
import com.xforceplus.security.login.request.PasswordLoginRequest;
import com.xforceplus.security.login.response.LoginTokenResponse;
import com.xforceplus.security.strategy.filter.AbstractStrategyFilter;
import com.xforceplus.security.strategy.filter.PostLoadValidationFilter;
import com.xforceplus.security.strategy.model.PostLoadUserStrategy;
import com.xforceplus.security.login.context.LoginContext;
import com.xforceplus.utils.DateUtils;
import com.xforceplus.utils.RegExUtil;
import lombok.experimental.SuperBuilder;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

/**
 * 加载完Account后的安全策略处理器
 * @author geewit
 */
@Slf4j
@SuperBuilder
public class PostLoadUserStrategyFilter extends AbstractStrategyFilter<PostLoadUserStrategy>
        implements PostLoadValidationFilter<PostLoadUserStrategy> {

    @Override
    public PostLoadUserStrategy defaultStrategy() {
        return new PostLoadUserStrategy();
    }

    @Override
    public boolean support(LoginContext<? extends LoginRequest> loginContext) {
        log.debug("execute {}Filter.support", this.strategyClass().getSimpleName());
        if (!PostLoadValidationFilter.super.support(loginContext)) {
            return false;
        }
        if (loginContext.getUser() == null || loginContext.getAccount() == null) {
            throw new AuthenticationException(Rep.AccountCode.FAIL, "获取用户失败");
        }
        return true;
    }

    /**
     * 
     * @param loginContext 登录上下文
     */
    @Override
    public void executePostLoadValid(LoginContext<? extends LoginRequest> loginContext) {
        log.debug("execute {}Filter.executePostLoadValid", this.strategyClass().getSimpleName());
        User user = loginContext.getUser();
        Account account = loginContext.getAccount();
        PostLoadUserStrategy strategy = this.loadCurrentStrategy(loginContext);
        if (strategy == null || !strategy.isEnabled()) {
            log.debug("execute {}Filter.strategy disabled, do nothing", this.strategyClass().getSimpleName());
            return;
        }
        if (strategy.isCheckAccountStatus() && (account.getStatus() == null || account.getStatus() != 1)) {
            throw new AuthenticationException(Rep.AccountCode.FAIL, "账号未启用");
        }
        if (strategy.isCheckUserStatus() && (user.getStatus() == null || user.getStatus() != 1)) {
            throw new AuthenticationException(Rep.AccountCode.FAIL, "用户未启用");
        }
        if (user.getExpiredDate() != null && DateUtils.beforeNow(user.getExpiredDate(), false)) {
            throw new AuthenticationException(Rep.AccountCode.FAIL, "用户已过期");
        }
        LoginRequest loginRequest = loginContext.getLoginRequest();
        if (loginRequest instanceof PasswordLoginRequest) {
            boolean isTel = account.getTelPhone() != null && RegExUtil.checkMobile(account.getTelPhone());
            boolean isEmail = account.getEmail() != null && RegExUtil.checkEmail(account.getEmail());
            AccountType accountType = (isTel || isEmail) ? AccountType.PHONE_EMAIL : AccountType.OTHER;
            LoginTokenResponse response = loginContext.getResponse();
            if (response == null) {
                response = LoginTokenResponse.builder().build();
                loginContext.setResponse(response);
            }
            response.setAccountType(accountType);
            response.setPasswdLength(account.getPasswdLength());
            if ((account.getChangePasswordFlag() != null && account.getChangePasswordFlag())) {
                response.setUsername(StringUtils.defaultString(loginContext.getLoginName(), account.getUsername()));
                throw new AuthenticationException(Rep.AccountCode.CHANGE_PASSWORD, "请修改密码", response);
            }
            if ((account.getBindAuthFlag() != null && account.getBindAuthFlag())) {
                response.setUsername(StringUtils.defaultString(loginContext.getLoginName(), account.getUsername()));
                throw new AuthenticationException(Rep.AccountCode.BIND_AUTH, "绑定手机或者邮箱", response);
            }
        }
    }
}
