package com.xforceplus.security.strategy.filter.impl;

import com.xforceplus.entity.User;
import com.xforceplus.security.login.request.LoginRequest;
import com.xforceplus.security.strategy.filter.AbstractStrategyFilter;
import com.xforceplus.security.strategy.filter.PostGenerateTokenFilter;
import com.xforceplus.security.strategy.model.ResponseCookieStrategy;
import com.xforceplus.security.login.context.LoginContext;
import com.xforceplus.security.utils.CookieUtils;
import com.xforceplus.tenant.security.token.domain.UserType;
import lombok.experimental.SuperBuilder;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.http.ResponseCookie;
import org.springframework.web.util.UriUtils;

import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.HashSet;
import java.util.Set;

/**
 * token生产后响应加cookie的安全策略处理器
 * @author geewit
 */
@Slf4j
@SuperBuilder
public class ResponseCookieStrategyFilter extends AbstractStrategyFilter<ResponseCookieStrategy>
        implements PostGenerateTokenFilter<ResponseCookieStrategy> {

    @Override
    public ResponseCookieStrategy defaultStrategy() {
        return new ResponseCookieStrategy();
    }

    @Override
    public boolean support(LoginContext<? extends LoginRequest> loginContext) {
        log.debug("execute {}Filter.support", this.strategyClass().getSimpleName());
        return PostGenerateTokenFilter.super.support(loginContext);
    }

    /**
     * 
     * @param loginContext 登录上下文
     */
    @Override
    public void executePostGenerateToken(LoginContext<? extends LoginRequest> loginContext) {
        log.debug("execute {}Filter.executePostGenerateToken", this.strategyClass().getSimpleName());
        ResponseCookieStrategy strategy = this.loadCurrentStrategy(loginContext);
        if (strategy == null || !strategy.isEnabled()) {
            log.debug("execute {}Filter.strategy disabled, do nothing", this.strategyClass().getSimpleName());
            return;
        }
        String domain = StringUtils.defaultString(strategy.getDomain(), loginContext.getHost());
        int maxAge = strategy.getMaxAge() != null && strategy.getMaxAge() > 0 ? strategy.getMaxAge() : ResponseCookieStrategy.DEFAULT_MAXAGE;
        Duration duration = Duration.ofHours(maxAge);

        if (StringUtils.isNotBlank(domain)) {
            Set<ResponseCookie> responseCookies = new HashSet<>();
            if (StringUtils.isNotBlank(loginContext.getToken())) {
                responseCookies.add(CookieUtils.buildCookie(UserType.USER.tokenKey(), loginContext.getToken(), duration, domain, strategy.isTopDomain()));
            }
            User user = loginContext.getUser();
            responseCookies.add(CookieUtils.buildCookie("userId", String.valueOf(user.getId()), duration, domain, strategy.isTopDomain()));
            if (StringUtils.isNotBlank(user.getUsername())) {
                responseCookies.add(CookieUtils.buildCookie(UserType.USER.usernameKey(), UriUtils.encode(user.getUsername(), StandardCharsets.UTF_8), duration, domain, strategy.isTopDomain()));
                responseCookies.add(CookieUtils.buildCookie("userName", UriUtils.encode(user.getUsername(), StandardCharsets.UTF_8), duration, domain, strategy.isTopDomain()));
            } else {
                responseCookies.add(CookieUtils.buildCookie("userName", UriUtils.encode(" ", StandardCharsets.UTF_8), duration, domain, strategy.isTopDomain()));
            }
            loginContext.setCookies(responseCookies);
        }
    }
}
