package com.xforceplus.security.strategy.filter.impl;

import com.xforceplus.api.model.AccountModel;
import com.xforceplus.business.reponse.code.Rep;
import com.xforceplus.business.tenant.service.UserService;
import com.xforceplus.entity.Account;
import com.xforceplus.entity.User;
import com.xforceplus.security.strategy.filter.AbstractStrategyFilter;
import com.xforceplus.security.strategy.filter.LoadUserFilter;
import com.xforceplus.security.login.request.LoginRequest;
import com.xforceplus.security.login.request.TwoFactorPasswordLoginRequest;
import com.xforceplus.security.strategy.model.TwoFactorStrategy;
import com.xforceplus.security.login.context.LoginContext;
import com.xforceplus.security.login.exception.AuthenticationException;
import lombok.experimental.SuperBuilder;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.redis.core.StringRedisTemplate;

/**
 * 双因子密码校验规则和加载用户规则
 * @author geewit
 */
@Slf4j
@SuperBuilder
public class TwoFactorPasswordLoadUserStrategyFilter extends AbstractStrategyFilter<TwoFactorStrategy>
        implements LoadUserFilter<TwoFactorStrategy> {

    /**
     * 二次认证密码
     */
    private static final String TWO_FACTOR_PWD_PREFIX = "TWO_FACTOR_PWD_";

    @Override
    public TwoFactorStrategy defaultStrategy() {
        return new TwoFactorStrategy();
    }

    @Override
    public boolean support(LoginContext<? extends LoginRequest> loginContext) {
        log.debug("execute {}Filter.support", this.strategyClass().getSimpleName());
        if (!LoadUserFilter.super.support(loginContext)) {
            return false;
        }
        if (loginContext.getLoginRequest() == null) {
            log.debug("execute {}Filter.loginContext.loginRequest = null, do nothing", this.strategyClass().getSimpleName());
            return false;
        }
        if (!(loginContext.getLoginRequest() instanceof TwoFactorPasswordLoginRequest)) {
            log.debug("execute {}Filter.loginContext.loginRequest not instanceof TwoFactorPasswordLoginRequest, do nothing", this.strategyClass().getSimpleName());
            return false;
        }
        return true;
    }

    /**
     * @param loginContext 登录信息上下文
     */
    @Override
    public void executeLoadUser(LoginContext<? extends LoginRequest> loginContext) {
        log.debug("execute {}Filter.executeLoadUser", this.strategyClass().getSimpleName());
        TwoFactorPasswordLoginRequest request = (TwoFactorPasswordLoginRequest)loginContext.getLoginRequest();
        if (StringUtils.isBlank(request.getPassword())) {
            throw new AuthenticationException(Rep.AccountCode.FAIL, "密码不能为空！");
        }
        String processId = request.getProcessId();
        String redisKey = TWO_FACTOR_PWD_PREFIX + processId;
        StringRedisTemplate redisTemplate = applicationContext.getBean(StringRedisTemplate.class);
        String username = redisTemplate.opsForValue().get(redisKey);
        if (StringUtils.isBlank(username)) {
            throw new AuthenticationException(Rep.AccountCode.FAIL, "请先进行手机号登录验证！");
        }
        if (loginContext.getLoginName() == null) {
            loginContext.setLoginName(username);
        }

        AccountModel.Request.Login login = new AccountModel.Request.Login();
        login.setUsername(loginContext.getLoginName());
        User user = loginContext.getUser();
        if (user == null) {
            UserService userService = applicationContext.getBean(UserService.class);
            user = userService.findByUsername(username, 0);
        }
        if (user == null) {
            throw new AuthenticationException(Rep.AccountCode.FAIL, "手机或者邮箱有误，无法登录。");
        }
        loginContext.setTenantIds(user.getTenantIds());
        loginContext.setCurrentTenantId(user.getTenantId());
        TwoFactorStrategy strategy = this.loadCurrentStrategy(loginContext);
        log.debug("strategy: {}", strategy);
        Account account = user.getAccount();
        boolean enableTwoFactor = account.getDoubleAuthFlag() != null && account.getDoubleAuthFlag();
        log.debug("enableTwoFactor = {}", enableTwoFactor);
        if ((strategy == null || !strategy.isEnabled()) && !enableTwoFactor) {
            log.info("execute {}Filter.strategy disabled, do nothing", this.strategyClass().getSimpleName());
            throw new AuthenticationException(Rep.AccountCode.FAIL, "租户策略无需双因子认证");
        }

        loginContext.setUser(user);
        loginContext.setAccount(account);
        Long tenantId = user.getTenantId();
        loginContext.setCurrentTenantId(tenantId);
        Long accountId = user.getAccountId();
        loginContext.setAccountId(accountId);

    }
}
