package com.xforceplus.security.strategy.filter.impl;

import com.xforceplus.business.reponse.code.Rep;
import com.xforceplus.business.tenant.service.UserService;
import com.xforceplus.entity.Account;
import com.xforceplus.entity.User;
import com.xforceplus.security.login.context.LoginContext;
import com.xforceplus.security.login.exception.AuthenticationException;
import com.xforceplus.security.login.request.LoginRequest;
import com.xforceplus.security.login.request.TwoFactorSmsLoginRequest;
import com.xforceplus.security.strategy.filter.AbstractStrategyFilter;
import com.xforceplus.security.strategy.filter.LoadUserFilter;
import com.xforceplus.security.strategy.model.TwoFactorStrategy;
import com.xforceplus.utils.RegExUtil;
import com.xforececlound.message.api.EmailAuthCodeApi;
import com.xforececlound.message.api.SmsAuthCodeApi;
import com.xforececlound.message.model.EmailValidateReq;
import com.xforececlound.message.model.SmsValidateReq;
import com.xforececlound.message.model.SmsValidateResp;
import lombok.experimental.SuperBuilder;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.redis.core.StringRedisTemplate;

/**
 * 双因子短信加载用户规则
 *
 * @author geewit
 */
@Slf4j
@SuperBuilder
public class TwoFactorSmsLoadUserStrategyFilter extends AbstractStrategyFilter<TwoFactorStrategy>
        implements LoadUserFilter<TwoFactorStrategy> {

    /**
     * 二次认证密码
     */
    private static final String TWO_FACTOR_PWD_PREFIX = "TWO_FACTOR_PWD_";

    /**
     * 二次认证手机/邮箱验证码
     */
    private static final String TWO_FACTOR_CODE_PREFIX = "TWO_FACTOR_CODE_";

    @Override
    public TwoFactorStrategy defaultStrategy() {
        return new TwoFactorStrategy();
    }

    @Override
    public boolean support(LoginContext<? extends LoginRequest> loginContext) {
        log.debug("execute {}Filter.support", this.strategyClass().getSimpleName());
        if (!LoadUserFilter.super.support(loginContext)) {
            return false;
        }
        LoginRequest loginRequest = loginContext.getLoginRequest();
        if (loginRequest == null) {
            log.debug("{}Filter.loginContext.loginRequest = null, do nothing", this.strategyClass().getSimpleName());
            return false;
        }
        if (!(loginRequest instanceof TwoFactorSmsLoginRequest)) {
            log.debug("{}Filter.loginContext.loginRequest not instanceof TwoFactorSmsLoginRequest, do nothing", this.strategyClass().getSimpleName());
            return false;
        }
        return true;
    }

    /**
     * @param loginContext 登录信息上下文
     */
    @Override
    public void executeLoadUser(LoginContext<? extends LoginRequest> loginContext) {
        log.debug("execute {}Filter.executeLoadUser", this.strategyClass().getSimpleName());
        TwoFactorSmsLoginRequest request = (TwoFactorSmsLoginRequest) loginContext.getLoginRequest();
        String processId = request.getProcessId();
        String key = TWO_FACTOR_CODE_PREFIX + processId;
        StringRedisTemplate redisTemplate = applicationContext.getBean(StringRedisTemplate.class);
        String username = redisTemplate.opsForValue().get(key);
        if (StringUtils.isBlank(username)) {
            throw new AuthenticationException(Rep.AccountCode.FAIL, "请先进行用户名密码登录！");
        }

        if (StringUtils.isBlank(request.getMsgId()) || StringUtils.isBlank(request.getCipher())) {
            throw new AuthenticationException(Rep.AccountCode.FAIL, "验证码不能为空!");
        }

        User user = loginContext.getUser();
        if (user == null) {
            UserService userService = applicationContext.getBean(UserService.class);
            user = userService.findByUsername(username, 0);
        }

        if (user == null) {
            throw new AuthenticationException(Rep.AccountCode.FAIL, "手机或者邮箱有误，无法登录。");
        }
        loginContext.setUser(user);
        Account account = user.getAccount();
        loginContext.setAccount(account);
        loginContext.setTenantIds(user.getTenantIds());
        Long tenantId = user.getTenantId();
        loginContext.setCurrentTenantId(tenantId);
        TwoFactorStrategy strategy = this.loadCurrentStrategy(loginContext);
        boolean enableTwoFactor = account.getDoubleAuthFlag() != null && account.getDoubleAuthFlag();
        log.debug("enableTwoFactor = {}", enableTwoFactor);
        if ((strategy == null || !strategy.isEnabled()) && !enableTwoFactor) {
            log.info("execute {}Filter.strategy disabled, do nothing", this.strategyClass().getSimpleName());
            throw new AuthenticationException(Rep.AccountCode.FAIL, "租户策略无需双因子认证");
        }

        Long accountId = user.getAccountId();
        loginContext.setAccountId(accountId);

        boolean checked;
        try {
            if (RegExUtil.checkMobile(username)) {
                SmsValidateReq smsValidateReq = new SmsValidateReq();
                smsValidateReq.setMobile(username);
                smsValidateReq.setCode(request.getCipher());
                smsValidateReq.setMsgId(request.getMsgId());
                SmsAuthCodeApi smsAuthCodeApi = applicationContext.getBean(SmsAuthCodeApi.class);
                SmsValidateResp smsValidateResp = smsAuthCodeApi.validate(String.valueOf(user.getTenantId()), smsValidateReq);
                checked = smsValidateResp.isCheck();
            } else if (RegExUtil.checkEmail(username)) {
                EmailValidateReq emailValidateReq = new EmailValidateReq();
                emailValidateReq.setEmail(username);
                emailValidateReq.setCode(request.getCipher());
                emailValidateReq.setMsgId(request.getMsgId());
                EmailAuthCodeApi emailAuthCodeApi = applicationContext.getBean(EmailAuthCodeApi.class);
                SmsValidateResp smsValidateResp = emailAuthCodeApi.validate(String.valueOf(user.getTenantId()), emailValidateReq);
                checked = smsValidateResp.isCheck();
            } else {
                throw new AuthenticationException(Rep.AccountCode.FAIL, "账号必须为手机或者邮箱！");
            }
        } catch (Exception e) {
            log.error("valid.auth.code.error:{}", username, e);
            throw new AuthenticationException(Rep.AccountCode.FAIL, "验证码验证异常");
        }

        if (!checked) {
            throw new AuthenticationException(Rep.AccountCode.FAIL, "验证码验证失败");
        }

    }
}
