package com.xforceplus.security.strategy.filter.impl;

import com.xforceplus.business.reponse.code.Rep;
import com.xforceplus.entity.Account;
import com.xforceplus.security.login.request.LoginRequest;
import com.xforceplus.security.login.request.PasswordLoginRequest;
import com.xforceplus.security.login.response.LoginFailResponse;
import com.xforceplus.security.login.request.SmsLoginRequest;
import com.xforceplus.security.login.response.LoginTokenResponse;
import com.xforceplus.security.strategy.filter.AbstractStrategyFilter;
import com.xforceplus.security.strategy.filter.PostLoadValidationFilter;
import com.xforceplus.security.strategy.model.TwoFactorType;
import com.xforceplus.security.login.context.LoginContext;
import com.xforceplus.security.strategy.model.TwoFactorStrategy;
import com.xforceplus.security.login.exception.AuthenticationException;
import io.geewit.utils.uuid.UUIDUtils;
import lombok.experimental.SuperBuilder;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.redis.core.StringRedisTemplate;

import java.util.concurrent.TimeUnit;

/**
 * 双因子(同时适配短信和密码)加载用户后校验规则
 * @author geewit
 */
@Slf4j
@SuperBuilder
public class TwoFactorValidStrategyFilter extends AbstractStrategyFilter<TwoFactorStrategy>
        implements PostLoadValidationFilter<TwoFactorStrategy> {

    /**
     * 二次认证密码
     */
    private static final String TWO_FACTOR_PWD_PREFIX = "TWO_FACTOR_PWD_";

    /**
     * 二次认证手机/邮箱验证码
     */
    private static final String TWO_FACTOR_CODE_PREFIX = "TWO_FACTOR_CODE_";

    @Override
    public TwoFactorStrategy defaultStrategy() {
        return new TwoFactorStrategy();
    }

    @Override
    public boolean support(LoginContext<? extends LoginRequest> loginContext) {
        log.debug("execute {}Filter.support", this.strategyClass().getSimpleName());
        if (!PostLoadValidationFilter.super.support(loginContext)) {
            return false;
        }
        if (loginContext.getLoginRequest() == null) {
            log.debug("this {}Filter.loginContext.loginRequest = null, do nothing", this.strategyClass().getSimpleName());
            return false;
        }

        if (!(loginContext.getLoginRequest() instanceof PasswordLoginRequest) && !(loginContext.getLoginRequest() instanceof SmsLoginRequest)) {
            log.debug("this {}Filter.loginContext.loginRequest not instanceof PasswordLoginRequest and not instanceof SmsLoginRequest, do nothing", this.strategyClass().getSimpleName());
            return false;
        }
        return true;
    }

    /**
     * @param loginContext 登录信息上下文
     */
    @Override
    public void executePostLoadValid(LoginContext<? extends LoginRequest> loginContext) {
        log.debug("execute {}Filter.executePostLoadValid", this.strategyClass().getSimpleName());
        Account account = loginContext.getAccount();
        boolean enableTwoFactor = account.getDoubleAuthFlag() != null && account.getDoubleAuthFlag();
        TwoFactorStrategy strategy = this.loadCurrentStrategy(loginContext);
        enableTwoFactor = enableTwoFactor || strategy.isEnabled();
        log.debug("execute {}Filter.enableTwoFactor = {}", this.strategyClass().getSimpleName(), enableTwoFactor);
        if (!enableTwoFactor) {
            return;
        }
        StringRedisTemplate redisTemplate = applicationContext.getBean(StringRedisTemplate.class);
        //region 密码登录逻辑
        if (loginContext.getLoginRequest() instanceof PasswordLoginRequest) {
            log.debug("{}Filter.loginContext.loginRequest not instanceof PasswordLoginRequest, do nothing", this.strategyClass().getSimpleName());
            String senderKey = null;

            if (TwoFactorType.MOBILE.equals(strategy.getPerferType())) {
                if (!StringUtils.isBlank(account.getTelPhone())) {
                    senderKey = account.getTelPhone();
                } else if (!StringUtils.isBlank(account.getEmail())) {
                    senderKey = account.getEmail();
                }
            } else {
                if (!StringUtils.isBlank(account.getEmail())) {
                    senderKey = account.getEmail();
                } else if (!StringUtils.isBlank(account.getTelPhone())) {
                    senderKey = account.getTelPhone();
                }
            }
            if (senderKey != null) {
                String processId = UUIDUtils.randomUUID();
                String key = TWO_FACTOR_CODE_PREFIX + processId;
                int timeout = strategy.getTimeout() != null && strategy.getTimeout() > 0 ? strategy.getTimeout() : TwoFactorStrategy.DEFAULT_TIMEOUT;
                redisTemplate.opsForValue().set(key, senderKey, timeout, TimeUnit.MINUTES);
                LoginFailResponse response = LoginFailResponse.builder()
                        .username(senderKey)
                        .processId(processId)
                        .build();
                throw new AuthenticationException(Rep.AccountCode.DOUBLE_AUTH_EMAIL_PHONE, "二次认证手机号/或邮箱", response);
            }
        }
        //endregion

        //region 短信登录逻辑
        else if (loginContext.getLoginRequest() instanceof SmsLoginRequest) {
            log.debug("{}Filter.loginContext.loginRequest instanceof SmsLoginRequest", this.strategyClass().getSimpleName());
            String processId = UUIDUtils.randomUUID();
            String key = TWO_FACTOR_PWD_PREFIX + processId;
            SmsLoginRequest loginRequest = (SmsLoginRequest)loginContext.getLoginRequest();
            int timeout = strategy.getTimeout() != null && strategy.getTimeout() > 0 ? strategy.getTimeout() : TwoFactorStrategy.DEFAULT_TIMEOUT;
            redisTemplate.opsForValue().set(key, loginRequest.getPhone(), timeout, TimeUnit.MINUTES);
            throw new AuthenticationException(Rep.AccountCode.DOUBLE_AUTH_PWD, "二次认证密码", LoginFailResponse.builder().processId(processId).build());
        }
        //endregion

        LoginTokenResponse response = loginContext.getResponse();
        if (response == null) {
            response = LoginTokenResponse.builder().build();
            loginContext.setResponse(response);
        }
        response.setDoubleAuthFlag(true);
    }

}
