package com.xforceplus.security.strategy.filter.impl;

import com.google.common.eventbus.Subscribe;
import com.xforceplus.api.model.LoginLogModel;
import com.xforceplus.business.account.service.LoginLogService;
import com.xforceplus.security.login.request.LoginRequest;
import com.xforceplus.security.strategy.filter.AbstractStrategyFilter;
import com.xforceplus.security.strategy.model.UpdateLoginTimeStrategy;
import com.xforceplus.security.login.context.LoginContext;
import com.xforceplus.security.strategy.filter.PostLoginSuccessFilter;
import lombok.experimental.SuperBuilder;
import lombok.extern.slf4j.Slf4j;

import java.util.Calendar;

/**
 * 登录成功后异步保存 loginTime
 * @author geewit
 */
@Slf4j
@SuperBuilder
public class UpdateLoginTimeStrategyFilter extends AbstractStrategyFilter<UpdateLoginTimeStrategy>
        implements PostLoginSuccessFilter<UpdateLoginTimeStrategy> {

    public void init() {
        ASYNC_EVENT_BUS.register(this);
    }

    @Override
    public UpdateLoginTimeStrategy defaultStrategy() {
        return new UpdateLoginTimeStrategy();
    }

    @Override
    public boolean support(LoginContext<? extends LoginRequest> loginContext) {
        log.debug("execute {}Filter.support", this.strategyClass().getSimpleName());
        if (!PostLoginSuccessFilter.super.support(loginContext)) {
            return false;
        }
        if (loginContext.getAccountId() == null) {
            log.debug("{}Filter.loginContext.accountId = null, do nothing", this.strategyClass().getSimpleName());
            return false;
        }
        if (loginContext.getUser() == null) {
            log.debug("{}Filter.loginContext.user = null, do nothing", this.strategyClass().getSimpleName());
            return false;
        }
        if (loginContext.getLoginName() == null) {
            log.debug("{}Filter.loginContext.loginName = null, do nothing", this.strategyClass().getSimpleName());
            return false;
        }
        return true;
    }

    /**
     * 异步更新最近登录时间
     * @param loginContext 登录信息上下文
     */
    @Override
    public void executePostLoginSuccess(LoginContext<? extends LoginRequest> loginContext) {
        log.debug("execute {}Filter.executePostLoginSuccess", this.strategyClass().getSimpleName());
        if (loginContext == null || !loginContext.getSuccess()) {
            log.debug("this {}Filter.loginContext.isSuccess = false, do nothing", strategyClass().getSimpleName());
            return;
        }
        UpdateLoginTimeStrategy strategy = this.loadCurrentStrategy(loginContext);
        if (strategy == null || !strategy.isEnabled()) {
            log.debug("execute {}Filter.strategy disabled, do nothing", this.strategyClass().getSimpleName());
            return;
        }
        LoginLogModel.Request.Save save = new LoginLogModel.Request.Save();
        Long accountId = loginContext.getAccountId();
        Long userId = loginContext.getUser().getId();
        save.setAccountId(accountId);
        save.setUserId(userId);
        save.setLoginName(loginContext.getLoginName());
        save.setLoginTime(Calendar.getInstance().getTime());
        ASYNC_EVENT_BUS.post(save);
    }

    @Subscribe
    public void doProcess(LoginLogModel.Request.Save save) {
        try {
            LoginLogService loginLogService = applicationContext.getBean(LoginLogService.class);
            loginLogService.create(save);
        } catch (Exception e) {
            log.warn(e.getMessage());
        }
    }
}
