package com.xforceplus.security.strategy.model;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.annotations.ApiModelProperty;
import lombok.Getter;
import lombok.Setter;
import lombok.ToString;
import org.apache.commons.lang3.ObjectUtils;
import org.jetbrains.annotations.NotNull;

import java.util.Calendar;

/**
 * 密码登录错次校验安全策略
 * @author geewit
 */
@JsonIgnoreProperties(ignoreUnknown = true)
@JsonInclude(JsonInclude.Include.NON_NULL)
@ToString
@Setter
@Getter
public class AccountLoginFailStrategy implements Strategy, Comparable<AccountLoginFailStrategy> {
    public static final String ACCOUNT_LOGIN_FAILS_PREFIX = "ACCOUNT_LOGIN_FAILS_";

    public static final int DEFAULT_THRESHOLD = 10;
    public static final int DEFAULT_TIMEOUT = 30;

    @Override
    public String getName() {
        return AccountLoginFailStrategy.class.getSimpleName();
    }

    @Override
    public String getDescription() {
        return "密码登录错次校验安全策略";
    }

    /**
     * 需要锁定账户错次阈值
     */
    @ApiModelProperty(value = "需要锁定账户错次阈值", example = "10")
    private Integer threshold = DEFAULT_THRESHOLD;

    /**
     * 锁定账户时长 (MINUTE)
     */
    @ApiModelProperty(value = "锁定账户时长 (MINUTE)", example = "30")
    private Integer timeout = DEFAULT_TIMEOUT;

    private boolean enabled = true;

    @Override
    public int compareTo(@NotNull AccountLoginFailStrategy that) {
        return Integer.compare(ObjectUtils.defaultIfNull(that.threshold, DEFAULT_THRESHOLD),
                ObjectUtils.defaultIfNull(this.threshold, DEFAULT_THRESHOLD));
    }

    public static String lockMessage(int expireMinutes) {
        Calendar expiredCalender = Calendar.getInstance();
        expiredCalender.add(Calendar.MINUTE, expireMinutes);
        String message = String.format("用户名/密码错误，账户已锁定，请于%1$tY-%1$tm-%1$td %1$tH:%1$tM:%1$tS后再试", expiredCalender);
        return message;
    }
}
