package com.xforceplus.security.strategy.model;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.annotations.ApiModelProperty;
import lombok.Getter;
import lombok.Setter;
import lombok.ToString;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

/**
 * 密码格式安全策略
 * @author geewit
 */
@JsonIgnoreProperties(ignoreUnknown = true)
@JsonInclude(JsonInclude.Include.NON_NULL)
@ToString
@Setter
@Getter
public class PasswordPatternStrategy implements Strategy, Comparable<PasswordPatternStrategy> {
    public PasswordPatternStrategy() {
    }

    @Override
    public String getName() {
        return PasswordPatternStrategy.class.getSimpleName();
    }

    @Override
    public String getDescription() {
        return "密码格式安全策略";
    }

    public static final Map<String, Pair<Integer, Pattern>> PATTERN_MAP = new ConcurrentHashMap<>();

    static {
        PATTERN_MAP.put("密码格式为8到16位，英文+数字组合", Pair.of(1, Pattern.compile("^(?=.*[0-9])(?=.*[A-z])[0-9A-z!?@#$%^&*_-]{8,16}$")));
        PATTERN_MAP.put("密码格式为12到16位，英文+数字组合", Pair.of(2, Pattern.compile("^(?=.*[0-9])(?=.*[A-z])[0-9A-z!?@#$%^&*_-]{12,16}$")));
        PATTERN_MAP.put("密码格式为8到16位，英文+数字+符号组合", Pair.of(3, Pattern.compile("^(?![0-9!?@#$%^&*_-]+$)(?![A-z!?@#$%^&*_-]+$)(?![0-9A-z]+$)[0-9A-z!?@#$%^&*_-]{8,16}$")));
        PATTERN_MAP.put("密码格式为12到16位，英文+数字+符号组合", Pair.of(4, Pattern.compile("^(?![0-9!?@#$%^&*_-]+$)(?![A-z!?@#$%^&*_-]+$)(?![0-9A-z]+$)[0-9A-z!?@#$%^&*_-]{12,16}$")));
    }

    public static Pattern findPattern(String patternValue) {
        Pair<Integer, Pattern> pair = PasswordPatternStrategy.PATTERN_MAP.get(patternValue);
        Pattern pattern;
        if (pair == null) {
            try {
                pattern = Pattern.compile(patternValue);
                pair = Pair.of(PasswordPatternStrategy.PATTERN_MAP.size(), pattern);
                PasswordPatternStrategy.PATTERN_MAP.put(patternValue, pair);
            } catch (PatternSyntaxException e) {
                throw new IllegalArgumentException("非法的正则表达式");
            }
        } else {
            pattern = pair.getValue();
        }
        return pattern;
    }

    @ApiModelProperty(value = "最近使用过的密码保存条数", example = "密码格式为8到16位，英文+数字组合")
    private String pattern = "密码格式为8到16位，英文+数字组合";

    public String getMessage() {
        return "密码格式不正确，" + pattern + "，请重新输入。";
    }

    @JsonIgnore
    public int minLength() {
        Pattern pattern = findPattern(this.pattern);
        if (pattern == null) {
            return 0;
        }
        String patternValue = pattern.pattern();
        String lastMinLengthValue = StringUtils.substringAfterLast(patternValue, "{");
        lastMinLengthValue = StringUtils.substringBefore(lastMinLengthValue, ",");
        int lastMinLength;
        try {
            lastMinLength = Integer.parseInt(lastMinLengthValue);
        } catch (NumberFormatException e) {
            lastMinLength = 0;
        }
        return lastMinLength;
    }

    @Override
    public int compareTo(@NotNull PasswordPatternStrategy strategy) {
        Pair<Integer, Pattern> thisPair = PasswordPatternStrategy.PATTERN_MAP.get(this.pattern);
        Integer thisIndex = thisPair != null ? thisPair.getKey() : Integer.MAX_VALUE;
        Pair<Integer, Pattern> thatPair = PasswordPatternStrategy.PATTERN_MAP.get(strategy.pattern);
        Integer thatIndex = thatPair != null ? thatPair.getKey() : Integer.MAX_VALUE;
        return Integer.compare(thisIndex, thatIndex);
    }
}
