package com.xforceplus.security.strategy.service;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.xforceplus.dao.TenantPolicyDao;
import com.xforceplus.entity.TenantPolicy;
import com.xforceplus.event.publisher.MessagePublisher;
import com.xforceplus.security.login.context.LoginContext;
import com.xforceplus.security.login.request.LoginRequest;
import com.xforceplus.security.strategy.model.*;
import io.geewit.utils.uuid.UUIDUtils;
import io.geewit.web.utils.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Service;

import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.*;

/**
 * @author geewit
 */
@Slf4j
@Service
public class StrategyService {

    private final MessagePublisher messagePublisher;
    private final TenantPolicyDao tenantPolicyDao;
    private final LoadingCache<Class<? extends Strategy>, Map<Long, ? extends Strategy>> strategyCache;

    public StrategyService(@Qualifier("refreshingStrategyRedisPublisher") MessagePublisher messagePublisher, TenantPolicyDao tenantPolicyDao, Environment environment) {
        this.messagePublisher = messagePublisher;
        this.tenantPolicyDao = tenantPolicyDao;
        this.strategyCache = CacheBuilder
                .newBuilder().expireAfterWrite(Duration.of(environment.getProperty("xforce.tenant.strategy.cache.time-live-minutes", Long.class, 3L), ChronoUnit.MINUTES))
                .build(new CacheLoader<Class<? extends Strategy>, Map<Long, ? extends Strategy>>() {
                    @Override
                    public Map<Long, ? extends Strategy> load(Class<? extends Strategy> clazz) {
                        return StrategyService.this.loadStrategyMapFromDb((Class<? extends Strategy>) clazz);
                    }
                });
    }

    public <S extends Strategy> S loadStrategy(Long tenantId, Class<S> clazz) {
        log.debug("StrategyService loadStrategy");
        tenantId = ObjectUtils.defaultIfNull(tenantId, 0L);
        Map<Long, ? extends Strategy> tenantStrategyMap = strategyCache.getUnchecked(clazz);
        S strategy = (S) tenantStrategyMap.get(tenantId);
        return strategy;
    }

    public <S extends Strategy> Map<Long, S> loadStrategiesMap(LoginContext<? extends LoginRequest> loginContext, Class<S> clazz) {
        Set<Long> tenantIds = loginContext.getTenantIds();
        return this.loadStrategiesMap(tenantIds, clazz);
    }

    public <S extends Strategy> Map<Long, S> loadStrategiesMap(Set<Long> tenantIds, Class<S> clazz) {
        if (tenantIds == null || tenantIds.isEmpty()) {
            log.info("tenantIds == null || tenantIds.isEmpty, return null");
            return null;
        }
        Map<Long, ? extends Strategy> tenantStrategyMap = strategyCache.getUnchecked(clazz);
        Map<Long, S> strategyMap = new HashMap<>();
        tenantIds.forEach(tenantId -> {
            S strategy = (S) tenantStrategyMap.get(tenantId);
            if (strategy != null) {
                strategyMap.put(tenantId, strategy);
            }
        });
        return strategyMap;
    }

    private <S extends Strategy> Map<Long, S> loadStrategyMapFromDb(Class<S> clazz) {
        log.debug("StrategyService loadStrategyMapFromDb");
        String strategyName = clazz.getSimpleName();
        List<TenantPolicy> policies = tenantPolicyDao.findByName(strategyName);
        Map<Long, S> tenantStrategyMap = new HashMap<>();
        for (TenantPolicy tenantPolicy : policies) {
            S strategy = null;
            if (tenantPolicy != null) {
                String policyValue = tenantPolicy.getPolicy();
                log.debug("policyValue = {}", policyValue);
                try {
                    strategy = JsonUtils.fromJson(policyValue, clazz);
                } catch (Exception e) {
                    log.warn("租户策略配置有误(id:{})", tenantPolicy.getId());
                }

            }
            if (strategy != null) {
                tenantStrategyMap.put(tenantPolicy.getTenantId(), strategy);
            }
        }
        return tenantStrategyMap;
    }

    public List<? super Strategy> allStrategies() {
        List<? super Strategy> strategies = new ArrayList<>();
        strategies.add(new AccountLoginFailStrategy());
        strategies.add(new CaptchaStrategy());
        strategies.add(new GenerateTokenStrategy());
        strategies.add(new LoadUserByPasswordStrategy());
        strategies.add(new LoadUserBySmsStrategy());
        strategies.add(new NewAccountStrategy());
        strategies.add(new PasswordExpiredStrategy());
        strategies.add(new PasswordPatternStrategy());
        strategies.add(new PasswordRepetitionStrategy());
        strategies.add(new PostLoadUserStrategy());
        strategies.add(new ResponseCookieStrategy());
        strategies.add(new TwoFactorStrategy());
        strategies.add(new UpdateLoginTimeStrategy());
        return strategies;
    }

    public String refresh() {
        log.debug("execute refresh");
        String refreshId = UUIDUtils.randomUUID();
        messagePublisher.publish(refreshId);
        this.refreshCache();
        return refreshId;
    }

    public void refreshCache() {
        log.debug("execute refreshCache");
        strategyCache.invalidateAll();
    }
}
