package com.xforceplus.utils;

import com.xforceplus.tenant.core.exception.UnknownException;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.multipart.MultipartFile;

import java.io.*;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Objects;
import java.util.stream.Collectors;

@SuppressWarnings("all")
public class FileUtils {

    /**
     * 转换M单位（1024*1024）
     */
    public static final long FILE_SIZE_BYTE_UTIL = 1024 * 1024L;
    /**
     * 默认上传大小为15M，
     */
    public static final long DEFAULT_UPLOAD_MAX_FILE_SIZE = 15L;
    private static final Logger logger = LoggerFactory.getLogger(FileUtils.class);

    /**
     * 检查文件是否大于15（按参数）M，大于15M则throw new 异常
     *
     * @param file        导入文件
     * @param maxFileSize 最大值
     */
    public static void checkFileSize(MultipartFile file, long maxFileSize) {
        //校验文件大小 (byte转换为M)
        long size = file.getSize() / FILE_SIZE_BYTE_UTIL;
        if (size >= maxFileSize) {
            throw new UnknownException("导入文件不能超过15M");
        }
    }

    /**
     * 读取文件列表
     * @param path 路径
     * @return String 文件内容
     * @throws IOException
     */
    public static String getResourcesFile(String path) throws IOException {
        InputStream inputStream = FileUtils.class.getResourceAsStream(path);
        return readFile(inputStream);
    }

    /**
     * 读取文件列表
     * @param path 路径
     * @return String 文件内容
     * @throws IOException
     */
    public static String getFilePath(String path) {
        URL url = FileUtils.class.getResource("/");
        return url.getPath();
    }

    /**
     * 读取配置文件
     */
    public static String readFile(InputStream inputStream) throws IOException {
        InputStreamReader reader = new InputStreamReader(inputStream, StandardCharsets.UTF_8);
        BufferedReader bufferedReader = new BufferedReader(reader);
        String value = bufferedReader.lines().collect(Collectors.joining());
        bufferedReader.close();
        return filter(value);
    }

    /**
     * 过滤输入字符串, 剔除多行注释以及替换掉反斜杠
     */
    private static String filter(String input) {
        return input.replaceAll("/\\*[\\s\\S]*?\\*/", StringUtils.EMPTY);
    }


    /**
     * 按照行读取文件内容
     *
     * @param file
     * @return
     */
    public static String readFileByLine(File file) {
        if (null == file) {
            return null;
        }
        StringBuilder s = new StringBuilder();
        BufferedReader br = null;
        try {
            br = new BufferedReader(new FileReader(file));
            String temp;
            while ((temp = br.readLine()) != null) {
                s.append(temp);
            }
        } catch (Exception e) {
            logger.error("readFileByLine  readLine", e);
        } finally {
            try {
                Objects.requireNonNull(br).close();
            } catch (IOException e) {
                logger.error("readFileByLine requireNonNull ", e);
            }
        }
        return s.toString();
    }

    /**
     * 判断文件大小
     *
     * @param len  文件长度
     * @param size 限制大小
     * @param unit 限制单位（B,K,M,G）
     * @return
     */
    public static boolean checkFileSize(Long len, int size, String unit) {
        double fileSize = 0;
        switch (unit.toUpperCase()) {
            case "B":
                fileSize = (double) len;
                break;
            case "K":
                fileSize = (double) len / 1024;
                break;
            case "M":
                fileSize = (double) len / 1048576;
                break;
            case "G":
                fileSize = (double) len / 1073741824;
                break;
            default: {
                break;
            }
        }
        return fileSize <= size;
    }
}
