package com.xforceplus.utils;

import com.xforceplus.domain.resource.RequestUri;
import com.xforceplus.domain.resource.RequestUriAuthz;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.lang.Nullable;
import org.springframework.util.AntPathMatcher;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;

/**
 * @author geewit
 */
@SuppressWarnings("all")
public class RequestMappingHelper {
    private final static Logger logger = LoggerFactory.getLogger(RequestMappingHelper.class);


    private final static AntPathMatcher PATH_MATCHER = new AntPathMatcher();

    /**
     * @param requestUri
     * @param requestMappings
     * @param useTrailingSlashMatch 是否使用'\'
     * @return
     */
    @Nullable
    public static RequestUriAuthz lookupRequestUri(RequestUri requestUri, Map<RequestUri, RequestUriAuthz> requestMappings, boolean useTrailingSlashMatch) {
        //region Direct match?
        RequestUriAuthz matchResult = requestMappings.get(requestUri);
        if (matchResult != null && !PATH_MATCHER.isPattern(requestUri.getRequestPath())) {
            // Bean name or resolved handler?
            return matchResult;
        }
        //endregion Direct match.

        //region Pattern match?
        List<String> matchingPatterns = new ArrayList<>();
        for (RequestUri registeredRequestUri : requestMappings.keySet()) {
            String registeredPattern = registeredRequestUri.getRequestPath();
            if (requestUri.getRequestMethod().equals(registeredRequestUri.getRequestMethod())) {
                if (PATH_MATCHER.match(registeredPattern, requestUri.getRequestPath())) {
                    matchingPatterns.add(registeredPattern);
                } else if (useTrailingSlashMatch) {
                    if (!registeredPattern.endsWith("/") && PATH_MATCHER.match(registeredPattern + "/", requestUri.getRequestPath())) {
                        matchingPatterns.add(registeredPattern + "/");
                    }
                }
            }
        }

        String bestMatch = null;
        Comparator<String> patternComparator = PATH_MATCHER.getPatternComparator(requestUri.getRequestPath());
        if (!matchingPatterns.isEmpty()) {
            matchingPatterns.sort(patternComparator);
            if (logger.isTraceEnabled() && matchingPatterns.size() > 1) {
                logger.trace("Matching patterns " + matchingPatterns);
            }
            bestMatch = matchingPatterns.get(0);
        }
        if (bestMatch != null) {
            matchResult = requestMappings.get(new RequestUri(bestMatch, requestUri.getRequestMethod()));
            if (matchResult == null) {
                if (bestMatch.endsWith("/")) {
                    matchResult = requestMappings.get(new RequestUri(bestMatch.substring(0, bestMatch.length() - 1), requestUri.getRequestMethod()));
                }
                if (matchResult == null) {
                    throw new IllegalArgumentException(
                            "Could not find best pattern match [" + bestMatch + "]");
                }
            }
        }
        //endregion Pattern match.

        logger.debug("No match result found...");
        return null;
    }
}
