package com.xforceplus.utils.excel;

import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.EasyExcelFactory;
import com.alibaba.excel.ExcelReader;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.analysis.ExcelReadExecutor;
import com.alibaba.excel.read.metadata.ReadSheet;
import com.alibaba.excel.support.ExcelTypeEnum;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import com.xforceplus.utils.excel.listener.ExcelConvertListener;
import com.xforceplus.utils.excel.model.ExcelToJsonProperty;
import com.xforceplus.utils.excel.model.JsonToExcelProperty;
import io.geewit.web.utils.JsonUtils;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.multipart.MultipartFile;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.*;

@SuppressWarnings("all")
public class ExcelConvertUtil {
    private static Logger logger = LoggerFactory.getLogger(ExcelConvertUtil.class);


    public static String excelToJson(ExcelToJsonProperty property) {
        checkProperties(property);
        MultipartFile file = property.getFile();
        String json = property.getJson().replaceAll("\\s+", StringUtils.EMPTY);
        Integer startRow = property.getStartRow();
        Integer startSheet = property.getStartSheet();
        InputStream inputStream = null;

        try {
            inputStream = file.getInputStream();
            String inputJson = generateInputStreamToJson(inputStream, json, startRow, startSheet);
            return inputJson;
        } catch (IOException e) {
            logger.error("excelToJson exception", e);
            throw new IllegalArgumentException(e.getMessage());
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    logger.error("excelToJson Io Exception ", e);
                }
            }
        }
    }

    public static void jsonToExcel(List<JsonToExcelProperty> properties, String fileName) {
        checkJsonToExcelProperty(properties, fileName);
        OutputStream out = null;
        ExcelWriter writer = null;

        try {
            out = new FileOutputStream(fileName);
            writer = EasyExcelFactory.getWriter(out);

            for (int i = 0; i < properties.size(); ++i) {
                JsonToExcelProperty property = properties.get(i);
                String sheetName = StringUtils.isEmpty(property.getSheetName()) ? "sheet" + i++ : property.getSheetName();
                String json = property.getJson();
                Map<String, String> rules = property.getRules();
                List<List<String>> headers = createTestListStringHead(json, rules);
                //replace deprecated api
                WriteSheet writeSheet = EasyExcel.writerSheet(i+1,sheetName).build();
//                Sheet sheet = new Sheet(i + 1, 0);
//                writeSheet.setSheetNo(i + 1);
//                writeSheet.setSheetName(sheetName);
                writeSheet.setHead(headers);
                List<List<Object>> datas = createTestListObject(json, rules);
                writer.write(datas, writeSheet);
            }
        } catch (Exception var20) {
            logger.error("jsonToExcel error :", var20);
        } finally {
            if (writer != null) {
                writer.finish();
            }

            if (out != null) {
                try {
                    out.close();
                } catch (IOException var19) {
                    logger.error("outinputstream close error ", var19);
                }
            }

        }

    }

    private static String generateInputStreamToJson(InputStream inputStream, String json, Integer startRow, Integer startSheet) {
        ExcelConvertListener excelListener = new ExcelConvertListener();
        excelListener.setJson(json);
        ExcelReader excelReader = EasyExcelFactory.getReader(inputStream, excelListener);
        ExcelReadExecutor excelExecutor = excelReader.excelExecutor();
        List<ReadSheet> sheets = excelExecutor.sheetList();
        logger.info("all sheets {}", sheets);
        if (CollectionUtils.isEmpty(sheets)) {
            throw new IllegalArgumentException("current excel has no sheet,please import a right excel!");
        } else {
            List<Map> datas = Lists.newArrayList();

            for (int i = startSheet - 1; i < sheets.size(); ++i) {
                ReadSheet sheet = (ReadSheet) sheets.get(i);
//                sheet.setHeadLineMun(startRow - 1);
                excelReader.read(sheet);
                List<Map> data = excelListener.getData();
                datas.addAll(data);
            }

            if (CollectionUtils.isEmpty(datas)) {
                throw new IllegalArgumentException("convert result is '[]' ,please check your file or request params is correct!");
            } else {
                return JSON.toJSONString(datas);
            }
        }
    }

    private static List<List<Object>> createTestListObject(String json, Map<String, String> rules) {
        List<Map<String, String>> maps = JsonUtils.fromJson(json, new TypeReference<List<Map<String, String>>>() {
        });
        List<List<Object>> datas = Lists.newArrayList();
        Iterator<Map<String, String>> iterator = maps.iterator();

        while (iterator.hasNext()) {
            Map<String, String> map = iterator.next();
            List<String> keys = new ArrayList<>(map.keySet());
            List<String> ruleKeys = new ArrayList<>(rules.keySet());
            List<Object> values = Lists.newArrayList();
            keys.forEach((k) -> {
                if (ruleKeys.contains(k)) {
                    String value = map.get(k);
                    if (StringUtils.isEmpty(value)) {
                        value = "";
                    }

                    values.add(value);
                }

            });
            datas.add(values);
        }

        return datas;
    }

    private static List<List<String>> createTestListStringHead(String json, Map<String, String> rules) {
        List<LinkedHashMap> datas = JSONObject.parseArray(json, LinkedHashMap.class);
        Map<String, String> map = (Map) datas.get(0);
        List<List<String>> headers = Lists.newArrayList();
        Iterator<String> iterator = map.keySet().iterator();

        while (iterator.hasNext()) {
            String key = iterator.next();
            List<String> header = Lists.newArrayList();
            String chinessHeader = rules.get(key);
            if (!StringUtils.isEmpty(chinessHeader)) {
                header.add(chinessHeader);
                headers.add(header);
            }
        }

        return headers;
    }

    private static void checkJsonToExcelProperty(List<JsonToExcelProperty> propertys, String fileName) {
        if (CollectionUtils.isEmpty(propertys)) {
            throw new IllegalArgumentException("properties can not be empty!");
        } else {
            Iterator<JsonToExcelProperty> iterator = propertys.iterator();

            String json;
            do {
                if (!iterator.hasNext()) {
                    Objects.requireNonNull(fileName, "fileName can not be null!");
                    if (!fileName.endsWith(ExcelTypeEnum.XLS.getValue()) && !fileName.endsWith(ExcelTypeEnum.XLSX.getValue())) {
                        throw new IllegalArgumentException("excel must endwith .xls or .xlsx!");
                    }

                    return;
                }

                JsonToExcelProperty property = iterator.next();
                json = property.getJson();
                Map<String, String> rules = property.getRules();
                Objects.requireNonNull(rules, "rules can not be null!");
            } while (!StringUtils.isEmpty(json));

            throw new IllegalArgumentException("json can not be null!");
        }
    }

    private static void checkProperties(ExcelToJsonProperty property) {
        MultipartFile file = property.getFile();
        String json = property.getJson();
        String filename = file.getOriginalFilename();
        Integer startRow = property.getStartRow();
        Integer startSheet = property.getStartSheet();
        Objects.requireNonNull(file, "file can not be null!");
        Objects.requireNonNull(filename, "fileName can not be null!");
        if (StringUtils.isEmpty(json)) {
            throw new IllegalArgumentException("the result json can not be null!");
        } else if (!filename.endsWith(ExcelTypeEnum.XLS.getValue()) && !filename.endsWith(ExcelTypeEnum.XLSX.getValue())) {
            throw new IllegalArgumentException("excel must endwith .xls or .xlsx!");
        } else if (startRow <= 0 || startSheet <= 0) {
            throw new IllegalArgumentException("开始的行startRow 或者 开始读取的sheet 最小要从 1 开始!");
        }
    }

}
