package com.xforceplus.utils.excel;

import com.alibaba.excel.support.ExcelTypeEnum;
import com.xforceplus.utils.FileUtils;
import com.xforceplus.utils.excel.exception.ImportException;
import com.xforceplus.utils.excel.model.ExcelToJsonProperty;
import com.xforceplus.utils.excel.model.JsonToExcelProperty;
import io.geewit.core.exception.ProcessedException;
import io.geewit.web.utils.JsonUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;
import org.springframework.web.multipart.MultipartFile;

import javax.validation.ConstraintViolation;
import javax.validation.Validator;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

@SuppressWarnings("all")
public class ExcelUtils {
    public static final String FILE_SIZE_M = "M";
    /**
     * 日期
     */
    private static final String DATE = "Date";
    /**
     * 文件最大小限制
     */
    private static final int MAX_FILE_SIZE = 20;
    private final static Logger logger = LoggerFactory.getLogger(ExcelUtils.class);

    public static <T> List<T> list(MultipartFile file, String templateName, Class<T> clazz) {

        if (null == file) {
            logger.error("上传文件不存在");
            throw new IllegalArgumentException("上传文件不存在");
        }
        //获取上传文件名,包含后缀
        String filename = file.getOriginalFilename();

        if (!StringUtils.endsWith(filename, ExcelTypeEnum.XLS.getValue()) && !StringUtils.endsWith(filename, ExcelTypeEnum.XLSX.getValue())) {
            throw new IllegalArgumentException("请上传 .xls文件或 .xlsx文件");
        }

        boolean sizeFlag = FileUtils.checkFileSize(file.getSize(), MAX_FILE_SIZE, FILE_SIZE_M);
        if (!sizeFlag) {
            throw new IllegalArgumentException("文件不得大于20M");
        }

        String templateJson = null;
        try {
            templateJson = FileUtils.getResourcesFile("/templates/" + templateName);
        } catch (IOException e) {
            logger.warn(e.getMessage(), e);
        }
        if (null == templateJson) {
            logger.error("没有找到json模板={},", templateName);
            throw new ProcessedException("文件导入失败，请稍后重试");
        }

        ExcelToJsonProperty property = new ExcelToJsonProperty();
        property.setFile(file);
        property.setJson(templateJson);
//        property.setStartRow(1);

        String result;
        try {
            result = ExcelConvertUtil.excelToJson(property);
        } catch (Exception e) {
            logger.error("调用excel转json异常templateName={},Exception={}", templateName, e);
            throw new IllegalArgumentException("Excel解析异常，导入失败:" + e.getMessage());
        }
        if (StringUtils.isBlank(result)) {
            logger.error("result为空templateName={}", templateName);
            throw new IllegalArgumentException("导入失败,excel内容为空");
        }

        List<T> list = new ArrayList<>();
        try {
            list = JsonUtils.toList(result, clazz);
        } catch (RuntimeException e) {
            logger.error("json转list<T>templateJson={},Exception={}", templateJson, e);
//            if(e.getCause() != null && e.getCause() instanceof InvalidFormatException){
//
//            } else
            if (e.getMessage().contains(DATE)) {
                throw new ProcessedException("日期格式不正确");
            } else if (e.getMessage().contains("InvalidFormatException")) {
                throw new ProcessedException("参数格式不正确");
            }
        } catch (Exception e) {
            logger.error("json转list<T>templateJson={},Exception={}", templateJson, e);
            throw new ProcessedException("文件导入失败，请稍后重试");
        }
        return list;
    }

    public static <T> List<String> checkField(T t, Validator validator) {
        Set<ConstraintViolation<T>> constraintViolations = validator.validate(t);
        List<String> msgError = constraintViolations.stream().map(ConstraintViolation::getMessage).collect(Collectors.toCollection(() -> new ArrayList<>(constraintViolations.size())));
        return msgError;
    }

    public static <T> void validField(T t, Validator validator) throws ImportException {
        Set<ConstraintViolation<T>> constraintViolations = validator.validate(t);
        List<String> msgError = constraintViolations.stream().map(ConstraintViolation::getMessage).collect(Collectors.toCollection(() -> new ArrayList<>(constraintViolations.size())));
        if (msgError.size() > 0) {
            throw new ImportException(msgError);
        }
    }


    private static String createTempExcelFile() {
        return StringUtils.appendIfMissing(System.getProperties().getProperty("java.io.tmpdir"), File.separator)
                + System.currentTimeMillis() + ".xlsx";
    }


    public static String buildExcelFile(String json, Map<String, String> rules, String sheetName) {

        if (StringUtils.isBlank(json) || CollectionUtils.isEmpty(rules)) {
            return StringUtils.EMPTY;
        }
        List<JsonToExcelProperty> properties = new ArrayList<>();
        JsonToExcelProperty jsonToExcelProperty = new JsonToExcelProperty();
        jsonToExcelProperty.setJson(json);

        jsonToExcelProperty.setRules(rules);
        jsonToExcelProperty.setSheetName(sheetName);
        properties.add(jsonToExcelProperty);
        String fileName = createTempExcelFile();
        ExcelConvertUtil.jsonToExcel(properties, fileName);
        return fileName;

    }

}
