package com.xforceplus.tenant.data.auth.aop.aspect;


import com.fasterxml.jackson.databind.ObjectMapper;
import com.xforceplus.tenant.data.auth.aop.client.ObjectRuleCheckServiceClient;
import com.xforceplus.tenant.data.domain.authorization.Authorization;
import com.xforceplus.tenant.data.domain.authorization.AuthorizedUser;
import com.xforceplus.tenant.data.domain.context.DataAuth;
import com.xforceplus.tenant.data.domain.context.DataAuthContextHolder;
import com.xforceplus.tenant.data.domain.result.CheckResult;
import com.xforceplus.tenant.data.domain.result.CheckStatus;
import com.xforceplus.tenant.data.rule.object.context.ObjectAgreement;
import com.xforceplus.tenant.security.autoscan.annotation.AuthorizedDefinition;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.tenant.security.token.domain.IRole;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author zhouxin
 */
@Aspect
@Slf4j
@Component
@ConditionalOnProperty(name="uc.data.auth.enabled", havingValue ="true")
public class AuthorizedDefinitionAspect {

    @Resource
    ObjectRuleCheckServiceClient objectRuleCheckServiceClient;

    @Value("${uc.data.auth.appId:0}")
    private Long appId;

    private static ObjectMapper objectMapper = new ObjectMapper();

    @Around("@annotation(authorizedDefinition)")
    public Object around(ProceedingJoinPoint point, AuthorizedDefinition authorizedDefinition) {
        try {
            // pub接口不做限制
            boolean skipAuth = !authorizedDefinition.authentication() && !authorizedDefinition.authorization();
            if (skipAuth) {
                return point.proceed();
            }

            String[] resourceCodes = authorizedDefinition.resources();

            if (ArrayUtils.isEmpty(resourceCodes)) {
                throw new RuntimeException("resource code must be specified!");
            }

            IAuthorizedUser currentUser = UserInfoHolder.get();

            if (null == currentUser) {
                throw new RuntimeException("login required, can not retrieve user info!");
            }

            Set<IRole> roles = currentUser.getRoles();

            Set<Long> roleIds = roles.stream().map(IRole::getId).collect(Collectors.toSet());

            AuthorizedUser user = new AuthorizedUser(currentUser.getId(), currentUser.getUsername(), roleIds);
            user.setAppId(appId);
            user.setTenantId(currentUser.getTenantId());
            user.setTaxNums(currentUser.getTaxNums());
            user.setCompanyIds(currentUser.getCompanies());
            Authorization authorization = new Authorization(user);

            // 开启了入参前置校验，走object规则校验
            if (authorizedDefinition.dataPreAuth() && StringUtils.isNotBlank(authorizedDefinition.dataEntityCode())) {
                // 获取请求参数封装类(dto)
                Object[] args = point.getArgs();
                if (args != null && args.length > 0) {
                    //将整个方法的参数转换为一个Json数据库
                    Map<String, Object> map = this.convertMethodParam(point);
                    String content=objectMapper.writeValueAsString(map);

                    ObjectAgreement objectAgreement = ObjectAgreement
                            .builder()
                            .content(content)
                            .authorization(authorization)
                            .resourceCode(resourceCodes[0])
                            .entityCode(authorizedDefinition.dataEntityCode())
                            .build();

                    CheckResult checkResult = objectRuleCheckServiceClient.check(objectAgreement);
                    log.info("objectAgreement  checkResult status:{},message:{}", checkResult.getStatus(), checkResult.getMessage());
                    if (!CheckStatus.PASS.equals(checkResult)) {
                        throw new RuntimeException(checkResult.getMessage());
                    }
                }
            }

            // 开启了数据范围校验, 设置上下文
            if (authorizedDefinition.dataScopeAuth()) {
                DataAuth auth = new DataAuth();
                auth.setRequired(true);
                authorization.setResourceCode(resourceCodes[0]);
                auth.setAuthorization(authorization);
                DataAuthContextHolder.setDataAuth(auth);
            }
            return point.proceed();
        } catch (Throwable e) {
            throw new RuntimeException(e.getMessage());
        } finally {
            DataAuthContextHolder.clean();
        }
    }

    /**
     * 将参数转换为Map对象
     * @param point 切入点
     * @return Map<String, Object> 参数对象
     */
    protected Map<String, Object> convertMethodParam(ProceedingJoinPoint point) {
        //获取Class 名称
        Signature signature = point.getSignature();
        MethodSignature methodSignature = (MethodSignature) signature;
        methodSignature.getParameterNames();
        String[] argNames = methodSignature.getParameterNames();
        Class[] parameterTypes = methodSignature.getParameterTypes();
        Object[] args = point.getArgs();
        Map<String, Object> map = new HashMap<>(argNames.length);
        for (int i = 0; i < argNames.length; i++) {
            Class<?> classType = parameterTypes[i];
            Object arg = args[i];
            if (classType.isInstance(arg)) {
                map.put(argNames[i], arg);
            }
        }
        return map;
    }
}