package com.xforceplus.ultraman.bocp.gen.engine;

import com.baomidou.mybatisplus.core.toolkit.StringPool;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.baomidou.mybatisplus.generator.config.ConstVal;
import com.baomidou.mybatisplus.generator.config.GlobalConfig;
import com.baomidou.mybatisplus.generator.config.po.TableInfo;
import com.baomidou.mybatisplus.generator.config.rules.FileType;
import com.xforceplus.ultraman.bocp.gen.config.*;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;
import org.apache.velocity.app.VelocityEngine;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.text.SimpleDateFormat;
import java.util.*;

public class ModuleVelocityTemplateEngine {
    protected static final Logger logger = LoggerFactory.getLogger(ModuleVelocityTemplateEngine.class);
    private static final String DOT_VM = ".vm";
    private VelocityEngine velocityEngine;

    /**
     * 配置信息
     */
    private BocpConfigBuilder bocpConfigBuilder;

    /**
     * 模板引擎初始化
     */
    public ModuleVelocityTemplateEngine init(BocpConfigBuilder bocpConfigBuilder) {
        this.bocpConfigBuilder = bocpConfigBuilder;
        if (null == velocityEngine) {
            Properties p = new Properties();
            p.setProperty(ConstVal.VM_LOAD_PATH_KEY, ConstVal.VM_LOAD_PATH_VALUE);
            p.setProperty(Velocity.FILE_RESOURCE_LOADER_PATH, StringPool.EMPTY);
            p.setProperty(Velocity.ENCODING_DEFAULT, ConstVal.UTF8);
            p.setProperty(Velocity.INPUT_ENCODING, ConstVal.UTF8);
            p.setProperty("file.resource.loader.unicode", StringPool.TRUE);
            velocityEngine = new VelocityEngine(p);
        }
        return this;
    }

    public void writer(Map<String, Object> objectMap, String templatePath, String outputFile) throws Exception {
        if (StringUtils.isEmpty(templatePath)) {
            return;
        }

        Template template = velocityEngine.getTemplate(templatePath, ConstVal.UTF8);
        try (FileOutputStream fos = new FileOutputStream(outputFile);
             OutputStreamWriter ow = new OutputStreamWriter(fos, ConstVal.UTF8);
             BufferedWriter writer = new BufferedWriter(ow)) {
            template.merge(new VelocityContext(objectMap), writer);
        }
        System.out.println("模板:" + templatePath + ";  文件:" + outputFile);
        logger.debug("模板:" + templatePath + ";  文件:" + outputFile);
    }

    public String templateFilePath(String filePath) {
        if (null == filePath || filePath.contains(DOT_VM)) {
            return filePath;
        }
        return filePath + DOT_VM;
    }


    /**
     * 输出 java xml 文件
     */
    public ModuleVelocityTemplateEngine batchOutput() {
        try {
            ModuleConfig moduleConfig = getBocpConfigBuilder().getModuleConfig();
            ApplicationConfig applicationConfig = getBocpConfigBuilder().getApplicationConfig();
            Map<String, String> pathInfo = getBocpConfigBuilder().getPathInfo();
            BocpTemplateConfig template = getBocpConfigBuilder().getBocpTemplate();
            if (null != moduleConfig.getName() && null != applicationConfig.getName()) {
                Map<String, Object> objectMap = getObjectMap();
                String parentPomFile = String.format(pathInfo.get(BocpConstVal.MODULE_PARENT_POM_PATH) + File.separator + "%s" + ConstVal.XML_SUFFIX, "pom");
                writer(objectMap, templateFilePath(template.getModuleParentPom()), parentPomFile);

                String gitigoreFile = String.format(pathInfo.get(BocpConstVal.MODULE_GITIGNORE_PATH) + File.separator + "%s", ".gitigore");
                writer(objectMap, templateFilePath(template.getModuleGitignore()), gitigoreFile);

                String gitlabCiYmlFile = String.format(pathInfo.get(BocpConstVal.MODULE_GITLAB_PATH) + File.separator + "%s" + BocpConstVal.YML_SUFFIX, ".gitlab-ci");
                writer(objectMap, templateFilePath(template.getModuleGitlab()), gitlabCiYmlFile);

                String applicationYmlFile =  String.format(pathInfo.get(BocpConstVal.MODULE_APPLICATION_YML_PATH) + File.separator + "%s" + BocpConstVal.YML_SUFFIX, "application");
                writer(objectMap, templateFilePath(template.getModuleApplicationYml()), applicationYmlFile);

                String clientPomFile = String.format(pathInfo.get(BocpConstVal.MODULE_CLIENT_POM_PATH) + File.separator + "%s" + ConstVal.XML_SUFFIX, "pom");
                writer(objectMap, templateFilePath(template.getModuleClientPom()), clientPomFile);

                String pomFile = String.format(pathInfo.get(BocpConstVal.MODULE_POM_PATH) + File.separator + "%s" + ConstVal.XML_SUFFIX, "pom");
                writer(objectMap, templateFilePath(template.getModulePom()), pomFile);

                String servicePomFile = String.format(pathInfo.get(BocpConstVal.MODULE_SERVICE_POM_PATH) + File.separator + "%s" + ConstVal.XML_SUFFIX, "pom");
                writer(objectMap, templateFilePath(template.getModuleServicePom()), servicePomFile);

                String domainPomFile = String.format(pathInfo.get(BocpConstVal.MODULE_DOMAIN_POM_PATH) + File.separator + "%s" + ConstVal.XML_SUFFIX, "pom");
                writer(objectMap, templateFilePath(template.getModuleDomainPom()), domainPomFile);

//                String serviceApplicationFile = String.format(pathInfo.get(BocpConstVal.MODULE_SERVICE_APPLICATION_PATH) + File.separator + "%s" + ConstVal.JAVA_SUFFIX, moduleConfig.getAlias() + "ServiceApplication");
//                writer(objectMap, templateFilePath(template.getModuleServiceApplication()), serviceApplicationFile);
            }
        } catch (Exception e) {
            logger.error("无法创建文件，请检查配置信息！", e);
        }
        return this;
    }

    /**
     * 处理输出目录
     */
    public ModuleVelocityTemplateEngine mkdirs() {
        getBocpConfigBuilder().getPathInfo().forEach((key, value) -> {
            File dir = new File(value);
            if (!dir.exists()) {
                boolean result = dir.mkdirs();
                if (result) {
                    logger.debug("创建目录： [" + value + "]");
                }
            }
        });
        return this;
    }


    /**
     * 打开输出目录
     */
    public void open() {
        String outDir = getBocpConfigBuilder().getGlobalConfig().getOutputDir();
        if (getBocpConfigBuilder().getGlobalConfig().isOpen()
                && StringUtils.isNotEmpty(outDir)) {
            try {
                String osName = System.getProperty("os.name");
                if (osName != null) {
                    if (osName.contains("Mac")) {
                        Runtime.getRuntime().exec("open " + outDir);
                    } else if (osName.contains("Windows")) {
                        Runtime.getRuntime().exec("cmd /c start " + outDir);
                    } else {
                        logger.debug("文件输出目录:" + outDir);
                    }
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }


    /**
     * 渲染对象 MAP 信息
     *
     *
     * @return ignore
     */
    public Map<String, Object> getObjectMap() {
        Map<String, Object> objectMap = new HashMap<>(30);
        BocpConfigBuilder config = getBocpConfigBuilder();
        //将参数放入map中用于模板解析
        objectMap.put("restControllerStyle", config. getStrategyConfig().isRestControllerStyle());
        objectMap.put("config", config);
        objectMap.put("application", config.getApplicationConfig());
        objectMap.put("project", config.getProjectConfig());
        objectMap.put("module", config.getModuleConfig());
        objectMap.put("package", config.getPackageInfo());
        objectMap.put("dataSource", config.getDataSourceConfig());
        GlobalConfig globalConfig = config.getGlobalConfig();
        objectMap.put("author", globalConfig.getAuthor());
        objectMap.put("idType", globalConfig.getIdType() == null ? null : globalConfig.getIdType().toString());
        objectMap.put("logicDeleteFieldName", config.getStrategyConfig().getLogicDeleteFieldName());
        objectMap.put("versionFieldName", config.getStrategyConfig().getVersionFieldName());
        objectMap.put("activeRecord", globalConfig.isActiveRecord());
        objectMap.put("kotlin", globalConfig.isKotlin());
        objectMap.put("swagger2", globalConfig.isSwagger2());
        objectMap.put("date", new SimpleDateFormat("yyyy-MM-dd").format(new Date()));
        objectMap.put("enableCache", globalConfig.isEnableCache());
        objectMap.put("baseResultMap", globalConfig.isBaseResultMap());
        objectMap.put("superEntityClass", getSuperClassName(config.getSuperEntityClass()));
        objectMap.put("superMapperClassPackage", config.getSuperMapperClass());
        objectMap.put("superMapperClass", getSuperClassName(config.getSuperMapperClass()));
        objectMap.put("superServiceClassPackage", config.getSuperServiceClass());
        objectMap.put("superServiceClass", getSuperClassName(config.getSuperServiceClass()));
        objectMap.put("superServiceImplClassPackage", config.getSuperServiceImplClass());
        objectMap.put("superServiceImplClass", getSuperClassName(config.getSuperServiceImplClass()));
        objectMap.put("superControllerClassPackage", config.getSuperControllerClass());
        objectMap.put("superControllerClass", getSuperClassName(config.getSuperControllerClass()));
        return Objects.isNull(config.getInjectionConfig()) ? objectMap : config.getInjectionConfig().prepareObjectMap(objectMap);
    }


    /**
     * 获取类名
     *
     * @param classPath ignore
     * @return ignore
     */
    private String getSuperClassName(String classPath) {
        if (StringUtils.isEmpty(classPath)) {
            return null;
        }
        return classPath.substring(classPath.lastIndexOf(StringPool.DOT) + 1);
    }


    /**
     * 检测文件是否存在
     *
     * @return 文件是否存在
     */
    protected boolean isCreate(FileType fileType, String filePath) {
        BocpConfigBuilder cb = getBocpConfigBuilder();
        // 全局判断【默认】
        File file = new File(filePath);
        boolean exist = file.exists();
        if (!exist) {
            file.getParentFile().mkdirs();
        }
        return !exist || getBocpConfigBuilder().getGlobalConfig().isFileOverride();
    }

    /**
     * 文件后缀
     */
    protected String suffixJavaOrKt() {
        return getBocpConfigBuilder().getGlobalConfig().isKotlin() ? ConstVal.KT_SUFFIX : ConstVal.JAVA_SUFFIX;
    }


    public BocpConfigBuilder getBocpConfigBuilder() {
        return bocpConfigBuilder;
    }

    public void setBocpConfigBuilder(BocpConfigBuilder bocpConfigBuilder) {
        this.bocpConfigBuilder = bocpConfigBuilder;
    }
}
