/*
 *    Copyright (c) 2020-2027, wangzheng All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the pig4cloud.com developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: wangzheng (wangzheng@xforceplus.com)
 */

package com.xforceplus.ultramanbocp.service.impl;

import com.xforceplus.ultramanbocp.entity.Invoice;
import com.xforceplus.ultramanbocp.mapper.InvoiceMapper;
import com.xforceplus.ultramanbocp.service.IInvoiceService;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.xforceplus.ultraman.bocp.gen.annotation.BoService;
import org.springframework.stereotype.Service;
import java.util.List;
import java.util.ArrayList;
import java.util.Map;
import java.util.HashMap;
import java.util.Optional;
import java.io.Serializable;
import java.util.stream.Collectors;
import java.util.function.Function;

import com.xforceplus.ultraman.bocp.gen.sql.pojo.ConditionQueryRequest;

import com.baomidou.mybatisplus.extension.service.IService;
import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;

import com.xforceplus.ultraman.bocp.gen.combo.ComboConfig;
import com.xforceplus.ultraman.bocp.gen.combo.ComboOptions;
import org.springframework.beans.factory.annotation.Autowired;
import com.xforceplus.ultramanbocp.entity.ComboInvoice;
import com.xforceplus.ultramanbocp.entity.InvoiceDetail;
import com.xforceplus.ultramanbocp.service.IInvoiceDetailService;
import com.xforceplus.ultramanbocp.mapper.InvoiceDetailMapper;
import com.xforceplus.ultramanbocp.entity.Company;
import com.xforceplus.ultramanbocp.service.ICompanyService;
import com.xforceplus.ultramanbocp.mapper.CompanyMapper;

/**
 * <p>
 *  服务实现类
 * </p>
 *
 * @author ultraman
 * @since 2020-12-02
 */
@BoService(value = "1226413425763627010")
@Service
public class InvoiceServiceImpl extends ServiceImpl<InvoiceMapper, Invoice> implements IInvoiceService {


    @Autowired
    IInvoiceDetailService invoiceDetailServiceImpl;
    @Autowired
    InvoiceDetailMapper invoiceDetailMapper;
    @Autowired
    ICompanyService companyServiceImpl;
    @Autowired
    CompanyMapper companyMapper;

    private ComboConfig comboConfig = new ComboConfig();

    @Override
    public IInvoiceService ofOptions(ComboOptions comboOptions) {
        this.comboConfig.changeOptions(comboOptions);
        return this;
    }

    @Override
    public IInvoiceService setLevelLimit(Integer levelLimit) {
        this.comboConfig.changeOptions(new ComboOptions(levelLimit));
        return this;
    }

    private void resetConfig() {
        this.comboConfig.reset();
    }

    @Override
    public List<Map> querys(Map<String, Object> params) {
        ConditionQueryRequest req = (ConditionQueryRequest) params.get("request");
        List<Map> result = this.baseMapper.querys(params);
        result.stream().forEach(rs ->
            req.getEntity().getEntities().stream()
            .filter(entity -> rs.containsKey(entity.getCode()))
            .forEach(entity -> {
                Map entityRs = (HashMap) rs.get(entity.getCode());
                entityRs.keySet().stream().forEach(key ->
                    rs.put(entity.getCode() + "." + key, entityRs.get(key))
                );
                rs.remove(entity.getCode());
            })
        );
        return result;
    }

    @Override
    public IPage<ComboInvoice> comboPage(IPage<Invoice> page, Wrapper<Invoice> queryWrapper) {
        List<Invoice> invoices = baseMapper.selectPage(page, queryWrapper).getRecords();

        List<Long> invoiceIds = invoices.stream().map(Invoice::getId).collect(Collectors.toList());
        List<String> invoiceCompanyCodes = invoices.stream().map(Invoice::getCompanyCode).collect(Collectors.toList());


        Map<Long, List<InvoiceDetail>> invoiceDetailMap = new HashMap<>();

        Map<String, Company> companyMap = new HashMap<>();

        if(comboConfig.getLevelLimit() >= 2) {
            invoiceDetailMap = invoiceDetailMapper.selectList(Wrappers.<InvoiceDetail>lambdaQuery().in(InvoiceDetail::getInvoiceId, invoiceIds))
                .stream().collect(Collectors.groupingBy(InvoiceDetail::getInvoiceId));
            companyMap = companyMapper.selectList(Wrappers.<Company>lambdaQuery().in(Company::getCode, invoiceCompanyCodes))
                .stream().collect(Collectors.toMap(Company::getCode, Function.identity()));
        }

        if(comboConfig.getLevelLimit() >= 3) {
        }

        List<ComboInvoice> comboInvoices = new ArrayList<>();
        for(Invoice invoice:invoices) {
            ComboInvoice comboInvoice = new ComboInvoice(invoice);
            if(comboConfig.getLevelLimit() >= 2) {
                Optional.ofNullable(invoiceDetailMap.get(invoice.getId())).ifPresent(invoiceDetails -> comboInvoice.setInvoiceDetails(invoiceDetails));
                Optional.ofNullable(companyMap.get(invoice.getCompanyCode())).ifPresent(company -> comboInvoice.setCompany(company));
            }
            if(comboConfig.getLevelLimit() >= 3) {
            }
            comboInvoices.add(comboInvoice);
        }

        IPage<ComboInvoice> pageResult = new Page<>();
        pageResult.setTotal(page.getTotal());
        pageResult.setRecords(comboInvoices);

        this.resetConfig();

        return pageResult;
    }

    @Override
    public List<ComboInvoice> comboList(Wrapper<Invoice> queryWrapper) {
        List<Invoice> invoices = baseMapper.selectList(queryWrapper);

        List<Long> invoiceIds = invoices.stream().map(Invoice::getId).collect(Collectors.toList());
        List<String> invoiceCompanyCodes = invoices.stream().map(Invoice::getCompanyCode).collect(Collectors.toList());

        Map<Long, List<InvoiceDetail>> invoiceDetailMap = new HashMap<>();
        Map<String, Company> companyMap = new HashMap<>();

        if(comboConfig.getLevelLimit() >= 2) {
            invoiceDetailMap = invoiceDetailMapper.selectList(Wrappers.<InvoiceDetail>lambdaQuery().in(InvoiceDetail::getInvoiceId, invoiceIds))
                .stream().collect(Collectors.groupingBy(InvoiceDetail::getInvoiceId));
            companyMap = companyMapper.selectList(Wrappers.<Company>lambdaQuery().in(Company::getCode, invoiceCompanyCodes))
                .stream().collect(Collectors.toMap(Company::getCode, Function.identity()));
        }

        if(comboConfig.getLevelLimit() >= 3) {
        }

        List<ComboInvoice> comboInvoices = new ArrayList<>();
        for(Invoice invoice:invoices) {
            ComboInvoice comboInvoice = new ComboInvoice(invoice);
            if(comboConfig.getLevelLimit() >= 2) {
                Optional.ofNullable(invoiceDetailMap.get(invoice.getId())).ifPresent(invoiceDetails -> comboInvoice.setInvoiceDetails(invoiceDetails));
                Optional.ofNullable(companyMap.get(invoice.getCompanyCode())).ifPresent(company -> comboInvoice.setCompany(company));
            }
            if(comboConfig.getLevelLimit() >= 3) {
            }
            comboInvoices.add(comboInvoice);
        }
        return comboInvoices;
    }

    @Override
    public ComboInvoice comboGetById(Serializable id) {
        Invoice invoice = baseMapper.selectById(id);
        if(invoice == null) {
            return null;
        }

        ComboInvoice comboInvoice = new ComboInvoice(invoice);
        if(comboConfig.getLevelLimit() >= 2) {
            List<InvoiceDetail> invoiceDetails = invoiceDetailServiceImpl.list(Wrappers.<InvoiceDetail>lambdaQuery().eq(InvoiceDetail::getInvoiceId, invoice.getId()));
            comboInvoice.setInvoiceDetails(invoiceDetails);
            Company company = companyServiceImpl.getByCode(invoice.getCompanyCode());
            comboInvoice.setCompany(company);
        }

        if(comboConfig.getLevelLimit() >= 3) {
       }

        return comboInvoice;
    }

    @Override
    public boolean comboSave(ComboInvoice comboEntity) {
        baseMapper.insert(comboEntity);
        Optional.ofNullable(comboEntity.getInvoiceDetails()).ifPresent(invoiceDetails -> {
            invoiceDetails.stream().forEach(bo -> {
                bo.setInvoiceId(comboEntity.getId());
                invoiceDetailMapper.insert(bo);
            });
        });
        Optional.ofNullable(comboEntity.getCompany()).ifPresent(bo -> {
            bo.setCode(comboEntity.getCompanyCode());
            companyMapper.insert(bo);
        });
        return true;
    }

    @Override
    public boolean comboUpdateById(ComboInvoice comboEntity) {
        baseMapper.updateById(comboEntity);
        Optional.ofNullable(comboEntity.getInvoiceDetails()).ifPresent(invoiceDetails -> {
            invoiceDetails.stream().forEach(invoiceDetail -> {
                invoiceDetailMapper.updateById(invoiceDetail);
            });
        });
        Optional.ofNullable(comboEntity.getCompany()).ifPresent(bo ->{
            companyMapper.updateById(bo);
        });
        return true;
    }

    @Override
    public boolean comboRemoveById(Serializable id) {
        Invoice invoice = baseMapper.selectById(id);
        if(invoice == null) { return false; }

        baseMapper.deleteById(id);

        List<InvoiceDetail> invoiceDetails = invoiceDetailMapper.selectList(Wrappers.<InvoiceDetail>lambdaQuery().eq(InvoiceDetail::getInvoiceId, invoice.getId()));
        invoiceDetails.stream().forEach(bo -> invoiceDetailMapper.deleteById(bo.getId()));
        Company company = companyMapper.selectOne(Wrappers.<Company>lambdaQuery().eq(Company::getCode, invoice.getCompanyCode()));
        Optional.ofNullable(company).ifPresent(bo -> {
            companyMapper.deleteById(bo.getId());
        });
        return true;
    }

}
