package com.xforceplus.ultraman.adapter.core.impl;

import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.sdk.core.calcite.oqs.DataQueryProvider;
import com.xforceplus.ultraman.sdk.core.calcite.oqs.strategy.QueryProviderSelectStrategy;
import com.xforceplus.ultraman.sdk.core.calcite.oqs.strategy.QueryStrategy;
import com.xforceplus.ultraman.sdk.infra.metrics.MetricsDefine;
import com.xforceplus.ultraman.sdk.invocation.invoke.InvocationManager;
import io.micrometer.core.annotation.Timed;
import io.vavr.Tuple2;
import lombok.extern.slf4j.Slf4j;
import org.apache.calcite.DataContext;
import org.apache.calcite.rel.RelFieldCollation;
import org.apache.calcite.rel.RelNode;
import org.apache.calcite.rel.core.AggregateCall;
import org.apache.calcite.rel.hint.RelHint;
import org.apache.calcite.rel.type.RelDataType;
import org.apache.calcite.rel.type.StructKind;
import org.apache.calcite.rex.RexNode;
import org.apache.calcite.util.Pair;

import java.util.*;

@Slf4j
public class AdaptiveDataQueryProvider implements DataQueryProvider {

    private List<DataQueryProvider> dataQueryProviders;

    private List<QueryProviderSelectStrategy> queryProviderStrategies;

    private List<QueryStrategy> queryStrategies;

    public AdaptiveDataQueryProvider(List<DataQueryProvider> dataQueryProviders
            , List<QueryProviderSelectStrategy> queryProviderStrategies
            , List<QueryStrategy> queryStrategies) {
        this.dataQueryProviders = dataQueryProviders;
        this.queryProviderStrategies = queryProviderStrategies;
        this.queryStrategies = queryStrategies;
    }

    @Override
    public QueryProviderType type() {
        return QueryProviderType.PROXY;
    }

    @Timed(
            value = MetricsDefine.PROCESS_DELAY_LATENCY_SECONDS,
            percentiles = {0.5, 0.9, 0.99},
            extraTags = {"query", "adaptive"}
    )
    @Override
    public List<Object> query(String app, IEntityClass entityClass, String profile, RelDataType type, List<RexNode> ops, List<Map.Entry<String, Tuple2<StructKind, Class>>> fields, List<Pair<RexNode, String>> projects, List<Map.Entry<String, RelFieldCollation.Direction>> sort, Long offset, Long fetch, List<String> groupBy, List<AggregateCall> aggs
            , List<RelHint> hints
            , RelNode rawTree
            , DataContext dataContext) {
        if (dataQueryProviders.size() == 1) {
            return doWithStrategy(dataQueryProviders.get(0), app, entityClass, profile, type, ops, fields, projects, sort, offset, fetch, groupBy, aggs, hints, rawTree, dataContext);
        } else {
            if (!hints.isEmpty()) {
                //if a fallback query
                Optional<RelHint> fallBackHint = hints.stream().filter(x -> x.hintName
                        .equalsIgnoreCase(InvocationManager.InvocationType.FALLBACK.name().toLowerCase())).findAny();
                if (fallBackHint.isPresent()) {
                    //fallback
                    // always do with master
                    Optional<DataQueryProvider> masterOp = dataQueryProviders.stream().filter(x -> x.type() == QueryProviderType.MASTER).findAny();
                    if(masterOp.isPresent()) {
                        log.debug("Call with Fallback code:{}, profile:{}", entityClass.code(), profile);
                        return doWithStrategy(masterOp.get(), app, entityClass, profile, type, ops, fields, projects, sort, offset, fetch, groupBy, aggs, hints, rawTree, dataContext);
                    }
                }

                //if an index query
                Optional<RelHint> indexQueryHint = hints.stream().filter(x -> x.hintName
                        .equalsIgnoreCase("index_search")).findAny();

                if (indexQueryHint.isPresent()) {
                    //fallback
                    // always do with master
                    Optional<DataQueryProvider> indexOp = dataQueryProviders.stream().filter(x -> x.type() == QueryProviderType.INDEX).findAny();
                    if(indexOp.isPresent()) {
                        log.debug("Call with Index code:{}, profile:{}", entityClass.code(), profile);
                        return doWithStrategy(indexOp.get(), app, entityClass, profile, type, ops, fields, projects, sort, offset, fetch, groupBy, aggs, hints, rawTree, dataContext);
                    }
                }
            }
            
            Optional<Map<DataQueryProvider, Double>> reduce = queryProviderStrategies.stream().sorted()
                    .map(x -> x.score(app, entityClass, profile, type, ops, fields, projects, sort, offset, fetch, groupBy, aggs, hints, rawTree, dataContext, dataQueryProviders)).reduce((a, b) -> {
                        HashMap<DataQueryProvider, Double> dataQueryProviderDoubleHashMap = new HashMap<>(a);
                        b.forEach((k, v) -> {
                            dataQueryProviderDoubleHashMap.merge(k, v, Double::sum);
                        });
                        return dataQueryProviderDoubleHashMap;
                    });
            if (reduce.isPresent()) {
                Map<DataQueryProvider, Double> dataQueryProviderDoubleMap = reduce.get();
                Optional<Map.Entry<DataQueryProvider, Double>> max = dataQueryProviderDoubleMap.entrySet().stream().max(Comparator.comparingDouble(Map.Entry::getValue));
                if (max.isPresent()) {
                    return doWithStrategy(max.get().getKey(), app, entityClass, profile, type, ops, fields, projects, sort, offset, fetch, groupBy, aggs, hints, rawTree, dataContext);
                }
            }
        }

        throw new RuntimeException("No Suitable Query Provider");
    }

    public List<Object> doWithStrategy(DataQueryProvider dataQueryProvider, String app, IEntityClass entityClass, String profile, RelDataType type, List<RexNode> ops, List<Map.Entry<String, Tuple2<StructKind, Class>>> fields, List<Pair<RexNode, String>> projects, List<Map.Entry<String, RelFieldCollation.Direction>> sort, Long offset, Long fetch
            , List<String> groupBy, List<AggregateCall> aggs
            , List<RelHint> hints, RelNode rawTree, DataContext dataContext) {

        queryStrategies.stream().sorted().filter(x -> x.accept(dataQueryProvider.type()))
                .forEach(x -> x.doWithInput(dataQueryProvider, app, entityClass, profile, type, ops, fields, projects, sort, offset, fetch, groupBy, aggs, hints, rawTree, dataContext));
        return dataQueryProvider.query(app, entityClass, profile, type, ops, fields, projects, sort, offset, fetch, groupBy, aggs, hints, rawTree, dataContext);
    }
}
