package com.xforceplus.ultraman.adapter.core.impl;

import com.google.common.annotations.VisibleForTesting;
import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.ultraman.billing.client.aspect.BillingScope;
import com.xforceplus.ultraman.core.EntityWriteService;
import com.xforceplus.ultraman.core.pojo.OqsEngineResult;
import com.xforceplus.ultraman.metadata.domain.record.EmptyValue;
import com.xforceplus.ultraman.metadata.domain.record.GeneralRecord;
import com.xforceplus.ultraman.metadata.domain.record.Record;
import com.xforceplus.ultraman.metadata.domain.vo.DataCollection;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.engine.EntityClassGroup;
import com.xforceplus.ultraman.metadata.entity.*;
import com.xforceplus.ultraman.metadata.entity.calculation.AbstractCalculation;
import com.xforceplus.ultraman.metadata.entity.calculation.Lookup;
import com.xforceplus.ultraman.metadata.entity.impl.Entity;
import com.xforceplus.ultraman.metadata.entity.legacy.impl.ColumnField;
import com.xforceplus.ultraman.metadata.entity.legacy.impl.EntityField;
import com.xforceplus.ultraman.oqsengine.plus.common.StringUtils;
import com.xforceplus.ultraman.oqsengine.plus.master.mysql.MysqlSqlDialectEx;
import com.xforceplus.ultraman.sdk.core.auth.AuthBuilder;
import com.xforceplus.ultraman.sdk.core.config.ExecutionConfig;
import com.xforceplus.ultraman.sdk.core.facade.EntityFacade;
import com.xforceplus.ultraman.sdk.core.facade.ProfileFetcher;
import com.xforceplus.ultraman.sdk.core.facade.result.*;
import com.xforceplus.ultraman.sdk.core.pipeline.OperationType;
import com.xforceplus.ultraman.sdk.core.pipeline.TransformerPipeline;
import com.xforceplus.ultraman.sdk.core.pojo.UpdateConfig;
import com.xforceplus.ultraman.sdk.core.rel.legacy.*;
import com.xforceplus.ultraman.sdk.core.rel.utils.ExpTreeToRel;
import com.xforceplus.ultraman.sdk.infra.base.id.IdGenerator;
import com.xforceplus.ultraman.sdk.infra.exceptions.RecordMissingException;
import com.xforceplus.ultraman.sdk.infra.logging.LoggingPattern;
import com.xforceplus.ultraman.sdk.infra.metrics.MetricsDefine;
import com.xforceplus.ultraman.sdk.infra.query.LazyFetchIterator;
import com.xforceplus.ultraman.sdk.infra.utils.CompletableFutureUtils;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import io.vavr.Tuple3;
import io.vavr.control.Either;
import lombok.extern.slf4j.Slf4j;
import org.apache.calcite.rel.RelNode;
import org.apache.calcite.rel.rel2sql.RelToSqlConverter;
import org.apache.calcite.sql.SqlNode;
import org.apache.calcite.tools.FrameworkConfig;
import org.apache.calcite.tools.RelRunner;
import org.apache.calcite.util.Util;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.transaction.annotation.Transactional;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.xforceplus.ultraman.adapter.utils.CommonHelper.getValuesFromMap;
import static com.xforceplus.ultraman.sdk.infra.logging.LoggingUtils.logErrorPattern;

/**
 * local entity facade
 */
@BillingScope("oqssdk")
@Slf4j
public class LocalEntityFacadeImpl implements EntityFacade {

    private final static int QUERY_STEP = 1000;

    @Qualifier("CalciteDS")
    @Lazy
    @Autowired
    private DataSource dataSource;

    @Lazy
    @Autowired
    private FrameworkConfig config;
    @Autowired
    private ProfileFetcher fetcher;
    @Autowired
    private EntityClassEngine engine;
    @Autowired
    private ContextService contextService;
    @Autowired
    private EntityWriteService writeService;
    @Autowired
    private IdGenerator<Long> idGenerator;
    @Autowired
    private TransformerPipeline transformerPipeline;

    private final Counter queryCountTotal = Metrics.counter(MetricsDefine.OPERATION_COUNT_TOTAL, "action", "query");

    private final Counter insertCountTotal = Metrics.counter(MetricsDefine.OPERATION_COUNT_TOTAL, "action", "build");

    private final Counter insertMultiCountTotal = Metrics.counter(MetricsDefine.OPERATION_COUNT_TOTAL, "action", "buildMulti");

    private final Counter replaceOneCountTotal = Metrics.counter(MetricsDefine.OPERATION_COUNT_TOTAL, "action", "replaceOne");

    private final Counter replaceMultiCountTotal = Metrics.counter(MetricsDefine.OPERATION_COUNT_TOTAL, "action", "replaceMulti");

    private final Counter deleteOneCountTotal = Metrics.counter(MetricsDefine.OPERATION_COUNT_TOTAL, "action", "deleteOne");

    private final Counter deleteMultiCountTotal = Metrics.counter(MetricsDefine.OPERATION_COUNT_TOTAL, "action", "deleteMulti");

    @Autowired
    private ExecutionConfig executeConfig;
    @Autowired(required = false)
    private AuthBuilder authBuilder;

    private ForkJoinPool forkJoinPool = new ForkJoinPool(50);

    public LocalEntityFacadeImpl() {
    }

    public void setDataSource(DataSource dataSource) {
        this.dataSource = dataSource;
    }

    public void setConfig(FrameworkConfig config) {
        this.config = config;
    }

    public void setEngine(EntityClassEngine engine) {
        this.engine = engine;
    }

    public void setContextService(ContextService contextService) {
        this.contextService = contextService;
    }

    public void setWriteService(EntityWriteService writeService) {
        this.writeService = writeService;
    }

    public void setIdGenerator(IdGenerator<Long> idGenerator) {
        this.idGenerator = idGenerator;
    }

    public void setTransformerPipeline(TransformerPipeline transformerPipeline) {
        this.transformerPipeline = transformerPipeline;
    }

    public void setExecuteConfig(ExecutionConfig executeConfig) {
        this.executeConfig = executeConfig;
    }

    public void setAuthBuilder(AuthBuilder authBuilder) {
        this.authBuilder = authBuilder;
    }

    @Override
    public Optional<IEntityClass> load(String boId, String profile) {
        return engine.load(boId, profile);
    }

    @Override
    public Optional<IEntityClass> load(String boId, String profile, String version) {
        return engine.load(boId, profile, version);
    }

    @Override
    public Optional<IEntityClass> loadByCode(String bocode, String profile) {
        return engine.loadByCode(bocode, profile);
    }

    @Override
    public Optional<IEntityClass> loadByCode(String bocode, String profile, String version) {
        return engine.loadByCode(bocode, profile, version);
    }

    @VisibleForTesting
    private IEntity toIEntity(IEntityClass entityClass, long id) {
        return Entity.Builder.anEntity().withEntityClassRef(entityClass.ref()).withId(id).build();
    }

    @VisibleForTesting
    private IEntity toIEntity(EntityClassGroup entityClass, Map<String, Object> body) {
        long id = Optional.ofNullable(body.get("id")).map(Object::toString).map(Long::parseLong).orElse(0L);
        long operationTime = System.currentTimeMillis();
        Object o = Optional.ofNullable(body.get("update_time")).orElse(body.get("create_time"));
        if (o != null) {
            try {
                operationTime = Long.parseLong(o.toString());
            } catch (Throwable throwable) {
                log.error("", throwable);
            }
        }
        return Entity.Builder.anEntity().withEntityClassRef(entityClass.getEntityClass().ref()).withId(id).withValues(getValuesFromMap(entityClass, body)).withTime(operationTime).build();
    }


    public Map<IEntityClass, Map<String, Object>> split(EntityClassGroup group, Map<String, Object> body, Map<String, Object> context, OperationType operationType) {

        Map<IEntityClass, Map<String, Object>> mapping = new HashMap<>();

        body.forEach((k, v) -> {
            Optional<ColumnField> column = group.column(k);
            if (column.isPresent()) {
                AbstractCalculation calculation = column.get().getOriginObject().config().getCalculation();
                if (calculation instanceof Lookup) {
                    if (operationType == OperationType.UPDATE || operationType == OperationType.REPLACE) {
                        long classId = ((Lookup) calculation).getClassId();
                        long fieldId = ((Lookup) calculation).getFieldId();
                        long refId = ((Lookup) calculation).getRelationId();
                        Optional<IEntityClass> relatedEntityClass = group.classEngine().load(Long.toString(classId), group.realProfile());
                        if (relatedEntityClass.isPresent()) {
                            IEntityClass entityClass = relatedEntityClass.get();
                            Optional<IEntityField> field = entityClass.field(fieldId);
                            Optional<IRelation> relation = group.relation(refId);
                            if(relation.isPresent()) {
                                IRelation iRelation = relation.get();
                                if(iRelation.getRelationType().equalsIgnoreCase("TO_ONE")) {
                                    String name = iRelation.getName();
                                    Object refIdValue = body.get(name.concat(".id"));
                                    if(refIdValue != null) {
                                        if (field.isPresent()) {
                                            mapping.compute(entityClass, (a, b) -> {
                                                if (b == null) {
                                                    b = new HashMap<>();
                                                }
                                                b.put(field.get().name(), v);
                                                return b;
                                            });
                                        }

                                        //add id
                                        mapping.compute(entityClass, (a, b) -> {
                                            if (b.get("id") == null) {
                                                b.put("id", refIdValue);
                                            }
                                            
                                            return b;
                                        });
                                    }
                                }
                            }
                        }
                    }
                } else {
                    mapping.compute(group.getEntityClass(), (a, b) -> {
                        if (b == null) {
                            b = new HashMap<>();
                        }
                        b.put(k, v);
                        return b;
                    });
                }
            }
        });

        return mapping;
    }

    private CompletionStage<Either<CreateOneResult, Long>> createInner(EntityClassGroup group, Map<String, Object> body, Map<String, Object> context) {
        try {
            List<Tuple2<IEntityField, Object>> pipelinedBody = transformerPipeline.valueSideHandleValue(group, body, OperationType.CREATE);
            Map<String, Object> newBody = new HashMap<>();
            pipelinedBody.forEach(x -> {
                newBody.put(x._1.name(), x._2);
            });

            IEntity entity = toIEntity(group, newBody);
            if (entity.id() == 0) {
                Long next = idGenerator.next();
                entity.resetId(next);
                body.put("id", next);
            }

            OqsEngineResult<Long> build = writeService.build(entity, context);
            if (build.isSuccess()) {
                insertCountTotal.increment();
                Long retLong = build.getValue().get();
                return CompletableFuture.completedFuture(Either.right(retLong));
            } else {
                return CompletableFuture.completedFuture(Either.left(CreateOneResult.failFrom(build.getResultStatus(), build.getMessage())));
            }
        } catch (Throwable ex) {
            return exceptional(ex);
        }
    }

    @Transactional
    @Override
    public CompletionStage<Either<CreateOneResult, Long>> create(IEntityClass entityClass, Map<String, Object> body, Map<String, Object> context) {

        boolean setProfile = false;
        String profile = fetcher.getProfile(context);
        if(StringUtils.isEmpty(profile)) {
            Object o = body.get("tenant_code");
            if(o != null) {
                profile = o.toString();
                setProfile = true;
                context.put("profile", profile);
                contextService.getAll().put("tenant_code", profile);
            }
        }

        EntityClassGroup group = engine.describe(entityClass, profile);
        if (entityClass.getType() == 2) {
            //is view   
            Map<IEntityClass, Map<String, Object>> split = split(group, body, context, OperationType.CREATE);
            if(split.isEmpty()) {
                return createInner(group, body, context);
            } else {
                List<CompletableFuture<Either<CreateOneResult, Long>>> collect = split.entrySet().stream().map(entry -> {
                    IEntityClass key = entry.getKey();
                    EntityClassGroup targetGroup = getEntityClassEngine().describe(key, group.realProfile());
                    Map<String, Object> value = entry.getValue();
                    return createInner(targetGroup, value, context);
                }).map(x -> (CompletableFuture<Either<CreateOneResult, Long>>) x).collect(Collectors.toList());
                if(setProfile) {
                    context.remove("profile");
                    contextService.getAll().remove("tenant_code");
                }
                //TODO
                return CompletableFutureUtils.sequence(collect).thenApply(x -> x.get(0));
            }
        } else {
            CompletionStage<Either<CreateOneResult, Long>> inner = createInner(group, body, context);
            if(setProfile) {
                context.remove("profile");
                contextService.getAll().remove("tenant_code");
            }
            return inner;
        }
    }

    private CompletionStage<Either<CreateMultiResult, Integer>> createMultiInner(IEntityClass entityClass, List<Map<String, Object>> body, String profile, boolean isSetProfile, Map<String, Object> context) {
        EntityClassGroup group = engine.describe(entityClass, profile);
        IEntity[] entities =
                body.stream()
                        .map(x -> {
                            List<Tuple2<IEntityField, Object>> pipelinedBody = transformerPipeline.valueSideHandleValue(group, x, OperationType.CREATE);
                            Map<String, Object> newBody = new HashMap<>();
                            pipelinedBody.forEach(piped -> {
                                newBody.put(piped._1.name(), piped._2);
                            });

                            IEntity iEntity = toIEntity(group, newBody);
                            return iEntity;
                        }).peek(x -> {
                            if (x.id() == 0) {
                                x.resetId(idGenerator.next());
                            }
                        }).toArray(IEntity[]::new);

        context.put("profile", profile);
        contextService.getAll().put("tenant_code", profile);
        OqsEngineResult<Long> build = writeService.build(entities, context);
        if(isSetProfile) {
            context.remove("profile");
            contextService.getAll().remove("tenant_code");
        }
        if (build.isSuccess()) {
            Long retLong = build.getValue().orElse(0L);
            insertMultiCountTotal.increment(entities.length);
            return CompletableFuture.completedFuture(Either.right(retLong.intValue()));
        } else {
            return CompletableFuture.completedFuture(Either.left(CreateMultiResult.from(new RuntimeException(build.getMessage()))));
        }
    }
    
    @Override
    public CompletionStage<Either<CreateMultiResult, Integer>> createMulti(IEntityClass entityClass, List<Map<String, Object>> body, Map<String, Object> context) {
        try {
            String profile = fetcher.getProfile(context);
            if(!StringUtils.isEmpty(profile)) {
                return createMultiInner(entityClass, body, profile,false, context);
            } else {
                Map<String, List<Map<String, Object>>> tenantCodeMapping = 
                        body.stream().collect(Collectors.groupingBy(x -> Optional.ofNullable(x.get("tenant_code")).map(Object::toString).orElse("")));

                List<CompletableFuture<Either<CreateMultiResult, Integer>>> futures = tenantCodeMapping.entrySet().stream().map(entry -> {
                    return createMultiInner(entityClass, entry.getValue(), entry.getKey(), true, context);
                }).map(x -> (CompletableFuture<Either<CreateMultiResult, Integer>>) x).collect(Collectors.toList());

                return CompletableFutureUtils.sequence(futures).thenApply(x -> x.get(0));

            }
        } catch (Throwable ex) {
            return exceptional(ex);
        }
    }

    private <T> CompletionStage<T> exceptional(Throwable ex) {
        CompletableFuture result = new CompletableFuture<>();
        result.completeExceptionally(ex);
        return result;
    }

//    @Override
//    public CompletionStage<Either<CreateMultiResult, Integer>> createMulti(IEntityClass entityClass, List<Map<String, Object>> body, Map<String, Object> context) {
//        return this.createMulti(entityClass, body.stream(), context);
//    }

    @Override
    public CompletionStage<Either<DeleteOneResult, Integer>> deleteOne(IEntityClass entityClass, Long id, Map<String, Object> context) {
        try {
            String profile = fetcher.getProfile(context);
            EntityClassGroup group = engine.describe(entityClass, profile);
            OqsEngineResult<Long> build = null;
            if (authBuilder != null && executeConfig.getUsePermission()) {
                ExpRel idRelatedQuery = new ExpQuery().filters(ExpCondition.call(ExpOperator.EQUALS, ExpField.ID, ExpValue.from(id))).range(1, 1);
                ExpContext expContext = new ExpContext();
                expContext.withContext(context);
                expContext.setSchema(group);
                /**
                 * TODO to many to one condition
                 */
                Map<Long, Set<String>> mapping = collectEntityClassIdMapping(idRelatedQuery, group);
                ExpRel permissionExpRel = authBuilder.getPermissionTreeCondition(mapping, expContext, profile, OperationType.DELETE);
                if (permissionExpRel != null) {
                    idRelatedQuery = idRelatedQuery.mergeAnd(permissionExpRel);
                }

                //IEntityClass entityClass, ExpRel expRel, Map<String, Object> context
                build = writeService.deleteByCondition(entityClass, idRelatedQuery, context);
            } else {
                build = writeService.delete(toIEntity(entityClass, id), context);
            }

            if (build.isSuccess()) {
                //Long retLong = build.getValue().get();
                deleteOneCountTotal.increment();
                return CompletableFuture.completedFuture(Either.right(1));
            } else {
                return CompletableFuture.completedFuture(Either.left(DeleteOneResult.from(new RuntimeException(build.getMessage()))));
            }
        } catch (Throwable ex) {
            return exceptional(ex);
        }
    }

    @Override
    public CompletionStage<Either<DeleteMultiResult, Integer>> deleteMulti(IEntityClass entityClass, List<Long> ids, Map<String, Object> context) {
        try {
            IEntity[] entities = ids.stream().map(x -> toIEntity(entityClass, x)).toArray(IEntity[]::new);
            String profile = entityClass.realProfile();
            EntityClassGroup group = engine.describe(entityClass, profile);
            OqsEngineResult<Long> build = null;
            if (authBuilder != null && executeConfig.getUsePermission()) {
                ExpRel idRelatedQuery = new ExpQuery().filters(ExpCondition.call(ExpOperator.IN, ExpField.ID, ExpValue.from(ids))).range(1, 10000);
                ExpContext expContext = new ExpContext();
                expContext.withContext(context);
                expContext.setSchema(group);
                /**
                 * TODO to many to one condition
                 */
                Map<Long, Set<String>> mapping = collectEntityClassIdMapping(idRelatedQuery, group);
                ExpRel permissionExpRel = authBuilder.getPermissionTreeCondition(mapping, expContext, profile, OperationType.DELETE);
                if (permissionExpRel != null) {
                    idRelatedQuery = idRelatedQuery.mergeAnd(permissionExpRel);
                }

                //IEntityClass entityClass, ExpRel expRel, Map<String, Object> context
                build = writeService.deleteByCondition(entityClass, idRelatedQuery, context);
            } else {
                build = writeService.delete(entities, context);
            }

            if (build.isSuccess()) {
                Long retLong = build.getValue().get();
                deleteMultiCountTotal.increment(retLong);
                return CompletableFuture.completedFuture(Either.right(retLong.intValue()));
            } else {
                return CompletableFuture.completedFuture(Either.left(DeleteMultiResult.from(new RuntimeException(build.getMessage()))));
            }
        } catch (Throwable ex) {
            return exceptional(ex);
        }
    }

    public static Map<IEntityClass, List<Map<String, Object>>> convertListMapToMapList(List<Map<IEntityClass, Map<String, Object>>> listMap) {
        return listMap.stream()
                .flatMap(map -> map.entrySet().stream())
                .collect(
                        HashMap::new,
                        (resultMap, entry) -> resultMap
                                .computeIfAbsent(entry.getKey(), key -> new ArrayList<>())
                                .add(entry.getValue()),
                        HashMap::putAll
                );
    }
    
    @Transactional
    @Override
    public CompletionStage<Either<UpdateMultiResult, Integer>> updateMulti(IEntityClass entityClass, List<Map<String, Object>> bodies, Map<String, Object> context) {
        String profile = entityClass.realProfile();
        EntityClassGroup group = engine.describe(entityClass, profile);

        List<Map<IEntityClass, Map<String, Object>>> split = bodies.stream().map(body -> {
            return split(group, body, context, OperationType.UPDATE);
        }).collect(Collectors.toList());

        Map<IEntityClass, List<Map<String, Object>>> groupedMapping = convertListMapToMapList(split);

        if(groupedMapping.isEmpty()) {
            //EntityClassGroup group, List<Map<String, Object>> inputBodies, UpdateConfig updateConfig, Map<String, Object> context, OperationType operationType
            return updateMultiInner(group, bodies, null, context, OperationType.UPDATE);
        } else {
            List<CompletableFuture<Either<UpdateMultiResult, Integer>>> collect = groupedMapping.entrySet().stream().map(entry -> {
                IEntityClass key = entry.getKey();
                EntityClassGroup targetGroup = getEntityClassEngine().describe(key, group.realProfile());
                List<Map<String, Object>> bodyOne = entry.getValue();
                return  updateMultiInner(targetGroup, bodyOne, null, context, OperationType.UPDATE);
            }).map(x -> (CompletableFuture<Either<UpdateMultiResult, Integer>>) x).collect(Collectors.toList());
            //TODO
            return CompletableFutureUtils.sequence(collect).thenApply(x -> x.get(0));
        }
    }

    @Override
    public CompletionStage<Either<UpdateMultiResult, Integer>> replaceMulti(IEntityClass entityClass, List<Map<String, Object>> bodies, Map<String, Object> context) {
        String profile = entityClass.realProfile();
        EntityClassGroup group = engine.describe(entityClass, profile);

        List<Map<IEntityClass, Map<String, Object>>> split = bodies.stream().map(body -> {
            return split(group, body, context, OperationType.REPLACE);
        }).collect(Collectors.toList());

        Map<IEntityClass, List<Map<String, Object>>> groupedMapping = convertListMapToMapList(split);

        if(groupedMapping.isEmpty()) {
            //EntityClassGroup group, List<Map<String, Object>> inputBodies, UpdateConfig updateConfig, Map<String, Object> context, OperationType operationType
            return updateMultiInner(group, bodies, null, context, OperationType.REPLACE);
        } else {
            List<CompletableFuture<Either<UpdateMultiResult, Integer>>> collect = groupedMapping.entrySet().stream().map(entry -> {
                IEntityClass key = entry.getKey();
                EntityClassGroup targetGroup = getEntityClassEngine().describe(key, group.realProfile());
                List<Map<String, Object>> bodyOne = entry.getValue();
                return  updateMultiInner(targetGroup, bodyOne, null, context, OperationType.REPLACE);
            }).map(x -> (CompletableFuture<Either<UpdateMultiResult, Integer>>) x).collect(Collectors.toList());
            //TODO
            return CompletableFutureUtils.sequence(collect).thenApply(x -> x.get(0));
        }
    }

    @Override
    public CompletionStage<Either<UpdateMultiResult, Integer>> updateMultiWithConfig(IEntityClass entityClass, List<Map<String, Object>> bodies, UpdateConfig updateConfig, Map<String, Object> context) {

        String profile = entityClass.realProfile();
        EntityClassGroup group = engine.describe(entityClass, profile);

        List<Map<IEntityClass, Map<String, Object>>> split = bodies.stream().map(body -> {
            return split(group, body, context, OperationType.UPDATE);
        }).collect(Collectors.toList());

        Map<IEntityClass, List<Map<String, Object>>> groupedMapping = convertListMapToMapList(split);

        if(groupedMapping.isEmpty()) {
            //EntityClassGroup group, List<Map<String, Object>> inputBodies, UpdateConfig updateConfig, Map<String, Object> context, OperationType operationType
            return updateMultiInner(group, bodies, updateConfig, context, OperationType.UPDATE);
        } else {
            List<CompletableFuture<Either<UpdateMultiResult, Integer>>> collect = groupedMapping.entrySet().stream().map(entry -> {
                IEntityClass key = entry.getKey();
                EntityClassGroup targetGroup = getEntityClassEngine().describe(key, group.realProfile());
                List<Map<String, Object>> bodyOne = entry.getValue();
                if(targetGroup.getEntityClass().id() == group.getEntityClass().id()) {
                    return updateMultiInner(targetGroup, bodyOne, updateConfig, context, OperationType.UPDATE);
                } else {
                    //remove related
                    return updateMultiInner(targetGroup, bodyOne, null, context, OperationType.UPDATE);
                }
            }).map(x -> (CompletableFuture<Either<UpdateMultiResult, Integer>>) x).collect(Collectors.toList());
            //TODO
            return CompletableFutureUtils.sequence(collect).thenApply(x -> x.get(0));
        }
    }

    @Override
    public CompletionStage<Either<UpdateMultiResult, Integer>> replaceMultiWithConfig(IEntityClass entityClass, List<Map<String, Object>> bodies, UpdateConfig updateConfig, Map<String, Object> context) {

        String profile = entityClass.realProfile();
        EntityClassGroup group = engine.describe(entityClass, profile);

        List<Map<IEntityClass, Map<String, Object>>> split = bodies.stream().map(body -> {
            return split(group, body, context, OperationType.REPLACE);
        }).collect(Collectors.toList());

        Map<IEntityClass, List<Map<String, Object>>> groupedMapping = convertListMapToMapList(split);

        if(groupedMapping.isEmpty()) {
            //EntityClassGroup group, List<Map<String, Object>> inputBodies, UpdateConfig updateConfig, Map<String, Object> context, OperationType operationType
            return updateMultiInner(group, bodies, updateConfig, context, OperationType.REPLACE);
        } else {
            List<CompletableFuture<Either<UpdateMultiResult, Integer>>> collect = groupedMapping.entrySet().stream().map(entry -> {
                IEntityClass key = entry.getKey();
                EntityClassGroup targetGroup = getEntityClassEngine().describe(key, group.realProfile());
                List<Map<String, Object>> bodyOne = entry.getValue();
                if(targetGroup.getEntityClass().id() == group.getEntityClass().id()) {
                    return updateMultiInner(targetGroup, bodyOne, updateConfig, context, OperationType.REPLACE);
                } else {
                    //remove related
                    return updateMultiInner(targetGroup, bodyOne, null, context, OperationType.REPLACE);
                }
            }).map(x -> (CompletableFuture<Either<UpdateMultiResult, Integer>>) x).collect(Collectors.toList());
            //TODO
            return CompletableFutureUtils.sequence(collect).thenApply(x -> x.get(0));
        }
    }

    @Override
    public CompletionStage<Either<UpdateOneResult, Integer>> updateById(IEntityClass entityClass, Long id, Map<String, Object> body, Map<String, Object> context) {
        EntityClassGroup group = engine.describe(entityClass, fetcher.getProfile(context));
        if (entityClass.getType() == 2) {
            return this.updateOneInnerWithView(group, id, body, UpdateConfig.builder().build(), context, OperationType.UPDATE);
        } else {
            return this.updateOneInner(group, id, body, UpdateConfig.builder().build(), context, OperationType.UPDATE);
        }
    }

    @Override
    public CompletionStage<Either<UpdateOneResult, Integer>> updateByIdOptimizeLock(IEntityClass entityClass, Long id
            , Map<String, Object> body, Map<String, Object> context) {
        EntityClassGroup group = engine.describe(entityClass, fetcher.getProfile(context));
        if (entityClass.getType() == 2) {
            return this.updateOneInnerWithView(group, id, body, UpdateConfig.builder().useOptimisticLock(true).build(), context, OperationType.UPDATE);
        } else {
            return this.updateOneInner(group, id, body, UpdateConfig.builder().useOptimisticLock(true).build(), context, OperationType.UPDATE);
        }
      
    }

    @Override
    public CompletionStage<Either<UpdateOneResult, Integer>> updateByIdWithConfig(IEntityClass entityClass, Long id, Map<String, Object> body, UpdateConfig updateConfig, Map<String, Object> context) {
        EntityClassGroup group = engine.describe(entityClass, fetcher.getProfile(context));
        if (entityClass.getType() == 2) {
            return this.updateOneInnerWithView(group, id, body, updateConfig, context, OperationType.UPDATE);
        } else {
            return this.updateOneInner(group, id, body, updateConfig, context, OperationType.UPDATE);
        }
    }

    @Override
    public CompletionStage<Either<UpdateOneResult, Integer>> replaceByIdWithConfig(IEntityClass entityClass, Long id, Map<String, Object> body, UpdateConfig updateConfig, Map<String, Object> context) {
        EntityClassGroup group = engine.describe(entityClass, fetcher.getProfile(context));
        if (entityClass.getType() == 2) {
            return this.updateOneInnerWithView(group, id, body, updateConfig, context, OperationType.REPLACE);
        } else {
            return this.updateOneInner(group, id, body, updateConfig, context, OperationType.REPLACE);
        }
    }

    @Override
    public CompletionStage<Either<UpdateOneResult, Integer>> updateById(IEntityClass entityClass, Long id, Map<String, Object> body, int version, Map<String, Object> context) {
        EntityClassGroup group = engine.describe(entityClass, fetcher.getProfile(context));
        if (entityClass.getType() == 2) {
            //is view
            return this.updateOneInnerWithView(group, id, body, UpdateConfig.builder().version(version).build(), context, OperationType.UPDATE);
        } else {
            return this.updateOneInner(group, id, body, UpdateConfig.builder().version(version).build(), context, OperationType.UPDATE);
        }
    }

    @Override
    public CompletionStage<Either<UpdateOneResult, Integer>> replaceById(IEntityClass entityClass, Long id, Map<String, Object> body, Map<String, Object> context) {
        EntityClassGroup group = engine.describe(entityClass, fetcher.getProfile(context));
        if (entityClass.getType() == 2) {
            return this.updateOneInnerWithView(group, id, body, UpdateConfig.builder().build(), context, OperationType.REPLACE);
        } else {
            return this.updateOneInner(group, id, body, UpdateConfig.builder().build(), context, OperationType.REPLACE);
        }
    }

    @Override
    public CompletionStage<Either<UpdateOneResult, Integer>> replaceByIdOptimizeLock(IEntityClass entityClass, Long id, Map<String, Object> body, Map<String, Object> context) {
        EntityClassGroup group = engine.describe(entityClass, fetcher.getProfile(context));
        if (entityClass.getType() == 2) {
            return this.updateOneInnerWithView(group, id, body, UpdateConfig.builder().useOptimisticLock(true).build(), context, OperationType.REPLACE);
        } else {
            return this.updateOneInner(group, id, body, UpdateConfig.builder().useOptimisticLock(true).build(), context, OperationType.REPLACE);
        }
    }

    @Override
    public CompletionStage<Either<UpdateOneResult, Integer>> replaceById(IEntityClass entityClass, Long id, Map<String, Object> body, int version, Map<String, Object> context) {
        EntityClassGroup group = engine.describe(entityClass, fetcher.getProfile(context));
        if (entityClass.getType() == 2) {
            //is view
            return this.updateOneInnerWithView(group, id, body, UpdateConfig.builder().version(version).build(), context, OperationType.REPLACE);
        } else {
            return this.updateOneInner(group, id, body, UpdateConfig.builder().version(version).build(), context, OperationType.REPLACE);
        }
    }

    private CompletionStage<Either<UpdateOneResult, Integer>> updateOneInnerWithView(EntityClassGroup group, Long id, Map<String, Object> body, UpdateConfig updateConfig, Map<String, Object> context, OperationType operationType) {
        Map<String, Object> newBody = new HashMap<>(body);
        newBody.put("id", id);
        Map<IEntityClass, Map<String, Object>> split = split(group, newBody, context, operationType);
        List<CompletableFuture<Either<UpdateOneResult, Integer>>> collect = split.entrySet().stream().map(entry -> {
            Map<String, Object> value = entry.getValue();
            if(value.get("id") != null && value.get("id") != EmptyValue.emptyValue) {
                IEntityClass key = entry.getKey();
                EntityClassGroup targetGroup = getEntityClassEngine().describe(key, group.realProfile());
                return updateOneInner(targetGroup, Long.parseLong(value.get("id").toString()), value
                        , updateConfig, context, operationType);
            }
            return null;
        }).filter(Objects::nonNull).map(x -> (CompletableFuture<Either<UpdateOneResult, Integer>>) x).collect(Collectors.toList());
        //TODO
        return CompletableFutureUtils.sequence(collect).thenApply(x -> x.get(0));
    }

    public CompletionStage<Either<UpdateOneResult, Integer>> updateOneInner(EntityClassGroup group, Long id, Map<String, Object> body, UpdateConfig updateConfig, Map<String, Object> context, OperationType operationType) {
        try {
            String profile = fetcher.getProfile(context);
            List<Tuple2<IEntityField, Object>> pipelinedBody = transformerPipeline.valueSideHandleValue(group, body, operationType);
            Map<String, Object> newBody = new HashMap<>();
            pipelinedBody.forEach(x -> {
                newBody.put(x._1.name(), x._2);
            });
            IEntity entity = toIEntity(group, newBody);
            entity.resetId(id);
            OqsEngineResult<Long> build = null;
            if (authBuilder != null && executeConfig.getUsePermission()) {
                ExpRel idRelatedQuery = new ExpQuery().filters(ExpCondition.call(ExpOperator.EQUALS, ExpField.ID, ExpValue.from(id))).range(1, 1);
                ExpContext expContext = new ExpContext();
                expContext.withContext(context);
                expContext.setSchema(group);
                /**
                 * TODO to many to one condition
                 */
                Map<Long, Set<String>> mapping = collectEntityClassIdMapping(idRelatedQuery, group);
                ExpRel permissionExpRel = authBuilder.getPermissionTreeCondition(mapping, expContext, profile, OperationType.UPDATE);
                if (permissionExpRel != null) {
                    idRelatedQuery = idRelatedQuery.mergeAnd(permissionExpRel);
                }

                build = writeService.replaceConditional(entity, updateConfig, idRelatedQuery, context);
            } else {
                build = writeService.replace(entity, updateConfig, context);
            }

            if (build.isSuccess()) {
                //Long retLong = build.getValue().get();
                replaceOneCountTotal.increment();
                return CompletableFuture.completedFuture(Either.right(1));
            } else {
                return CompletableFuture.completedFuture(Either.left(UpdateOneResult.failFrom(build.getResultStatus(), build.getMessage())));
            }
        } catch (Throwable ex) {
            return exceptional(ex);
        }
    }

    private CompletionStage<Either<UpdateMultiResult, Integer>> updateMultiInner(EntityClassGroup group, List<Map<String, Object>> inputBodies, UpdateConfig updateConfig, Map<String, Object> context, OperationType operationType) {
        try {
            String profile = fetcher.getProfile(context);
            boolean usingOptimize = Optional.ofNullable(updateConfig).map(UpdateConfig::getUseOptimisticLock).orElse(false);
            List<Long> ids = new ArrayList<>();
            IEntity[] entities = inputBodies.stream().map(x -> {
                List<Tuple2<IEntityField, Object>> pipelinedBody = transformerPipeline.valueSideHandleValue(group, x, operationType);
                Map<String, Object> newBody = new HashMap<>();
                pipelinedBody.forEach(piped -> {
                    newBody.put(piped._1.name(), piped._2);
                });
                IEntity iEntity = toIEntity(group, newBody);
                Optional.ofNullable(x.get("id")).map(Object::toString).map(Long::parseLong).ifPresent(iEntity::resetId);
                //init
                iEntity.resetVersion(-1);
                Optional.ofNullable(x.get("_sys_ver")).map(Object::toString).map(Integer::parseInt).ifPresent(iEntity::resetVersion);

                ids.add(iEntity.id());
                iEntity.resetOptimizeLock(usingOptimize);
                return iEntity;
            }).toArray(IEntity[]::new);

            OqsEngineResult<Long> build = null;
            if (authBuilder != null && executeConfig.getUsePermission()) {
                ExpRel dummy = new ExpQuery().filters(ExpCondition.call(ExpOperator.IN, ExpField.ID, ExpValue.from(ids))).range(1, 10000);
                ExpContext expContext = new ExpContext();
                expContext.withContext(context);
                expContext.setSchema(group);
                /**
                 * TODO to many to one condition
                 */
                Map<Long, Set<String>> mapping = collectEntityClassIdMapping(dummy, group);
                ExpRel permissionExpRel = authBuilder.getPermissionTreeCondition(mapping, expContext, profile, OperationType.UPDATE);
                if (permissionExpRel != null) {
                    Map<String, ExpRel> expRelMapping = new HashMap<>();
                    expRelMapping.put(group.getEntityClass().code(), permissionExpRel);
                    build = writeService.replaceConditional(entities, expRelMapping, context);
                } else {
                    Map<String, ExpRel> expRelMapping = new HashMap<>();
                    expRelMapping.put(group.getEntityClass().code(), dummy);
                    build = writeService.replaceConditional(entities, expRelMapping, context);
                }
            } else {
                build = writeService.replace(entities, context);
            }
            
            if (build.isSuccess()) {
                //Long retLong = build.getValue().get();
                if (build.getValue().isPresent()) {
                    replaceMultiCountTotal.increment(build.getValue().get());
                    return CompletableFuture.completedFuture(Either.right(build.getValue().get().intValue()));
                } else {
                    return CompletableFuture.completedFuture(Either.right(1));
                }
            } else {
                return CompletableFuture.completedFuture(Either.left(UpdateMultiResult.failFrom(build.getResultStatus(), build.getMessage())));
            }
        } catch (Throwable ex) {
            return exceptional(ex);
        }
    }

    /**
     * TOOD
     *
     * @param entityClass
     * @param id
     * @param body
     * @param version
     * @param context
     * @return
     */

    /**
     * update by rel
     *
     * @param entityClass
     * @param rel
     * @param body
     * @param context
     * @return
     */
    @Override
    public CompletionStage<Either<UpdateMultiResult, Integer>> updateByCondition(IEntityClass entityClass, ExpRel rel, Map<String, Object> body, Map<String, Object> context) {
        return updateByConditionInner(entityClass, rel, body, context, OperationType.UPDATE);
    }

    @Override
    public CompletionStage<Either<UpdateMultiResult, Integer>> replaceByCondition(IEntityClass entityClass, ExpRel rel, Map<String, Object> body, Map<String, Object> context) {
        return updateByConditionInner(entityClass, rel, body, context, OperationType.REPLACE);
    }
    
    @Override
    public CompletionStage<Either<DeleteMultiResult, Integer>> deleteByCondition(IEntityClass entityClass, ExpRel rel, Map<String, Object> context) {
        EntityClassGroup group = getReader(entityClass, context);
        ExpContext expContext = new ExpContext().withContext(context).setSchema(group);
        ExpRel expRel = transformerPipeline.querySideHandleValue(rel, expContext);

        if (authBuilder != null && executeConfig.getUsePermission()) {

            /**
             * TODO to many to one condition
             */
            Map<Long, Set<String>> mapping = collectEntityClassIdMapping(expRel, group);
            ExpRel permissionExpRel = authBuilder.getPermissionTreeCondition(mapping, expContext, group.realProfile(), OperationType.DELETE);
            if (permissionExpRel != null) {
                expRel = expRel.mergeAnd(permissionExpRel);
            }
        }

        OqsEngineResult<Long> updateResult = writeService.deleteByCondition(entityClass, expRel, context);

        if (updateResult.isSuccess()) {
            if (updateResult.getValue().isPresent()) {
                deleteMultiCountTotal.increment(updateResult.getValue().get());
                return CompletableFuture.completedFuture(Either.right(updateResult.getValue().get().intValue()));
            } else {
                return CompletableFuture.completedFuture(Either.right(1));
            }
        } else {
            return CompletableFuture.completedFuture(Either.left(DeleteMultiResult.from(new RuntimeException(updateResult.getMessage()))));
        }
    }

    private CompletionStage<Either<UpdateMultiResult, Integer>> updateByConditionInner(IEntityClass entityClass, ExpRel rel
            , Map<String, Object> body, Map<String, Object> context, OperationType operationType) {
        EntityClassGroup group = getReader(entityClass, context);
        ExpContext expContext = new ExpContext().withContext(context).setSchema(group);
        ExpRel expRel = transformerPipeline.querySideHandleValue(rel, expContext);

        List<Tuple2<IEntityField, Object>> pipelinedBody = transformerPipeline.valueSideHandleValue(group, body, operationType);
        Map<String, Object> newBody = new HashMap<>();
        pipelinedBody.forEach(x -> {
            newBody.put(x._1.name(), x._2);
        });
        IEntity example = toIEntity(group, newBody);
        example.markTime();

        if (authBuilder != null && executeConfig.getUsePermission()) {

            /**
             * TODO to many to one condition
             */
            Map<Long, Set<String>> mapping = collectEntityClassIdMapping(expRel, group);
            ExpRel permissionExpRel = authBuilder.getPermissionTreeCondition(mapping, expContext, group.realProfile(), operationType);
            if (permissionExpRel != null) {
                expRel = expRel.mergeAnd(permissionExpRel);
            }
        }

        OqsEngineResult<Long> updateResult = writeService.updateByCondition(entityClass, expRel, example, context);

        if (updateResult.isSuccess()) {
            //Long retLong = build.getValue().get();
            if (updateResult.getValue().isPresent()) {
                replaceMultiCountTotal.increment(updateResult.getValue().get());
                return CompletableFuture.completedFuture(Either.right(updateResult.getValue().get().intValue()));
            } else {
                return CompletableFuture.completedFuture(Either.right(1));
            }
        } else {
            return CompletableFuture.completedFuture(Either.left(UpdateMultiResult.from(new RuntimeException(updateResult.getMessage()))));
        }
    }

    private Map<Long, Set<String>> collectEntityClassIdMapping(ExpRel rawQuery, EntityClassGroup group) {

        if (rawQuery == null) {
            return Collections.emptyMap();
        }

        Long mainId = group.getEntityClass().id();
        Map<Long, String> idSet = new HashMap<>();
        idSet.put(mainId, "");

        FieldCollectors fieldCollectors = new FieldCollectors();

        List<ExpNode> filters = Optional.ofNullable(rawQuery.getFilters()).orElseGet(Collections::emptyList);
        List<ExpNode> projects = Optional.ofNullable(rawQuery.getProjects()).orElseGet(Collections::emptyList);

        filters.forEach(x -> {
            if(x != null) {
                x.accept(fieldCollectors);
            }
        });
        projects.forEach(x -> {
            if(x != null) {
                x.accept(fieldCollectors);
            }
        });

        Set<String> fields = fieldCollectors.fields;

        List<ColumnField> collect = fields.stream().map(x -> group.column(x)).filter(x -> x.isPresent()).map(x -> x.get()).collect(Collectors.toList());

        Map<Long, List<ColumnField>> mapping = collect.stream().collect(Collectors.groupingBy(x -> x.originEntityClass().id()));

        Map<Long, Set<String>> retMapping = new HashMap<>();
        mapping.forEach((k, v) -> {
            retMapping.compute(k, (k1, v1) -> {
                if (v1 == null) {
                    v1 = new HashSet<>();
                }

                v1.addAll(v.stream().map(c -> {
                    if (c.name().startsWith("_") && c.name().contains(".")) {
                        //related field
                        int i = c.name().lastIndexOf(".");
                        return c.name().substring(1, i);
                    } else {
                        return null;
                    }
                }).filter(Objects::nonNull).collect(Collectors.toSet()));
                return v1;
            });
        });

        retMapping.put(group.getEntityClass().id(), Collections.emptySet());

        return retMapping;
    }

//    private ExpRel getPermissionTreeCondition(Map<Long, List<String>> involvedIdsMapping, ExpContext expContext, String profile) {
//        if (executeConfig.getUsePermission()) {
//            if (clientDataRuleProvider != null) {
//                //old
//                return getPermissionTreeConditionLegacy(involvedIdsMapping, expContext, profile);
//            } else if (dataRuleProvider != null) {
//                //new
//                return getPermissionTreeConditionNew(involvedIdsMapping, expContext, profile);
//            }
//        }
//
//        return null;
//    }
    
    
    /**
     * query with expRel
     *
     * @param expContext
     * @param rel
     * @return
     */
    @Override
    public CompletionStage<Either<QueryResult, DataCollection<Record>>> query(ExpContext expContext, ExpRel rel) {
        try {
            String profile = fetcher.getProfile(expContext.getContext());
            
            boolean isAgg = rel.getProjects().stream().anyMatch(x -> x instanceof ExpFunc);
            
            if (authBuilder != null && executeConfig.getUsePermission()) {

                /**
                 * TODO to many to one condition
                 */
                Map<Long, Set<String>> mapping = collectEntityClassIdMapping(rel, expContext.getSchema());
                ExpRel permissionExpRel = authBuilder.getPermissionTreeCondition(mapping, expContext, profile, OperationType.QUERY);
                if (permissionExpRel != null) {
                    //List<ExpNode> previewProjects = rel.getProjects();
                    //ExpGroupBy previewGroup = rel.getGroupBy();
                    if(isAgg) {
                        //skip projects currently
                        //TODO
                        permissionExpRel.getProjects().clear();
                        rel = rel.mergeAnd(permissionExpRel);
                    } else {
                        rel = rel.mergeAnd(permissionExpRel);
                    }
                    
//                    if (permissionExpRel.getProjects() == null || permissionExpRel.getProjects().isEmpty()) {
//                        rel = rel.mergeOrProjects(previewProjects);
//                    }
//                    
//                    if (rel.getGroupBy() == null && previewGroup != null) {
//                        ((ExpQuery)rel).groupBy(previewGroup);
//                    }
                }
            }

            /**
             * add query side pipeline
             */
            if (transformerPipeline != null) {
                rel = transformerPipeline.querySideHandleValue(rel, expContext);
            }

            /**
             * current transform will extract or related toMany and To one to and or not allowed in or
             */
            RelNode opTreeS = ExpTreeToRel.toRelTree(expContext, rel, config, executeConfig);
//            SqlNode sqlNode = new RelToSqlConverter(MysqlSqlDialectEx.DEFAULT).visitRoot(opTreeS)
//                    .asStatement();
//            String targetSql = Util.toLinux(sqlNode.toSqlString(MysqlSqlDialectEx.DEFAULT).getSql())
//                    .replaceAll("\n", " ");
            try (Connection connection = dataSource.getConnection()) {
                RelRunner unwrap = connection.unwrap(RelRunner.class);
                try (PreparedStatement preparedStatement = unwrap.prepareStatement(opTreeS)) {
//                try(PreparedStatement preparedStatement = connection.prepareStatement(targetSql)) {
                    ResultSet resultSet = preparedStatement.executeQuery();
                    EntityClassGroup group = expContext.getSchema();

                    List<String> keys = rel.getProjects().stream().map(x -> (ExpField) x).map(x -> {
                        return Optional.ofNullable(x.getAlise()).orElse(x.getName());
                    }).collect(Collectors.toList());

                    if (keys.isEmpty()) {
                        keys = group.getAllFields().stream().map(x -> x.name()).collect(Collectors.toList());
                    }

                    Collection<IEntityField> fields = group.getAllFields();
                    //Caution toMap is dangerous
                    //yeh if the same relatedCode one2many and many2one will make the same field
                    Map<String, IEntityField> mapping = fields.stream().collect(Collectors.toMap(IEntityField::name, y -> y, (a, b) -> a));

                    AtomicBoolean hasAggField = new AtomicBoolean(false);
                    List<Record> records = new ArrayList<>();
                    List<Map<Tuple2<String, Long>, Object>> valuesList = new ArrayList<>();
                    while (resultSet.next()) {
                        Map<Tuple2<String, Long>, Object> values = new HashMap<>();
                        keys.forEach(key -> {
                            try {
                                if (key.equalsIgnoreCase("*")) {
                                    fields.forEach(x -> {
                                        try {
                                            String value = resultSet.getString(x.name());
                                            IEntityField targetField;
                                            targetField = mapping.get(x.name());
                                            Tuple2<String, Long> keyTuple = Tuple.of(x.name(), targetField.id());
                                            values.put(keyTuple, value);
                                        } catch (SQLException e) {
                                            logErrorPattern(log, LoggingPattern.CONDITION_QUERY_ERROR, e);
                                        }
                                    });
                                } else {
                                    String value = resultSet.getString(key);
                                    IEntityField targetField;
                                    if (key.startsWith("_")) {
                                        targetField = group.column(key).orElse(null);
                                    } else {
                                        targetField = mapping.get(key);
                                    }
                                    Tuple2<String, Long> keyTuple = null;
                                    if (targetField != null) {
                                        keyTuple = Tuple.of(key, targetField.id());
                                    } else {
                                        hasAggField.set(true);
                                        keyTuple = Tuple.of(key, -1L);
                                    }
                                    values.put(keyTuple, value);
                                }
                            } catch (SQLException e) {
                                logErrorPattern(log, LoggingPattern.CONDITION_QUERY_ERROR, e);
                            }
                        });

                        valuesList.add(values);
                    }

                    /**
                     * poly is a Map<String, List<Tuple3<Long, String, Long>>>
                     */
                    Object poly = contextService.getAll().get("poly");
                    Map<Long, Object> valuesMapping = new HashMap();
                    Map<Long, EntityClassGroup> groupMapping = new HashMap<>();
                    if (poly instanceof Map) {
                        contextService.getAll().remove("poly");
                        //query poly
                        Map<String, List<Tuple3<Long, String, Long>>> polyMapping = (Map<String, List<Tuple3<Long, String, Long>>>) poly;
                        for (Map.Entry<String, List<Tuple3<Long, String, Long>>> entry : polyMapping.entrySet()) {
                            List<Tuple3<Long, String, Long>> value = entry.getValue();
                            if (!value.isEmpty()) {
                                // query with the whole
                                Tuple3<Long, String, Long> sample = value.get(0);
                                String profileStr = sample._2;
                                Long targetEntityClassId = sample._3;

                                ExpContext polyContext = new ExpContext();
                                Optional<IEntityClass> targetEntityClass = engine.load(targetEntityClassId.toString(), profileStr);
                                if (targetEntityClass.isPresent()) {
                                    EntityClassGroup polyGroup = engine.describe(targetEntityClass.get(), profileStr);
                                    polyContext.setSchema(polyGroup);
                                    contextService.getAll().put("profile", Optional.ofNullable(polyGroup.realProfile()).orElse(""));
                                    Map<String, Object> ctx = new HashMap<>();
                                    ctx.put("only_query", true);
                                    polyContext.withContext(ctx);
                                    List<ExpNode> ids = value.stream().flatMap(x -> ExpValue.from(x._1).stream()).collect(Collectors.toList());
                                    ExpRel query = new ExpQuery().filters(ExpCondition.call(ExpOperator.IN, ExpField.field("id"), ids)).range(1, ids.size());
                                    RelNode polyQuery = ExpTreeToRel.toRelTree(polyContext, query, config, executeConfig);

                                    //target entityclass
                                    List<String> polyKeys = polyGroup.getAllFields().stream().map(x -> x.name()).collect(Collectors.toList());
                                    Collection<IEntityField> polyFields = polyGroup.getAllFields();
                                    //Caution toMap is dangerous
                                    //yeh if the same relatedCode one2many and many2one will make the same field
                                    Map<String, IEntityField> polyFieldsMapping = polyFields.stream().collect(Collectors.toMap(IEntityField::name, y -> y, (a, b) -> a));

                                    /**
                                     * current still depend on contextService to pass profile
                                     */
                                    SqlNode sqlNodePoly = new RelToSqlConverter(MysqlSqlDialectEx.DEFAULT).visitRoot(polyQuery).asStatement();
                                    String targetSqlPoly = Util.toLinux(sqlNodePoly.toSqlString(MysqlSqlDialectEx.DEFAULT).getSql()).replaceAll("\n", " ");
                                    try (PreparedStatement preparedStatement2 = unwrap.prepareStatement(polyQuery)) {
//                                    try (PreparedStatement preparedStatement2 = connection.prepareStatement(targetSqlPoly)) {
                                        ResultSet polyResult = preparedStatement2.executeQuery();
                                        contextService.getAll().remove("profile");
                                        while (polyResult.next()) {
                                            long realId = polyResult.getLong("ID");
                                            Map<Tuple2<String, Long>, Object> values = new HashMap<>();
                                            //create values from rs
                                            polyKeys.forEach(key -> {
                                                try {
                                                    if (key.equalsIgnoreCase("*")) {
                                                        polyFields.forEach(x -> {
                                                            try {
                                                                String polyValue = polyResult.getString(x.name());
                                                                IEntityField targetField;
                                                                targetField = polyFieldsMapping.get(x.name());
                                                                Tuple2<String, Long> keyTuple = Tuple.of(x.name(), targetField.id());
                                                                values.put(keyTuple, polyValue);
                                                            } catch (SQLException e) {
                                                                logErrorPattern(log, LoggingPattern.CONDITION_QUERY_ERROR, "DealWithStar", e);
                                                            }
                                                        });
                                                    } else {
                                                        String polyValue = polyResult.getString(key);
                                                        IEntityField targetField;
                                                        if (key.startsWith("_")) {
                                                            targetField = polyGroup.column(key).orElse(null);
                                                        } else {
                                                            targetField = polyFieldsMapping.get(key);
                                                        }
                                                        Tuple2<String, Long> keyTuple = Tuple.of(key, targetField.id());
                                                        values.put(keyTuple, polyValue);
                                                    }
                                                } catch (SQLException e) {
                                                    logErrorPattern(log, LoggingPattern.CONDITION_QUERY_ERROR, "DealWithPloy", e);
                                                }
                                            });

                                            valuesMapping.put(realId, values);
                                            groupMapping.put(realId, polyGroup);
                                        }
                                    }
                                }
                            }
                        }
                    }

                    //generate New Mapping
                    for (Map<Tuple2<String, Long>, Object> tuple2ObjectMap : valuesList) {
                        if (hasAggField.get()) {
                            //new IEntityFrom entityUp
                            List<IEntityField> freeFields = tuple2ObjectMap.entrySet().stream().map(x -> {
                                IEntityField field = new EntityField(0L, x.getKey()._1, FieldType.STRING);
                                return field;
                            }).collect(Collectors.toList());

                            GeneralRecord record = new GeneralRecord(freeFields);
                            tuple2ObjectMap.entrySet().forEach(v -> {
//                                if (!EmptyValue.isEmpty(v.getValue())) {
                                record.set(v.getKey()._1, v.getValue());
//                                }
                            });
                            records.add(record);
                        } else {
                            Optional<Long> id = tuple2ObjectMap.entrySet().stream().filter(entry -> {
                                        return entry.getKey()._1.equalsIgnoreCase("id");
                                    }).map(x -> x.getValue())
                                    //in-case some wrong id
                                    .filter(Objects::nonNull).map(Object::toString).map(Long::parseLong).findFirst();
                            Record record = null;
                            if (id.isPresent()) {
                                //do replace
                                Object targetValue = valuesMapping.get(id.get());
                                EntityClassGroup targetGroup = groupMapping.get(id.get());
                                if (targetValue != null && targetGroup != null) {
                                    record = targetGroup.toRecordNew((Map<Tuple2<String, Long>, Object>) targetValue, targetGroup.getEntityClass().id());
                                    Map<String, Object> retBody = record.toMap(null);
                                    List<Tuple2<IEntityField, Object>> tupleList = transformerPipeline.valueSideHandleValue(targetGroup, retBody, OperationType.RESULT);
                                    record.fromMap(getMapFromTuple(tupleList));
                                    records.add(record);
                                }
                            }

                            if (record == null) {
                                record = group.toRecordNew(tuple2ObjectMap, group.getEntityClass().id());
                                Map<String, Object> retBody = record.toMap(null);
                                List<Tuple2<IEntityField, Object>> tupleList = transformerPipeline.valueSideHandleValue(group, retBody, OperationType.RESULT);
                                record.fromMap(getMapFromTuple(tupleList));
                                records.add(record);
                            }
                        }
                    }

                    Object showCount = contextService.getAll().get("show_count");
                    Long rowNum = 0L;
                    if (showCount != null) {
                        rowNum = (long) showCount;
                    }

                    List<Record> recordList = mergeRecord(group, records, Collections.emptyMap(), fetcher.getProfile(expContext.getContext()));

                    Object verMapping = contextService.getAll().get("ver");
                    if (verMapping instanceof Map) {
                        recordList.forEach(x -> {
                            Long id = x.getId();
                            Integer ver = ((Map<Long, Integer>) verMapping).get(id);
                            if (ver != null) {
                                x.setVersion(ver);
                            }
                        });
                    }

                    contextService.getAll().remove("show_count");
                    contextService.getAll().remove("ver");
                    DataCollection<Record> retCollection = new DataCollection<>(rowNum.intValue(), recordList);
                    queryCountTotal.increment();
                    return CompletableFuture.completedFuture(Either.right(retCollection));
                }
            }
        } catch (Throwable e) {
            return exceptional(e);
        }


//        DataCollection<Record> objectDataCollection = new DataCollection<>(1000, Collections.emptyList());
//        CompletableFuture<Either<QueryResult, DataCollection<Record>>> eitherCompletableFuture = CompletableFuture.completedFuture(Either.right(objectDataCollection));
//        return eitherCompletableFuture;
    }

    private Map<String, Object> getMapFromTuple(List<Tuple2<IEntityField, Object>> tupleList) {
        Map<String, Object> newBody = tupleList.stream().filter(Objects::nonNull).collect(HashMap::new, (m, v) -> m.put(v._1().name(), v._2()), HashMap::putAll);
        return newBody;
    }

    private List<Record> mergeRecord(EntityClassGroup entityClassGroup, List<Record> records, Map<String, Map<Long, Record>> valueMap, String profile) {

        Set<String> rels = valueMap.keySet();
        return records.stream().map(x -> {
            List<IEntityField> newFields = new LinkedList<>();
            List<Object> values = new LinkedList<>();
            newFields.addAll(x.fields(entityClassGroup.getEntityClass()));
            values.addAll(x.values());
            for (String rel : rels) {
                Optional<Object> idOp = x.get(rel.concat(".id"));

                Optional<IEntityClass> relatedEntityClassOp = entityClassGroup.relatedEntityClass(rel);

                if (idOp.isPresent() && relatedEntityClassOp.isPresent()) {
                    Map<Long, Record> mapping = valueMap.get(rel);
                    if (mapping != null) {
                        Record record = mapping.get(Long.parseLong(idOp.get().toString()));

                        if (record != null) {
                            newFields.addAll(record.fields(rel, relatedEntityClassOp.get()));
                            values.addAll(record.values());
                        } else {
                            log.warn("record is null ");
                        }
                    }
                }
            }

            GeneralRecord generalRecord = new GeneralRecord(newFields);
            generalRecord.setValues(values);
            generalRecord.setId(x.getId());
            generalRecord.setTypeId(x.getTypeId());
            return generalRecord;
        }).collect(Collectors.toList());
    }

    /**
     * TODO
     *
     * @param entityClass
     * @param rel
     * @param context
     * @return
     */
    @Override
    public CompletionStage<Either<QueryResult, DataCollection<Record>>> query(IEntityClass entityClass, ExpRel rel, Map<String, Object> context) {
        EntityClassGroup group = engine.describe(entityClass, fetcher.getProfile(context));
        ExpContext expContext = new ExpContext();
        expContext.setSchema(group);
        expContext.withContext(context);
        return this.query(expContext, rel);
    }

    @Override
    public <T> Iterable<T> queryIterate(IEntityClass entityClass, ExpRel initRel, BiFunction<ExpRel, Record, ExpRel> conditionTransformer, Function<Record, T> transformer, Map<String, Object> context) {
        int step = QUERY_STEP;
        return () -> new LazyFetchIterator<>(initRel, expRel -> {
            log.debug("Current expRel {}", expRel);
            Either<QueryResult, DataCollection<Record>> queryResult = query(entityClass, expRel, context).toCompletableFuture().join();

            if (queryResult.isRight()) {
                return queryResult.get().getRows();
            } else {
                //how to handle when query is error?
                return Collections.emptyList();
            }
        }, (expRel, last) -> {
            return conditionTransformer.apply(expRel, last);
        }, request -> true, response -> response.isEmpty() || response.size() < step, transformer);
    }

    @Override
    public <T> Iterable<T> queryIterate(IEntityClass entityClass, ExpRel rel, Function<Record, T> transformer, boolean isLegacy, Map<String, Object> context) {
        AtomicInteger startIndex = new AtomicInteger(1);
        return this.queryIterate(entityClass, rel, (expRel, last) -> {
            return ExpFactory.withRange(expRel, new ExpRange(startIndex.incrementAndGet(), QUERY_STEP));
        }, transformer, context);
    }

    @Override
    public CompletionStage<Either<QueryOneResult, Record>> findOneById(IEntityClass entityClass, Long id, Map<String, Object> context) {
        ExpQuery expQuery = new ExpQuery().filters(ExpCondition.call(ExpOperator.EQUALS, ExpField.ID, ExpValue.from(id))).range(1, 1);
        return this.query(entityClass, expQuery, context).thenApply(x -> {
            boolean right = x.isRight();
            if (right) {
                DataCollection<Record> retCollection = x.get();
                if (retCollection.getRowNum() > 0) {
                    Record record = retCollection.getRows().get(0);
                    Either<QueryOneResult, Record> rightRet = Either.right(record);
                    return rightRet;
                } else {
                    Either<QueryOneResult, Record> notFound = Either.left(QueryOneResult.from(new RecordMissingException(RecordMissingException.getMsg(entityClass.code(), id.toString()))));
                    return notFound;
                }
            } else {
                QueryResult leftReason = x.getLeft();
                Either<QueryOneResult, Record> dataCollections = Either.left(QueryOneResult.from(new RuntimeException(leftReason.getMessage())));
                return dataCollections;
            }
        });
    }

    /**
     * TODO
     *
     * @param entityClass
     * @param subEntityClass
     * @param id
     * @param context
     * @return
     */
    @Override
    public CompletionStage<Either<QueryOneResult, Record>> findOneById(IEntityClass entityClass, IEntityClass subEntityClass, Long id, Map<String, Object> context) {
        return this.findOneById(entityClass, id, context);
    }

    @Override
    public CompletionStage<Integer> count(IEntityClass entityClass, ExpRel rel, Map<String, Object> context) {
        ExpQuery query = (ExpQuery) rel;
        query.range(1, 1);
        return query(entityClass, rel, context).thenApply(x -> {
            return x.map(DataCollection::getRowNum).getOrElseThrow(str -> {
                QueryResult queryResult = str;
                throw new RuntimeException(queryResult.getMessage());
            });
        });
    }

    @Override
    public EntityClassGroup getReader(IEntityClass entityClass, Map<String, Object> context) {
        return engine.describe(entityClass, fetcher.getProfile(context));
    }

    @Override
    public EntityClassEngine getEntityClassEngine() {
        return engine;
    }

    @Override
    public ProfileFetcher getFetcher() {
        return fetcher;
    }

    public void setFetcher(ProfileFetcher fetcher) {
        this.fetcher = fetcher;
    }

    @Override
    public void validate(IEntityClass entityClass, Map<String, Object> body) {
        transformerPipeline.validate(entityClass, (List<IEntityField>) entityClass.fields(), body);
    }

    class FieldCollectors implements ExpVisitor<Void> {

        private Set<String> fields = new HashSet<>();

        @Override
        public Void visit(ExpField field) {
            fields.add(field.getName());
            return null;
        }

        @Override
        public Void visit(ExpCondition rel) {
            rel.getExpNodes().forEach(x -> x.accept(this));
            return null;
        }

        @Override
        public Void visit(ExpValue value) {
            return null;
        }

        @Override
        public Void visit(ExpBi bi) {
            return null;
        }

        @Override
        public Void visit(ExpSort expSort) {
            return null;
        }

        @Override
        public Void visit(ExpRange range) {
            return null;
        }
    }
}
