package com.xforceplus.ultraman.adapter.query.strategy;

import com.xforceplus.metadata.schema.typed.BoIndex;
import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.engine.EntityClassGroup;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.entity.IEntityField;
import com.xforceplus.ultraman.metadata.entity.legacy.impl.EntityClass;
import com.xforceplus.ultraman.sdk.core.calcite.UltramanSearchTable;
import com.xforceplus.ultraman.sdk.core.calcite.UltramanTableScan;
import com.xforceplus.ultraman.sdk.core.calcite.oqs.DataQueryProvider;
import com.xforceplus.ultraman.sdk.core.calcite.oqs.OqsengineFilter;
import com.xforceplus.ultraman.sdk.core.calcite.oqs.strategy.QueryProviderSelectStrategy;
import com.xforceplus.ultraman.sdk.core.facade.ProfileFetcher;
import com.xforceplus.ultraman.sdk.infra.exceptions.EntityClassMissingException;
import com.xforceplus.ultraman.sdk.infra.logging.LoggingPattern;
import com.xforceplus.ultraman.sdk.infra.logging.LoggingUtils;
import com.xforceplus.ultraman.sdk.invocation.invoke.config.InvocationConfig;
import io.vavr.Tuple2;
import lombok.extern.slf4j.Slf4j;
import org.apache.calcite.DataContext;
import org.apache.calcite.plan.RelOptTable;
import org.apache.calcite.rel.BiRel;
import org.apache.calcite.rel.RelFieldCollation;
import org.apache.calcite.rel.RelNode;
import org.apache.calcite.rel.RelShuttle;
import org.apache.calcite.rel.core.AggregateCall;
import org.apache.calcite.rel.core.TableFunctionScan;
import org.apache.calcite.rel.core.TableScan;
import org.apache.calcite.rel.hint.RelHint;
import org.apache.calcite.rel.logical.*;
import org.apache.calcite.rel.type.RelDataType;
import org.apache.calcite.rel.type.RelDataTypeField;
import org.apache.calcite.rel.type.StructKind;
import org.apache.calcite.rex.*;
import org.apache.calcite.sql.SqlOperator;
import org.apache.calcite.util.Pair;
import org.apache.calcite.util.Sarg;

import java.util.*;
import java.util.concurrent.atomic.AtomicBoolean;

import static com.xforceplus.ultraman.sdk.infra.logging.LoggingPattern.GENERAL_UTILS_ERROR;
import static com.xforceplus.ultraman.sdk.infra.logging.LoggingUtils.logErrorPattern;
import static org.apache.calcite.sql.fun.SqlStdOperatorTable.*;

/**
 * query with id and unique always in master storage
 */
@Slf4j
public class MasterStickQueryStrategy implements QueryProviderSelectStrategy {

    private EntityClassEngine engine;

    private InvocationConfig invocationConfig;

    private ProfileFetcher fetcher;
    
    private ContextService contextService;

    public MasterStickQueryStrategy(EntityClassEngine engine, ProfileFetcher fetcher, ContextService contextService, InvocationConfig invocationConfig) {
        this.invocationConfig = invocationConfig;
        this.engine = engine;
        this.fetcher = fetcher;
        this.contextService = contextService;
    }

    @Override
    public String name() {
        return "master-stick";
    }

    @Override
    public Map<DataQueryProvider, Double> score(String app, IEntityClass entityClass, String profile, RelDataType type
            , List<RexNode> ops, List<Map.Entry<String, Tuple2<StructKind, Class>>> fields
            , List<Pair<RexNode, String>> projects, List<Map.Entry<String, RelFieldCollation.Direction>> sort
            , Long offset, Long fetch, List<String> groupBy, List<AggregateCall> aggs, List<RelHint> hints
            , RelNode rawTree, DataContext dataContext, List<DataQueryProvider> dataQueryProviderList) {

        Map<DataQueryProvider, Double> master = new HashMap<>();
        try {
            if (dataQueryProviderList.size() > 1) {
                //check if the query is id or unique
                boolean suitable = isSuitable(rawTree);
                if (suitable) {
                    Optional<DataQueryProvider> first = dataQueryProviderList.stream()
                            .filter(x -> x.type() == DataQueryProvider.QueryProviderType.MASTER)
                            .findFirst();

                    if (first.isPresent()) {
                        master.put(first.get(), Double.MAX_VALUE);
                    }
                }
            }
        } catch (Throwable throwable) {
            logErrorPattern(log, GENERAL_UTILS_ERROR, throwable);
        }
        return master;
    }

    private boolean isSuitable(RelNode rawTree) {
        boolean containsUnique = invocationConfig.isSearchMasterWithUniqueIndex();
        CheckCondition checkCondition = new CheckCondition(containsUnique);
        rawTree.accept(checkCondition);
        return checkCondition.suitable();
    }

    class CheckCondition implements RelShuttle {

        private boolean containsUnique;

        private boolean suitable = false;

        private IEntityClass entityClass;

        public CheckCondition(boolean containsUnique) {
            this.containsUnique = containsUnique;
        }

        public boolean suitable() {
            return suitable;
        }

        @Override
        public RelNode visit(TableScan scan) {
            if (scan instanceof UltramanTableScan) {
                UltramanSearchTable table = ((UltramanTableScan) scan).getSearchTable();
                String code = table.getCode();
                String profile = fetcher.getProfile(contextService.getAll());
                Optional<IEntityClass> entityClassOp = engine.loadByCode(code, profile);
                if (entityClassOp.isPresent()) {
                    entityClass = entityClassOp.get();
                } else {
                    throw new EntityClassMissingException(EntityClassMissingException.getMsg(code));
                }
            }

            return null;
        }

        @Override
        public RelNode visit(TableFunctionScan scan) {
            return null;
        }

        @Override
        public RelNode visit(LogicalValues values) {
            return null;
        }

        @Override
        public RelNode visit(LogicalFilter filter) {
            return null;
        }

        @Override
        public RelNode visit(LogicalCalc calc) {
            return null;
        }

        @Override
        public RelNode visit(LogicalProject project) {
            return null;
        }

        @Override
        public RelNode visit(LogicalJoin join) {
            return null;
        }

        @Override
        public RelNode visit(LogicalCorrelate correlate) {
            return null;
        }

        @Override
        public RelNode visit(LogicalUnion union) {
            return null;
        }

        @Override
        public RelNode visit(LogicalIntersect intersect) {
            return null;
        }

        @Override
        public RelNode visit(LogicalMinus minus) {
            return null;
        }

        @Override
        public RelNode visit(LogicalAggregate aggregate) {
            return null;
        }

        @Override
        public RelNode visit(LogicalMatch match) {
            return null;
        }

        @Override
        public RelNode visit(LogicalSort sort) {
            return null;
        }

        @Override
        public RelNode visit(LogicalExchange exchange) {
            return null;
        }

        @Override
        public RelNode visit(LogicalTableModify modify) {
            return null;
        }

        @Override
        public RelNode visit(RelNode node) {
            node.getInputs().forEach(x -> x.accept(this));

            if (node instanceof OqsengineFilter) {
                RexNode condition = ((OqsengineFilter) node).getCondition();
                ConditionFinderVisitor conditionFinderVisitor = new ConditionFinderVisitor(engine, node, entityClass);
                condition.accept(conditionFinderVisitor);
                suitable = conditionFinderVisitor.isSuitable();
            }
            return null;
        }
    }

    class ConditionFinderVisitor extends RexVisitorImpl<Void> {

        private boolean suitable = false;

        private RelNode currentNode;

        private EntityClassEngine engine;

        private IEntityClass currentEntityClass;

        private Set<String> names = new HashSet<>();

        private Set<Sarg> args = new HashSet<>();

        protected ConditionFinderVisitor(EntityClassEngine engine, RelNode currentNode, IEntityClass entityClass) {
            super(true);
            this.currentNode = currentNode;
            this.engine = engine;
            this.currentEntityClass = entityClass;
        }

        public boolean isSuitable() {
            if (names.isEmpty()) {
                return suitable;
            } else {
                Collection<BoIndex> boIndices = currentEntityClass.uniqueIndexes();
                AtomicBoolean atomicBoolean = new AtomicBoolean(false);
                boIndices.forEach(in -> {
                    String fieldIds = in.getFieldIds();
                    String[] namesArray = fieldIds.split(",");
                    Set<String> set = new HashSet<>();
                    for (String name : namesArray) {
                        currentEntityClass.field(name).ifPresent(entityField -> {
                            set.add(entityField.name());
                        });
                    }
                    if (names.containsAll(set)) {
                        atomicBoolean.set(true);
                    }
                });
                
                return atomicBoolean.get() || (names.contains("id") && (args.isEmpty() || args.stream().allMatch(Sarg::isPoints)));
            }
        }

        public void setSuitable(boolean suitable) {
            this.suitable = suitable;
        }

        @Override
        public Void visitCall(RexCall call) {
            SqlOperator operator = call.getOperator();
            if (operator == AND) {
                call.getOperands().stream()
                        .filter(x -> x instanceof RexCall)
                        .filter(x -> ((RexCall) x).getOperator() == EQUALS)
                        .forEach(x -> x.accept(this));
            } else if (operator == OR) {
                this.suitable = false;
            } else if (operator == EQUALS || operator == IN || operator == SEARCH) {
                //check if is id or unique
                call.getOperands().stream()
                        .filter(x -> x instanceof RexInputRef)
                        .forEach(x -> {
                            int index = ((RexInputRef) x).getIndex();
                            RelDataTypeField relDataTypeField = currentNode.getRowType().getFieldList().get(index);
                            names.add(relDataTypeField.getName().toLowerCase(Locale.ROOT));
                        });

                if (operator == SEARCH) {
                    call.getOperands().stream().filter(x -> x instanceof RexLiteral)
                            .map(x -> ((RexLiteral) x).getValue()).filter(x -> x instanceof Sarg)
                            .forEach(sarg -> {
                                args.add((Sarg) sarg);
                            });
                }
            } else {
                suitable = false;
            }
            return null;
        }

        private EntityClassGroup findRelatedEntityClass(RelNode relNode, List<EntityClassGroup> allRelatedEntityClasses) {
            Stack<RelNode> stack = new Stack<>();
            stack.push(relNode);
            while (!stack.isEmpty()) {
                RelNode next = stack.pop();
                if (next instanceof TableScan) {
                    RelOptTable table = ((TableScan) next).getTable();
                    String entityCode = table.getQualifiedName().get(1);
                    Optional<EntityClassGroup> first = allRelatedEntityClasses.stream()
                            .filter(x -> x.getEntityClass().code().equalsIgnoreCase(entityCode)).findFirst();
                    if (first.isPresent()) {
                        return first.get();
                    }
                } else {
                    //TODO current we treat the combined relNode as single always get the left one
                    //but we may get following case  BiRel or SingleRel
                    if (next instanceof BiRel) {
                        RelNode input = next.getInput(0);
                        stack.push(input);
                    } else {
                        RelNode input = next.getInput(0);
                        stack.push(input);
                    }
                }
            }

            throw new EntityClassMissingException(EntityClassMissingException.getMsg());
        }
    }

}
