package com.xforceplus.ultraman.adapter.auth.tree;

import com.xforceplus.tenant.data.auth.dto.ConditionDTO;
import com.xforceplus.tenant.data.auth.dto.SqlFieldConditionDTO;
import com.xforceplus.tenant.data.auth.dto.SqlFieldDTO;
import com.xforceplus.tenant.data.auth.dto.Status;
import com.xforceplus.tenant.data.domain.rule.Relation;
import com.xforceplus.tenant.data.domain.rule.RelationType;
import com.xforceplus.tenant.data.domain.rule.RuleConditionRelationship;

import com.xforceplus.ultraman.sdk.core.rel.tree.BinaryTreeNode;
import com.xforceplus.ultraman.sdk.core.rel.tree.LinkedBinaryTreeNode;
import com.xforceplus.ultraman.sdk.core.rel.tree.builder.TreeBuilder;
import com.xforceplus.ultraman.sdk.core.rel.tree.dsl.ConditionNode;
import com.xforceplus.ultraman.sdk.core.rel.tree.dsl.OperationNode;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import org.apache.commons.lang3.StringUtils;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * build condition tree using natural order
 */
public class DefaultTreeBuilder implements TreeBuilder<ConditionDTO> {

    /**
     * build conditions
     *
     * @param conditions
     * @return
     */
    @Override
    public BinaryTreeNode<ConditionNode> build(List<ConditionDTO> conditions, String relation) {

        BinaryTreeNode<ConditionNode> prev = null;

        for (ConditionDTO condition : conditions) {

            if (condition.getStatus() == Status.VALID) {

                /**
                 * every condition dto can convert to node
                 */
                ConditionDTONode node = new ConditionDTONode(condition);
                BinaryTreeNode<ConditionNode> nextMain = new LinkedBinaryTreeNode<>(node);

                Relation nextRelation = condition.getNextRelation();
                BinaryTreeNode<ConditionNode> nextOperation = null;
                if (nextRelation != null) {
                    OperationNode operationNode = toOperationNode(nextRelation);
                    nextOperation = new LinkedBinaryTreeNode<>(operationNode);
                }

                if (prev == null) {
                    if (nextOperation != null) {
                        prev = nextOperation;
                        nextOperation.setLeft(expandSqlField(nextMain, relation));
                    } else {
                        prev = nextMain;
                    }
                } else {
                    if (prev.getData() instanceof ConditionDTO) {
                        //TODO if use default and
                        throw new RuntimeException("Condition Tree has no link, TODO");
                    } else {
                        //rotate this tree
                        prev.setRight(expandSqlField(nextMain, relation));
                        if (nextOperation != null) {
                            nextOperation.setLeft(prev);
                            prev = nextOperation;
                        }
                    }
                }
            }
        }

        return prev;
    }

    private BinaryTreeNode<ConditionNode> buildSubTree(List<Tuple2<String, List<SqlFieldConditionDTO>>> sqlConditions) {
        BinaryTreeNode<ConditionNode> prev = null;

        for (Tuple2<String, List<SqlFieldConditionDTO>> tuple : sqlConditions) {
            for (SqlFieldConditionDTO condition : tuple._2()) {

                if (condition.getStatus() == Status.VALID || condition.getStatus() == null) {

                    /**
                     * every condition dto can convert to node
                     */
                    ConditionSQLNode node = new ConditionSQLNode(tuple._1(), condition);
                    BinaryTreeNode<ConditionNode> nextMain = new LinkedBinaryTreeNode<>(node);

                    RuleConditionRelationship nextRelation = condition.getDataRelation();
                    BinaryTreeNode<ConditionNode> nextOperation = null;
                    if (nextRelation != null && nextRelation != RuleConditionRelationship.UN_KNOWN) {
                        OperationNode operationNode = toOperationNode(nextRelation);
                        nextOperation = new LinkedBinaryTreeNode<>(operationNode);
                    }

                    if (prev == null) {
                        if (nextOperation != null) {
                            prev = nextOperation;
                            nextOperation.setLeft(nextMain);
                        } else {
                            prev = nextMain;
                        }
                    } else {
                        if (prev.getData() instanceof SqlFieldConditionDTO) {
                            //TODO if use default and
                            throw new RuntimeException("Condition Tree has no link, TODO");
                        } else {
                            //rotate this tree
                            prev.setRight(nextMain);
                            if (nextOperation != null) {
                                nextOperation.setLeft(prev);
                                prev = nextOperation;
                            }
                        }
                    }
                }
            }
        }

        return prev;
    }

    private OperationNode toOperationNode(RuleConditionRelationship relationship) {
        if (relationship == RuleConditionRelationship.AND) {
            return new OperationNode(OperationNode.OP.AND);
        } else if (relationship == RuleConditionRelationship.OR) {
            return new OperationNode(OperationNode.OP.OR);
        } else {
            return new OperationNode(OperationNode.OP.UNKNOWN);
        }
    }

    private OperationNode toOperationNode(Relation relation) {
        if (relation.getCode().equals(RelationType.AND.getCode())) {
            //and
            return new OperationNode(OperationNode.OP.AND);
        } else if (relation.getCode().equals(RelationType.OR.getCode())) {
            return new OperationNode(OperationNode.OP.OR);
        } else {
            return new OperationNode(OperationNode.OP.UNKNOWN);
        }
    }

    /**
     * expand conditionDTO to SqlCondition
     *
     * @return
     */
    private BinaryTreeNode<ConditionNode> expandSqlField(BinaryTreeNode<ConditionNode> nextMain, String relation) {
        BinaryTreeNode<ConditionNode> ptr = nextMain;

        if (ptr != null) {
            ConditionNode data = nextMain.getData();
            if (data != null && data.getNodeType() == ConditionNode.NodeType.RAW) {
                Object value = data.value();
                ConditionDTO dto = (ConditionDTO) value;
                List<SqlFieldDTO> sqlFields = dto.getRuleSqlFields();

                List<Tuple2<String, List<SqlFieldConditionDTO>>> sqlConditions = Optional.ofNullable(sqlFields)
                        .orElseGet(Collections::emptyList).stream().filter(x -> x.getStatus() == Status.VALID)
                        .map(x -> {
                            String fieldName = x.getFieldName();
                            if(!StringUtils.isEmpty(relation)) {
                                fieldName = "_".concat(relation).concat(".").concat(fieldName);
                            }
                            List<SqlFieldConditionDTO> ruleSqlFieldConditions = x.getRuleSqlFieldConditions();
                            return Tuple.of(fieldName, Optional.ofNullable(ruleSqlFieldConditions).orElseGet(Collections::emptyList));
                        }).collect(Collectors.toList());

                if (sqlConditions.isEmpty()) {
                    //empty should not return
                    return null;
                }

                return buildSubTree(sqlConditions);
            }
        }

        return null;
    }
}
