package com.xforceplus.ultraman.adapter.utils;

import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.engine.EntityClassGroup;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.entity.IEntityField;
import com.xforceplus.ultraman.metadata.entity.IRelation;
import com.xforceplus.ultraman.metadata.entity.legacy.impl.ColumnField;
import com.xforceplus.ultraman.metadata.helper.PropertyHelper;
import com.xforceplus.ultraman.oqsengine.sdk.*;
import com.xforceplus.ultraman.sdk.core.rel.legacy.*;
import com.xforceplus.ultraman.sdk.infra.exceptions.InvalidInputsException;
import io.vavr.Tuple2;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.util.*;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.xforceplus.ultraman.adapter.utils.IEntityClassHelper.toFieldUp;
import static com.xforceplus.ultraman.metadata.helper.PropertyHelper.extractRelated;
import static com.xforceplus.ultraman.metadata.helper.PropertyHelper.generateRelatedFieldName;


/**
 * a helper for rel Tree to convert to any query
 */
@Slf4j
public class RelTreeHelper {

    /**
     * flatten condition to map
     */
    static class FlattenVisitor implements ExpVisitor<Void> {

        private EntityClassGroup group;

        private String profile;

        private Map<String, Object> flatValue;

        public FlattenVisitor(EntityClassGroup group, String profile, Map<String, Object> flatValue) {
            this.group = group;
            this.profile = profile;
            this.flatValue = flatValue;
        }

        @Override
        public Void visit(ExpField field) {
            return null;
        }

        @Override
        public Void visit(ExpCondition rel) {
            convertTree(rel, this, group, false, false, ctx -> {
                if (Boolean.TRUE.equals(ctx.get("returnEmpty"))) {
                    //do nothing
                } else {
                    String code = (String) ctx.get("code");
                    ColumnField column = (ColumnField) ctx.get("column");
                    List<String> values = (List<String>) ctx.get("values");
                    Object relationId = ctx.get("relationId");

                    if (rel.getOperator().equals(ExpOperator.EQUALS)) {
                        flatValue.put(relationId == null ? code : relationId.toString()
                                .concat(".").concat(code), values);

                    }
                    String rawCode = code.concat("$$")
                            .concat(rel.getOperator().getShortName());
                    flatValue.put(relationId == null ? rawCode : relationId.toString()
                            .concat(".").concat(rawCode), values);
                }
            });

            return null;
        }

        @Override
        public Void visit(ExpValue value) {
            return null;
        }

        @Override
        public Void visit(ExpBi bi) {
            return null;
        }

        @Override
        public Void visit(ExpSort expSort) {
            return null;
        }

        @Override
        public Void visit(ExpRange range) {
            return null;
        }
    }

    static private Void convertTree(ExpCondition rel
            , ExpVisitor vistor
            , EntityClassGroup group
            , Boolean isOldJoinField
            , Boolean removeIdentifier
            , Consumer<Map<String, Object>> consumer) {

        Map<String, Object> ctx = new HashMap<>();

        if (rel.isAlwaysTrue()) {
            return null;
        }

        if (rel.isAlwaysFalse()) {
            //this query should be return null;
            ctx.put("returnEmpty", true);
            consumer.accept(ctx);
            return null;
        }

        if (rel.getOperator() == ExpOperator.AND) {
            rel.getExpNodes().forEach(x -> {
                x.accept(vistor);
            });

            return null;
        }

        if (rel.getOperator() == ExpOperator.OR) {
            throw new InvalidInputsException(InvalidInputsException.getMsg("Legacy mode not support OR in ConditionRequest"));
        }

        //or else

        List<ExpNode> expNodes = rel.getExpNodes();

        Optional<ExpNode> firstField = expNodes.stream().filter(x -> x instanceof ExpField).findFirst();
        if (firstField.isPresent()) {
            ExpField field = (ExpField) firstField.get();
            String name = field.getName();
            Tuple2<String, String> nameCode = extractRelated(name);

            String fieldCode;
            if (nameCode != null) {
                fieldCode = generateRelatedFieldName(nameCode._1(), nameCode._2());
            } else {
                fieldCode = name;
            }

            Optional<ColumnField> column = group.column(fieldCode);
            if (column.isPresent()) {

                List<String> values = expNodes.stream().filter(x -> x instanceof ExpValue).map(x -> {
                    return ((ExpValue) x).getStrValue();
                }).filter(Objects::nonNull).collect(Collectors.toList());

                String code = fieldCode;
                if (isOldJoinField) {
                    if (fieldCode.startsWith(PropertyHelper.REL_PREFIX)) {
                        code = name.substring(1);
                    } else {
                        code = name;
                    }
                }

                IEntityField targetField = column.get().originField();

                if (removeIdentifier) {
                    //remove all non current field identifier flag and find the inline field
                    if (column.get().originEntityClass().id() != group.getEntityClass().id()
                            && targetField.config() != null
                            && targetField.config().isIdentifie()) {

                        if (column.get().name().startsWith(PropertyHelper.REL_PREFIX)) {
                            Optional<ColumnField> alternativeColumn = group.column(column.get().name().substring(1));
                            if (alternativeColumn.isPresent()) {
                                targetField = alternativeColumn.get();
                            }
                        }

                        IEntityField cloneField = targetField.clone();
                        cloneField.config().identifie(false);
                        targetField = cloneField;
                    }
                }

                Long relationId = findRelationId(name, column.get(), group);


                ctx.put("code", code);
                ctx.put("rel", rel);
                ctx.put("values", values);
                ctx.put("column", column.get());
                ctx.put("relation", relationId);

                consumer.accept(ctx);
            }
        }

        return null;
    }

    static Long findRelationId(String rawName, ColumnField columnField, EntityClassGroup group) {
        //TODO
        //rawName means a the part is a related part
        if (rawName.contains(".") && rawName.startsWith("_")) {
            String relName = rawName.split("\\.")[0].substring(1);
            Stream<Long> id = Stream.of(group.getEntityClass().id());
            Stream<Long> idStream = group.getFatherEntityClass().stream().map(f -> f.id());
            Set<Long> idsSet = Stream.concat(id, idStream).collect(Collectors.toSet());
            Optional<IRelation> relOp = group.getAllRelations().stream()
                    .filter(x -> {
                        long relOwnerClassId = x.getRelOwnerClassId();
                        return idsSet.contains(relOwnerClassId) && x.getName().equalsIgnoreCase(relName);
                    })
                    .findAny();

            return relOp.map(x -> x.getId()).orElse(null);
        }

        return null;
    }

    static class ConditionVisitor implements ExpVisitor<Void> {

        /**
         * ConditionsUp.Builder conditionsUpBuilder = ConditionsUp.newBuilder();
         * IEntityClassReader reader = new IEntityClassReader(entityClass);
         * Stream<Optional<FieldConditionUp>> fieldInMainStream = Optional.ofNullable(conditions.getFields())
         * .orElseGet(Collections::emptyList).stream().map(fieldCondition -> {
         * return toFieldCondition(reader, fieldCondition);
         * });
         * <p>
         * //from relation to condition
         * Stream<Optional<FieldConditionUp>> fieldInRelationStream = conditions
         * .getEntities()
         * .stream().flatMap(entityCondition -> {
         * return toFieldConditionFromRel(entityClass, entityCondition);
         * });
         * <p>
         * conditionsUpBuilder.addAllFields(Stream.concat(fieldInMainStream, fieldInRelationStream)
         * .filter(Optional::isPresent)
         * .map(Optional::get)
         * .collect(Collectors.toList()));
         * return conditionsUpBuilder.build();
         */
        List<FieldConditionUp> fieldConditions = new LinkedList<>();

        private EntityClassGroup group;

        private Boolean isOldJoinField;

        private Boolean removeIdentifier;

        private boolean returnEmpty = false;

        private String profile;

        public List<FieldConditionUp> getFieldConditions() {
            return fieldConditions;
        }

        public Boolean getReturnEmpty() {
            return returnEmpty;
        }

        public ConditionVisitor(EntityClassGroup group, Boolean isOldJoinField, Boolean removeIdentifier, String profile) {
            this.group = group;
            this.isOldJoinField = isOldJoinField;
            this.removeIdentifier = removeIdentifier;
            this.profile = profile;
        }

        @Override
        public Void visit(ExpField field) {
            return null;
        }

        @Override
        public Void visit(ExpCondition rel) {

            RelTreeHelper.convertTree(rel, this, group, isOldJoinField, removeIdentifier, ctx -> {
                if (Boolean.TRUE.equals(ctx.get("returnEmpty"))) {
                    returnEmpty = true;
                } else {
                    String code = (String) ctx.get("code");
                    ColumnField column = (ColumnField) ctx.get("column");
                    List<String> values = (List<String>) ctx.get("values");
                    Object relationId = ctx.get("relation");

                    FieldConditionUp.Builder fieldBuilder = FieldConditionUp.newBuilder();
                    FieldConditionUp fieldCondition = fieldBuilder
                            .setCode(code)
                            .setOperation(FieldConditionUp.Op.valueOf(rel.getOperator().getShortName()))
                            //TODO this should warn dummy is space?
                            .addAllValues(values)
                            .setField(toFieldUp(column))
                            .setRelationId(relationId == null ? 0 : (Long) relationId)
                            .build();

                    fieldConditions.add(fieldCondition);
                }
            });

            return null;
        }

        /**
         * new fieldUp should also know the field origin entityClassId
         *
         * @param columnField
         * @return
         */
        FieldUp toFieldUp(ColumnField columnField) {
            IEntityField originField = columnField.getOriginObject();
            FieldUp fieldUp = IEntityClassHelper.toFieldUp(originField);
            //TODO check if this can be null
            long originClassId = columnField.originEntityClass().id();
            return fieldUp.toBuilder().setOwnerClassId(originClassId).build();
        }

        @Override
        public Void visit(ExpValue value) {
            return null;
        }

        @Override
        public Void visit(ExpBi bi) {
            return null;
        }

        @Override
        public Void visit(ExpSort expSort) {
            return null;
        }

        @Override
        public Void visit(ExpRange range) {
            return null;
        }
    }

    /**
     * check if contains the external
     */
    static class ExternalVisitor implements ExpVisitor<Void> {

        private boolean containsExternal = false;

        private Set<String> externalCode;

        public ExternalVisitor(Set<String> externalCode) {
            this.externalCode = externalCode;
        }

        @Override
        public Void visit(ExpField field) {
            return null;
        }

        @Override
        public Void visit(ExpCondition rel) {

            if (rel.isAlwaysTrue() || rel.isAlwaysFalse()) {
                return null;
            }

            if (rel.getOperator() != ExpOperator.AND || rel.getOperator() != ExpOperator.OR) {
                List<ExpNode> expNodes = rel.getExpNodes();
                Optional<ExpNode> conditionKey = expNodes.stream().filter(x -> x instanceof ExpField).findFirst();
                if (conditionKey.isPresent()) {
                    ExpField field = (ExpField) conditionKey.get();
                    if (field.getName().startsWith("_") && field.getName().contains(".")) {
                        String[] split = field.getName().split("\\.");
                        if (split.length > 1) {
                            String code = split[0].substring(1);
                            if (externalCode.contains(code)) {
                                containsExternal = true;
                                return null;
                            }
                        }
                    }

                }
            }
            return null;
        }

        @Override
        public Void visit(ExpValue value) {
            return null;
        }

        @Override
        public Void visit(ExpBi bi) {
            return null;
        }

        @Override
        public Void visit(ExpSort expSort) {
            return null;
        }

        @Override
        public Void visit(ExpRange range) {
            return null;
        }

        public boolean isContainsExternal() {
            return containsExternal;
        }

        public void setContainsExternal(boolean containsExternal) {
            this.containsExternal = containsExternal;
        }
    }


    /**
     * this is compatible with old version
     *
     * @param code
     * @param fieldName
     * @return
     */
    public static String generateOldRelatedFieldName(String code, String fieldName) {
        StringBuffer sb = new StringBuffer();
        sb.append(code);
        sb.append(".");
        sb.append(fieldName);
        return sb.toString();
    }

    public static boolean containsExternal(EntityClassGroup group, ExpRel tree) {
        IEntityClass entityClass = group.getEntityClass();
        if (entityClass.getType() == 1) {
            return true;
        } else {
            Set<String> externalCodes = group.getAllRelations()
                    .stream()
                    .filter(x -> {
                        Optional<IEntityClass> relatedEntityClass = group.relatedEntityClass(x.getName());
                        if (relatedEntityClass.isPresent()) {
                            return relatedEntityClass.get().getType() == 1;
                        } else {
                            return false;
                        }
                    })
                    .map(x -> x.getName()).collect(Collectors.toSet());


            ExternalVisitor externalVisitor = new ExternalVisitor(externalCodes);
            tree.accept(externalVisitor);
            return externalVisitor.isContainsExternal();
        }
    }

    /**
     * flat the query tree to map
     *
     * @param tree
     * @param expContext
     * @return
     */
    public static Map<String, Object> flatTree(
            ExpRel tree
            , ExpContext expContext
            , String profile
    ) {
        Map<String, Object> flatValue = new HashMap<>();
        EntityClassGroup group = expContext.getSchema();
        FlattenVisitor visitor = new FlattenVisitor(
                group
                , profile
                , flatValue);

        tree.accept(visitor);

        return flatValue;
    }

    /**
     * @param tree
     * @param expContext
     * @return
     */
    public static SelectByCondition relToCondition(
            ExpRel tree
            , ExpContext expContext
            , String profile
    ) {

        SelectByCondition.Builder builder = SelectByCondition.newBuilder();

        EntityClassGroup group = expContext.getSchema();

        builder.setEntity(IEntityClassHelper.toEntityUp(group.getEntityClass()));

        /**
         * TODO
         */
        ConditionVisitor visitor = new ConditionVisitor(group
                , false
                , false
                , profile
        );
        tree.accept(visitor);

        List<FieldConditionUp> fieldConditions = visitor.getFieldConditions();

        //TODO
        Boolean isEmpty = visitor.getReturnEmpty();

        //Fast roll to empty
        if (isEmpty) {
            return null;
        }

        //condition
        builder.setConditions(ConditionsUp.newBuilder().addAllFields(fieldConditions).build());

        //sort
        if (tree.getSorts() != null) {
            ExpSort sorts = tree.getSorts();

            List<FieldSortUp> fieldSorts = sorts.getSorts().stream().map(x -> {
                Optional<ColumnField> sortField = group.column(x.getCode());
                if (sortField.isPresent()) {
                    return FieldSortUp.newBuilder()
                            .setCode(x.getCode())
                            .setField(toFieldUp(sortField.get()))
                            .setOrder(FieldSortUp.Order.valueOf(x.getSort().getShort()))
                            .build();
                }
                return null;
            }).filter(Objects::nonNull).collect(Collectors.toList());
            builder.addAllSort(fieldSorts);
        }

        //range
        if (tree.getRange() != null && !tree.getRange().isAll()) {
            //TODO check
            builder.setPageNo(tree.getRange().getIndex());
            builder.setPageSize(tree.getRange().getSize());
        }

        //projection
        if (tree.getProjects() != null) {

            List<QueryFieldsUp> queryFields = new LinkedList<>();

            for (ExpNode project : tree.getProjects()) {
                ExpField projectField = (ExpField) project;
                Tuple2<String, String> relatedFieldNameTuple = extractRelated(projectField.getName());
                String fieldCode;
                if (relatedFieldNameTuple != null) {
                    String code = relatedFieldNameTuple._1();
                    fieldCode = generateRelatedFieldName(code, relatedFieldNameTuple._2());
                } else {
                    fieldCode = projectField.getName();
                }

                List<ColumnField> columns = new LinkedList<>();
                if (fieldCode.endsWith(".*")) {

                    String code = null;
                    if (relatedFieldNameTuple != null) {
                        code = relatedFieldNameTuple._1();
                    }

                    columns.addAll(group.columns(code));
                } else {
                    Optional<ColumnField> column = group.column(fieldCode);
                    column.ifPresent(columns::add);
                    if (!column.isPresent()) {
                        log.warn("Query Field {} is not in current schema {}", fieldCode, group.getEntityClass().code());
                    }
                }

                columns.forEach(columnField -> {
                    String code;
//                    if (!compatibility.is(UNDERSCORE_JOIN_FIELD)) {
                    String name = columnField.name();
                    if (name.startsWith(PropertyHelper.REL_PREFIX)) {
                        code = name.substring(1);
                    } else {
                        code = name;
                    }
//                    } else {
//                        code = projectField.getName();
//                    }

                    QueryFieldsUp queryField = QueryFieldsUp.newBuilder()
                            .setCode(code)
                            .setId(columnField.id())
                            .build();
                    queryFields.add(queryField);
                });
            }

            builder.addAllQueryFields(queryFields);
        }

        return builder.build();
    }

    /**
     * tree to grpc tree
     *
     * @param tree
     * @return
     */
    public static SelectByTree relToTree(ExpRel tree, ExpContext context, EntityClassEngine engine, String profile) {
        List<ExpNode> projects = tree.getProjects();
        List<ExpNode> filters = tree.getFilters();
        ExpSort sorts = tree.getSorts();
        ExpRange range = tree.getRange();

        IEntityClass schema = context.getSchema().getEntityClass();

        return SelectByTree.newBuilder()
                .setProjects(toProjects(projects, context, profile))
                .setFilters(toFilters(filters, context, engine, profile))
                .setSorts(toSorts(sorts, context, profile))
                .setRange(toRange(range))
                .setEntity(IEntityClassHelper.toEntityUp(schema))
                .build();
    }

    /**
     * filter node
     * nodes is a and clause
     *
     * @param nodes
     * @return
     */
    private static Filters toFilters(List<ExpNode> nodes, ExpContext context, EntityClassEngine engine, String profile) {

        EntityClassGroup group = context.getSchema();

        List<FilterNode> filterNode = nodes.stream()
                .peek(x -> x.setExpContext(context))
                .map(x -> RelTreeHelper.toFilterNode(x, group, engine))
                .filter(Objects::nonNull)
                .collect(Collectors.toList());

        return Filters.newBuilder().addAllNodes(filterNode).build();
    }

    private static FilterNode toFilterNode(ExpNode node, EntityClassGroup group, EntityClassEngine engine) {
        if (node instanceof ExpCondition) {
            if (((ExpCondition) node).isAlwaysTrue()) {
                //omit all true
                return null;
            } else {
                return FilterNode.newBuilder()
                        .setNodeType(0)
                        .setOperator(FilterNode.Operator.valueOf(((ExpCondition) node).getOperator().getShortName()))
                        .addAllNodes(((ExpCondition) node)
                                .getExpNodes().stream()
                                .map(x -> RelTreeHelper.toFilterNode(x, group, engine))
                                .filter(Objects::nonNull)
                                .collect(Collectors.toList()))
                        .build();
            }
        } else if (node instanceof ExpValue) {
            return FilterNode.newBuilder()
                    .setNodeType(2)
                    .setPayload(((ExpValue) node).getStrValue())
                    .build();
        } else if (node instanceof ExpField) {

            ExpField expField = ((ExpField) node);

            if (StringUtils.isEmpty(expField.getName()) && expField.getId() != null) {
                //TODO this only field
                return FilterNode.newBuilder()
                        .setNodeType(1)
                        .setPayload(expField.getId().toString())
                        .build();
            } else {
                Optional<ColumnField> column = group.column(expField.getName());
                if (column.isPresent()) {

                    Long relationId = findRelationId(expField.getName(), column.get(), group);

                    return FilterNode.newBuilder()
                            .setNodeType(1)
                            .setFieldUp(toFieldUp(column.get()))
                            .setRelationId(relationId == null ? 0 : relationId)
                            .build();
                } else {
                    throw new RuntimeException("Error fieldName " + expField.getName());
                }
            }
        }

        return null;
    }

    private static Projects toProjects(List<ExpNode> projects, ExpContext context, String profile) {
        return Projects.newBuilder()
                .addAllQueryFields(projects.stream()
                        .peek(x -> x.setExpContext(context))
                        .map(x -> {
                            return RelTreeHelper.toQueryFieldUp(x, profile);
                        })
                        .filter(Objects::nonNull)
                        .collect(Collectors.toList()))
                .build();
    }

    private static Optional<ColumnField> getColumnField(ExpContext context, String code) {

        if (context != null && context.getSchema() != null) {
            EntityClassGroup schema = context.getSchema();
            return schema.column(code);
        }

        return Optional.empty();
    }

    private static QueryFieldsUp toQueryFieldUp(ExpNode expNode, String profile) {
        if (expNode instanceof ExpField) {

            ExpContext context = expNode.getExpContext();
            Optional<ColumnField> column = getColumnField(context, ((ExpField) expNode).getName());
            if (column.isPresent()) {
                return QueryFieldsUp.newBuilder()
                        .setId(column.get().id())
                        .setEntityId(column.get().originEntityClass().id())
                        .setCode(((ExpField) expNode).getName())
                        .build();
            }
        }
        return null;
    }

    private static Sorts toSorts(ExpSort sorts, ExpContext context, String profile) {
        if (sorts == null) {
            return Sorts.newBuilder().build();
        }
        return Sorts.newBuilder()
                .addAllSort(sorts.getSorts().stream()
                        .map(x -> RelTreeHelper.toSortNode(x, context, profile))
                        .filter(Objects::nonNull)
                        .collect(Collectors.toList()))
                .build();
    }

    private static SortNode toSortNode(ExpSort.FieldSort sort, ExpContext context, String profile) {
        Optional<ColumnField> column = getColumnField(context, sort.getCode());
        return column.map(columnField -> SortNode.newBuilder()
                .setCode(sort.getCode())
                .setFieldId(columnField.id())
                .setOrder(sort.getSort() == ExpSort.Sort.ASCEND ?
                        SortNode.Order.asc : SortNode.Order.desc)
                .build()).orElse(null);
    }

    private static Range toRange(ExpRange range) {

        Range.Builder builder = Range.newBuilder();

        if (range != null && range.getIndex() != null) {
            builder.setPageIndex(range.getIndex());
        }

        if (range != null && range.getSize() != null) {
            builder.setPageSize(range.getSize());
        }

        return builder.buildPartial();
    }
}
