package com.xforceplus.ultraman.adapter.core.impl;

import akka.grpc.javadsl.SingleResponseRequestBuilder;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.xforceplus.ultraman.adapter.utils.IEntityClassHelper;
import com.xforceplus.ultraman.extension.oqsengine.v1.EntityGrpcExecutor;
import com.xforceplus.ultraman.extension.oqsengine.v1.model.StickySession;
import com.xforceplus.ultraman.extension.oqsengine.v1.model.WrappedValueContext;
import com.xforceplus.ultraman.metadata.domain.record.EmptyValue;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.entity.IEntityField;
import com.xforceplus.ultraman.oqsengine.sdk.*;
import io.vavr.Tuple2;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.stream.Collectors;

import static com.xforceplus.ultraman.adapter.constant.HeaderConstants.*;

/**
 * a more raw wrapper for entityServiceClient
 * turn domain to grpc-domain
 */
public class EntityServiceExecutor implements EntityGrpcExecutor {

    private Logger logger = LoggerFactory.getLogger(EntityServiceExecutor.class);

    private EntityServiceClient entityServiceClient;

    private static String LOCK_HEADER = "lock-header";
    private static String LOCK_TOKEN = "lock-token";
    private static String SIMPLIFY = "simplify";

    private ObjectMapper mapper;
    private String STICKY_SESSION_HEADER = "sticky-session";

    public EntityServiceExecutor(EntityServiceClient entityServiceClient) {
        this.entityServiceClient = entityServiceClient;
        this.mapper = new ObjectMapper();

        SimpleModule simpleModule = new SimpleModule();
        simpleModule.addSerializer(EmptyValue.class, new JsonSerializer<EmptyValue>() {
            @Override
            public void serialize(EmptyValue emptyValue, JsonGenerator jsonGenerator, SerializerProvider serializerProvider) throws IOException {
                jsonGenerator.writeNull();
            }
        });

        this.mapper.registerModule(simpleModule);
    }

    /**
     * append header in request
     *
     * @param builder
     * @param headerName
     * @param value
     * @param <R>
     * @param <P>
     * @return
     */
    private <R, P> SingleResponseRequestBuilder<R, P> appendHeader(SingleResponseRequestBuilder<R, P> builder, String headerName, String value) {
        if (value != null) {
            return builder.addHeader(headerName, value);
        }

        return builder;
    }

    /**
     * prepare the entityClass
     *
     * @param in
     * @return
     */
    @Override
    public CompletionStage<OperationResult> prepare(EntityUp in) {
        try {
            return entityServiceClient.prepare(in);
        } catch (Throwable throwable) {
            return exception(throwable);
        }
    }

    /**
     * prepare the appId
     *
     * @return
     */
    @Override
    public CompletionStage<OperationResult> prepare(String appId, String env) {
        try {

            SingleResponseRequestBuilder<EntityUp, OperationResult> prepare = entityServiceClient.prepare();

            prepare = prepare.addHeader("appid", appId).addHeader("env", env);

            return prepare.invoke(EntityUp.newBuilder().build());

        } catch (Throwable throwable) {
            return exception(throwable);
        }
    }


    /**
     * select One
     *
     * @param transId
     * @return
     */
    @Override
    public CompletionStage<OperationResult> selectOne(
              EntityUp entityUp
            , StickySession stickySession
            , String transId
            , String profile) {

        SingleResponseRequestBuilder<EntityUp, OperationResult> queryResultBuilder
                = entityServiceClient.selectOne();

        if (transId != null) {
            logger.info("Query with Transaction id:{} ", transId);
        } else {
            logger.debug("Query without Transaction");
        }

        queryResultBuilder = appendHeader(queryResultBuilder, TRANSACTION_HEADER, transId);
        queryResultBuilder = appendHeader(queryResultBuilder, STICKY_SESSION_HEADER, Optional.ofNullable(stickySession)
                .map(StickySession::getSticky).orElse("dummy"));
        queryResultBuilder = appendHeader(queryResultBuilder, PROFILE_HEADER, profile);

        try {
            return queryResultBuilder.invoke(entityUp);
        } catch (Throwable throwable) {
            return exception(throwable);
        }
    }

    /**
     * remove multi
     * @param isForce
     * @param transId
     * @param profile
     * @param userDisplayName
     * @param userName
     * @return
     */
    
    @Override
    public CompletionStage<OperationResult> removeMulti(List<EntityUp> entityUpList
            ,  boolean isForce
            ,  StickySession stickySession
            ,  String transId
            ,  String profile
            ,  String userDisplayName
            ,  String userName) {

        SingleResponseRequestBuilder<EntityMultiUp, OperationResult> removeBuilder = entityServiceClient.removeMulti();

        if (transId != null) {
            logger.info("Delete with Transaction id:{} ", transId);
        } else {
            logger.debug("Delete without Transaction, but will create auto");
        }

        removeBuilder = appendHeader(removeBuilder, TRANSACTION_HEADER, transId);
        removeBuilder = appendHeader(removeBuilder, STICKY_SESSION_HEADER, Optional.ofNullable(stickySession)
                .map(StickySession::getSticky).orElse("dummy"));
        removeBuilder = appendHeader(removeBuilder, PROFILE_HEADER, profile);
        removeBuilder = appendHeader(removeBuilder, DISPLAY_NAME_HEADER, userDisplayName);
        removeBuilder = appendHeader(removeBuilder, USERNAME_HEADER, userName);
        removeBuilder = appendHeader(removeBuilder, FORCE_HEADER, isForce ? "true" : null);

        try {
            return removeBuilder.invoke(toMulti(entityUpList));
        } catch (Throwable throwable) {
            return exception(throwable);
        }
    }

    /**
     * remove entity
     *
     * @param isForce
     * @param transId
     * @param userDisplayName
     * @param userName
     * @return
     */
    
    @Override
    public CompletionStage<OperationResult> remove(EntityUp entityUp
            ,  boolean isForce
            ,  StickySession stickySession
            ,  String transId
            ,  String profile
            ,  String userDisplayName
            ,  String userName) {

        SingleResponseRequestBuilder<EntityUp, OperationResult> removeBuilder = entityServiceClient.remove();

        if (transId != null) {
            logger.info("Delete with Transaction id:{} ", transId);
        } else {
            logger.debug("Delete without Transaction");
        }

        if(stickySession != null) {
            removeBuilder = appendHeader(removeBuilder, STICKY_SESSION_HEADER, stickySession.getSticky());
        }

        removeBuilder = appendHeader(removeBuilder, TRANSACTION_HEADER, transId);
        removeBuilder = appendHeader(removeBuilder, PROFILE_HEADER, profile);
        removeBuilder = appendHeader(removeBuilder, DISPLAY_NAME_HEADER, userDisplayName);
        removeBuilder = appendHeader(removeBuilder, USERNAME_HEADER, userName);
        removeBuilder = appendHeader(removeBuilder, FORCE_HEADER, isForce ? "true" : null);

        try {
            return removeBuilder.invoke(entityUp);
        } catch (Throwable throwable) {
            return exception(throwable);
        }
    }


    /**
     * combine multi entityUp to entityMultiup
     * @param entityUps
     * @return
     */
    private EntityMultiUp toMulti(List<EntityUp> entityUps){
        if(entityUps.isEmpty()) {
            throw new RuntimeException("List Cannot be empty");
        }

        EntityUp sample = entityUps.get(0);


        List<ValueListUp> valueListUp = entityUps.stream().map(x -> {
            ValueListUp valueList = ValueListUp.newBuilder()
                    .setObjId(x.getObjId())
                    .addAllValues(x.getValuesList())
                    .build();
            return valueList;
        }).collect(Collectors.toList());

        EntityMultiUp multiUp = EntityMultiUp.newBuilder()
                .setCode(sample.getCode())
                .setId(sample.getId())
                .addAllEntityClasses(sample.getEntityClassesList())
                .addAllFields(sample.getFieldsList())
                .addAllRelation(sample.getRelationList())
                .addAllValues(valueListUp)
                .build();

        return multiUp;
    }

    /**
     * add multi record
     * @param entityClass
     * @param transId
     * @param profile
     * @return
     */
    
    @Override
    public CompletionStage<OperationResult> buildMulti(
            IEntityClass entityClass
            , List<WrappedValueContext> valueContext
            ,  StickySession stickySession
            ,  String transId
            ,  String profile) {

        SingleResponseRequestBuilder<EntityMultiUp, OperationResult> buildBuilder = entityServiceClient.buildMulti();

        if (transId != null) {
            logger.info("Create with Transaction id:{} ", transId);
        } else {
            logger.debug("Create without Transaction");
        }

        if(stickySession != null) {
            buildBuilder = appendHeader(buildBuilder, STICKY_SESSION_HEADER, stickySession.getSticky());
        }

        buildBuilder = appendHeader(buildBuilder, TRANSACTION_HEADER, transId);
        buildBuilder = appendHeader(buildBuilder, PROFILE_HEADER, profile);

        //valueTuple --> List<ValueUp>
        //TODO @Important will filter object with null
        //TODO value will turn to String default if you have custom behavior should do in the
        // @see com.xforceplus.ultraman.oqsengine.sdk.service.operation.FieldOperationHandler

        List<EntityUp> entityUps = valueContext.stream().map(x -> {
            Long id = x.getId();
            List<Tuple2<IEntityField, Object>> valueTuple = x.getValueTuple();
            EntityUp entityUp = IEntityClassHelper.toEntityUp(entityClass, id, valueTuple);

            List<ValueUp> contextValueList = x.getContextField().stream().map(cf -> {
                String context = "";
                try {
                    context = mapper.writeValueAsString(cf._2());
                } catch (JsonProcessingException e) {
                    logger.warn("cannot transform context to value");
                }

                IEntityField field = cf._1();
                return ValueUp.newBuilder()
                        .setFieldType(field.type().getType())
                        .setFieldId(field.id())
                        .setContextStr(context)
                        .setName(field.name())
                        .build();

            }).collect(Collectors.toList());
            entityUp = entityUp.toBuilder().addAllValues(contextValueList).build();
            return entityUp;
        }).collect(Collectors.toList());

        try {
            return buildBuilder.invoke(toMulti(entityUps));
        } catch (Throwable throwable) {
            return exception(throwable);
        }
    }

    /**
     * @param entityClass
     * @param transId
     * @return
     */
    
    @Override
    public CompletionStage<OperationResult> build(IEntityClass entityClass
            , List<Tuple2<IEntityField, Object>> valueTuple
            , StickySession stickySession
            ,  String transId
            ,  String profile
            ,  Long id
            , List<Tuple2<IEntityField, Map<String, Object>>> contextField) {

        SingleResponseRequestBuilder<EntityUp, OperationResult> buildBuilder = entityServiceClient.build();

        if (transId != null) {
            logger.info("Create with Transaction id:{} ", transId);
        } else {
            logger.debug("Create without Transaction");
        }

        if (stickySession != null) {
            logger.info("Create with stick session id:{} ", stickySession.getSticky());
            buildBuilder = appendHeader(buildBuilder, STICKY_SESSION_HEADER, stickySession.getSticky());
        }

        buildBuilder = appendHeader(buildBuilder, TRANSACTION_HEADER, transId);
        buildBuilder = appendHeader(buildBuilder, PROFILE_HEADER, profile);

        //valueTuple --> List<ValueUp>
        //TODO @Important will filter object with null
        //TODO value will turn to String default if you have custom behavior should do in the
        // @see com.xforceplus.ultraman.oqsengine.sdk.service.operation.FieldOperationHandler
        EntityUp entityUp = IEntityClassHelper.toEntityUp(entityClass, id, valueTuple);

        /**
         * add all context field
         */
        List<ValueUp> contextValueList = contextField.stream().map(x -> {
            String context = "";
            try {
                context = mapper.writeValueAsString(x._2());
            } catch (JsonProcessingException e) {
                logger.warn("cannot transform context to value");
            }

            IEntityField field = x._1();
            return ValueUp.newBuilder()
                    .setFieldType(field.type().getType())
                    .setFieldId(field.id())
                    .setContextStr(context)
                    .setName(field.name())
                    .build();

        }).collect(Collectors.toList());

        entityUp = entityUp.toBuilder().addAllValues(contextValueList).build();

        try {
            return buildBuilder.invoke(entityUp);
        } catch (Throwable throwable) {
            return exception(throwable);
        }
    }

    /**
     * update multi
     * @param entityClass
     * @param valueContext
     * @param transId
     * @param profile
     * @return
     */
    
    @Override
    public CompletionStage<OperationResult> updateMulti(
              IEntityClass entityClass
            , List<WrappedValueContext> valueContext
            , StickySession stickySession, String transId
            , boolean isReplace
            , String profile) {

        SingleResponseRequestBuilder<EntityMultiUp, OperationResult> updateBuilder = entityServiceClient.replaceMulti();

        if (transId != null) {
            logger.info("{} with Transaction id:{} ", isReplace ? "ReplaceById" : "UpdateById", transId);
        } else {
            logger.debug("{} without Transaction", isReplace ? "ReplaceById" : "UpdateById");
        }

        if (stickySession != null) {
            logger.info("{} with Transaction id:{} ", isReplace ? "ReplaceById" : "UpdateById", transId);
            updateBuilder = appendHeader(updateBuilder, STICKY_SESSION_HEADER, stickySession.getSticky());
        }

        updateBuilder = appendHeader(updateBuilder, TRANSACTION_HEADER, transId);
        updateBuilder = appendHeader(updateBuilder, PROFILE_HEADER, profile);
        updateBuilder = appendHeader(updateBuilder, UPDATE_MODE_HEADER, isReplace ? "replace" : null);

        //valueTuple --> List<ValueUp>
        //TODO @Important will filter object with null
        //TODO value will turn to String default if you have custom behavior should do in the
        // @see com.xforceplus.ultraman.oqsengine.sdk.service.operation.FieldOperationHandler

        List<EntityUp> entityUps = valueContext.stream().map(x -> {
            Long id = x.getId();
            List<Tuple2<IEntityField, Object>> valueTuple = x.getValueTuple();
            EntityUp entityUp = IEntityClassHelper.toEntityUp(entityClass, id, valueTuple);

            List<ValueUp> contextValueList = x.getContextField().stream().map(cf -> {
                String context = "";
                try {
                    context = mapper.writeValueAsString(cf._2());
                } catch (JsonProcessingException e) {
                    logger.warn("cannot transform context to value");
                }

                IEntityField field = cf._1();
                return ValueUp.newBuilder()
                        .setFieldType(field.type().getType())
                        .setFieldId(field.id())
                        .setContextStr(context)
                        .setName(field.name())
                        .build();

            }).collect(Collectors.toList());
            entityUp = entityUp.toBuilder().addAllValues(contextValueList).build();
            return entityUp;
        }).collect(Collectors.toList());

        try {
            return updateBuilder.invoke(toMulti(entityUps));
        } catch (Throwable throwable) {
            return exception(throwable);
        }
    }



    /**
     * update by id
     *
     * @param transId
     * @param isReplace is replace will replace all the attribution or only update part which the value is not null
     * @return
     */
    
    @Override
    public CompletionStage<OperationResult> updateById(
            IEntityClass entityClass
            ,  Long id
            , List<Tuple2<IEntityField, Object>> valueTuple
            , List<Tuple2<IEntityField, Map<String, Object>>> contextField
            , StickySession stickySession
            ,  String transId, String profile
            ,  boolean isReplace
            , Integer version
    ) {

        SingleResponseRequestBuilder<EntityUp, OperationResult> updateBuilder = entityServiceClient.replace();

        if (transId != null) {
            logger.info("{} with Transaction id:{} ", isReplace ? "ReplaceById" : "UpdateById", transId);
        } else {
            logger.debug("{} without Transaction", isReplace ? "ReplaceById" : "UpdateById");
        }

        if (stickySession != null) {
            logger.info("{} with sticky-session id:{} ", isReplace ? "ReplaceById" : "UpdateById", stickySession.getSticky());
            updateBuilder = appendHeader(updateBuilder, STICKY_SESSION_HEADER, stickySession.getSticky());
        }

        updateBuilder = appendHeader(updateBuilder, TRANSACTION_HEADER, transId);
        updateBuilder = appendHeader(updateBuilder, PROFILE_HEADER, profile);
        updateBuilder = appendHeader(updateBuilder, UPDATE_MODE_HEADER, isReplace ? "replace" : null);

        EntityUp entityUp = IEntityClassHelper.toEntityUp(entityClass, id, valueTuple);

        if (version != null) {
            entityUp.toBuilder().setVersion(version);
        }

        List<ValueUp> contextValueList = contextField.stream().map(x -> {
            String context = "";
            try {
                context = mapper.writeValueAsString(x._2());
            } catch (JsonProcessingException e) {
                logger.warn("cannot transform context to value");
            }

            IEntityField field = x._1();
            return ValueUp.newBuilder()
                    .setFieldType(field.type().getType())
                    .setFieldId(field.id())
                    .setContextStr(context)
                    .setName(field.name())
                    .build();

        }).collect(Collectors.toList());

        entityUp = entityUp.toBuilder().addAllValues(contextValueList).build();

        try {
            return updateBuilder
                    .invoke(entityUp);
        } catch (Throwable throwable) {
            return exception(throwable);
        }
    }

    /**
     * TODO lock is not ok
     * update by condition
     *
     * @param entityClass
     * @param selectByCondition
     * @param valueTuple
     * @param transId
     * @param isReplace
     * @return
     */
    @Override
    public CompletionStage<OperationResult> updateByCondition(
            IEntityClass entityClass
            , SelectByCondition selectByCondition
            , List<Tuple2<IEntityField, Object>> valueTuple
            , StickySession stickySession
            , String transId, String profile, boolean isReplace
    ) {
        SingleResponseRequestBuilder<SelectByCondition, OperationResult> updateBuilder = entityServiceClient.replaceByCondition();
        if (transId != null) {
            logger.info("{} with Transaction id:{} ", isReplace ? "ReplaceById" : "UpdateById", transId);
        } else {
            logger.debug("{} without Transaction", isReplace ? "ReplaceById" : "UpdateById");
        }

        if (stickySession != null) {
            logger.info("{} with sticky-session id:{} ", isReplace ? "ReplaceById" : "UpdateById", stickySession);
            updateBuilder = appendHeader(updateBuilder, STICKY_SESSION_HEADER, stickySession.getSticky());
        }

        updateBuilder = appendHeader(updateBuilder, TRANSACTION_HEADER, transId);
        updateBuilder = appendHeader(updateBuilder, PROFILE_HEADER, profile);
        updateBuilder = appendHeader(updateBuilder, UPDATE_MODE_HEADER, isReplace ? "replace" : null);

//        if(sync.getLockInfo() != null && sync.getLockInfo().get() != null){
//            updateBuilder = appendHeader(updateBuilder, LOCK_HEADER, sync.getLockInfo().get().getUuid());
//            updateBuilder = appendHeader(updateBuilder, LOCK_TOKEN, sync.getLockInfo().get().getToken());
//        }

        EntityUp entityUp = IEntityClassHelper.toEntityUp(entityClass, 0L, valueTuple);

        selectByCondition = selectByCondition.toBuilder().setEntity(entityUp).build();

        try {
            return updateBuilder.invoke(selectByCondition);
        } catch (Throwable throwable) {
            return exception(throwable);
        }
    }

    /**
     * select by Condition
     * IEntityClass entityClass, List<Long> ids, ConditionQueryRequest condition
     */
    
    @Override
    public CompletionStage<OperationResult> selectByConditions( SelectByCondition selectByCondition
            ,  SelectByTree tree
            , StickySession stickySession
            ,  String transId,  String profile
            ,  boolean isSimplify) {

        SingleResponseRequestBuilder<SelectByCondition, OperationResult> searchBuilder = entityServiceClient.selectByConditions();

        if (transId != null) {
            logger.info("SelectByConditions with Transaction id:{} ", transId);
        } else {
            logger.debug("SelectByConditions without Transaction");
        }

        if (stickySession != null) {
            logger.info("SelectByConditions with sticky session id:{} ", stickySession);
            searchBuilder = appendHeader(searchBuilder, STICKY_SESSION_HEADER, stickySession.getSticky());
        }

        searchBuilder = appendHeader(searchBuilder, TRANSACTION_HEADER, transId);
        searchBuilder = appendHeader(searchBuilder, PROFILE_HEADER, profile);
        searchBuilder = appendHeader(searchBuilder, SIMPLIFY, Boolean.toString(isSimplify));

        if (tree != null) {
            selectByCondition = selectByCondition.toBuilder().setTree(tree).build();
        }

        /**
         * condition
         */
        try {
            return searchBuilder.invoke(selectByCondition);
        } catch (Throwable throwable) {
            return exception(throwable);
        }
    }

    private CompletableFuture<OperationResult> exception(Throwable throwable) {
        return CompletableFuture.completedFuture(OperationResult.newBuilder()
                .setCode(OperationResult.Code.EXCEPTION)
                .setMessage(throwable.getMessage())
                .build());
    }

    /**
     * select By Tree
     *
     * @return
     */
    @Override
    public CompletionStage<OperationResult> selectByTree(SelectByTree selectByTree, StickySession stickySession, String transId, String profile, boolean isSimplify) {

        SingleResponseRequestBuilder<SelectByTree, OperationResult> requestBuilder = entityServiceClient.selectByTreeFilter();

        if (transId != null) {
            logger.info("findRecordsByCondition with Transaction id:{} ", transId);
        } else {
            logger.debug("SelectByConditions without Transaction");
        }

        if (stickySession != null) {
            logger.info("findRecordsByCondition with sticky session id:{} ", stickySession);
            requestBuilder = appendHeader(requestBuilder, STICKY_SESSION_HEADER, stickySession.getSticky());
        }

        requestBuilder = appendHeader(requestBuilder, TRANSACTION_HEADER, transId);
        requestBuilder = appendHeader(requestBuilder, PROFILE_HEADER, profile);
        requestBuilder = appendHeader(requestBuilder, SIMPLIFY, Boolean.toString(isSimplify));

        /**
         * condition
         * TODO
         */
        try {
            return requestBuilder.invoke(selectByTree);
        } catch (Throwable throwable) {
            return exception(throwable);
        }
    }

    @Override
    public CompletionStage<ChangelogResponseList> getChangelogList(ChangelogRequest changelogRequest) {
        SingleResponseRequestBuilder<ChangelogRequest, ChangelogResponseList> builder = entityServiceClient.changelogList();
        try {
            return builder.invoke(changelogRequest);
        } catch (Throwable throwable) {
            CompletableFuture completableFuture = new CompletableFuture();
            completableFuture.completeExceptionally(throwable);
            return completableFuture;
        }
    }

    @Override
    public CompletionStage<OperationResult> replay(ReplayRequest replayRequest) {
        SingleResponseRequestBuilder<ReplayRequest, OperationResult> builder = entityServiceClient.replay();

        try {
            return builder.invoke(replayRequest);
        } catch (Throwable throwable) {
            return exception(throwable);
        }
    }

    @Override
    public CompletionStage<ChangelogCountResponse> getChangelogCount(ChangelogCountRequest request) {
        SingleResponseRequestBuilder<ChangelogCountRequest, ChangelogCountResponse> builder = entityServiceClient.changelogCount();

        try {
            return builder.invoke(request);
        } catch (Throwable throwable) {
            CompletableFuture completableFuture = new CompletableFuture();
            completableFuture.completeExceptionally(throwable);
            return completableFuture;
        }
    }

    @Override
    public CompletionStage<OperationResult> expand(TransRequest trans) {
        try {
            return entityServiceClient.expand(trans);
        } catch (Throwable throwable) {
            CompletableFuture completableFuture = new CompletableFuture();
            completableFuture.completeExceptionally(throwable);
            return completableFuture;
        }
    }
}
