package com.xforceplus.ultraman.adapter.remote.service.impl;

import com.xforceplus.ultraman.adapter.constant.ERROR;
import com.xforceplus.ultraman.adapter.remote.service.CurrentUpdateService;
import com.xforceplus.ultraman.metadata.grpc.ModuleUpResult;
import com.xforceplus.ultraman.metadata.sync.grpc.event.MetadataModulePreparedEvent;
import com.xforceplus.ultraman.metadata.sync.grpc.utils.VersionUtils;
import com.xforceplus.ultraman.oqsengine.sdk.EntityService;
import com.xforceplus.ultraman.oqsengine.sdk.EntityUp;
import com.xforceplus.ultraman.oqsengine.sdk.OperationResult;
import com.xforceplus.ultraman.sdk.infra.base.AuthConfig;
import com.xforceplus.ultraman.sdk.infra.logging.LoggingPattern;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationEventPublisher;

import java.util.*;
import java.util.concurrent.*;

import static com.xforceplus.ultraman.sdk.infra.logging.LoggingUtils.logErrorPattern;

/**
 * Current update Service for period retry
 */
@Slf4j
public class CurrentUpdateServiceImpl implements CurrentUpdateService {

    private EntityService entityService;

    private ApplicationEventPublisher applicationPublisher;

    private AuthConfig config;

    private Logger logger = LoggerFactory.getLogger(CurrentUpdateService.class);

    Map<Tuple2<String, Integer>, ModuleUpResult> mapping = new ConcurrentHashMap<>();

    Map<Tuple2<String, Integer>, Tuple2<ModuleUpResult, Integer>> fatalList = new ConcurrentHashMap<>();

    Map<Tuple2<String, Integer>, List<Tuple2<EntityUp, OperationResult>>> lists = new ConcurrentHashMap<>();

    ScheduledExecutorService scheduledExecutorService = Executors.newScheduledThreadPool(1);

    public CurrentUpdateServiceImpl(EntityService entityService, AuthConfig config, ApplicationEventPublisher applicationPublisher) {
        
        this.entityService = entityService;
        this.applicationPublisher = applicationPublisher;
        this.config = config;

        scheduledExecutorService.scheduleAtFixedRate(() -> schedule(), 2L, 5L, TimeUnit.SECONDS);
    }

    synchronized void schedule() {
        Set<Map.Entry<Tuple2<String, Integer>, ModuleUpResult>> entries = this.mapping.entrySet();
        entries.stream().forEach(x -> retry(x.getValue()));
    }

    @Override
    public void saveUpdateModule(ModuleUpResult result) {
        mapping.put(getTuple(result), result);
    }

    @Override
    public void recordResults(ModuleUpResult result, List<Tuple2<EntityUp, OperationResult>> reps) {
        Tuple2<String, Integer> key = getTuple(result);
        lists.put(key, reps);
        mapping.put(key, result);
    }

    /**
     * on clean throw
     *private
     * @param result
     */
    private synchronized void clean(ModuleUpResult result, Tuple2<EntityUp, OperationResult> retryTarget) {

        Tuple2<String, Integer> key = getTuple(result);

        List<Tuple2<EntityUp, OperationResult>> tuple2s = this.lists.get(key);

        List<Tuple2<EntityUp, OperationResult>> mutableList = new ArrayList<>(tuple2s);

        mutableList.remove(retryTarget);

        if (mutableList.size() == 0 && this.mapping.containsKey(key)) {
            logger.info("module {} {} is clean", result.getCode(), result.getVersion());
            this.mapping.remove(key);
            applicationPublisher.publishEvent(new MetadataModulePreparedEvent(result, config.getTenant(), config.getAppId()));
        } else {
            logger.warn("module {} {} Left bo {}", result.getCode(), result.getVersion(), mutableList.size());
            this.lists.put(key, mutableList);
        }
    }

    private synchronized void fatal(ModuleUpResult result, int errorCode) {
        Tuple2<String, Integer> key = getTuple(result);
        if (this.mapping.containsKey(key)) {
            this.mapping.remove(key);
            this.lists.remove(key);
            this.fatalList.put(key, Tuple.of(result, errorCode));
        }
    }

    @Override
    public void retry(ModuleUpResult result) {
        Tuple2<String, Integer> key = getTuple(result);
        List<Tuple2<EntityUp, OperationResult>> tuple2s = lists.get(key);
        tuple2s.forEach(x -> {
            CompletionStage<OperationResult> prepare = entityService.prepare(x._1);

            //TODO threadpools
            prepare.toCompletableFuture().thenAcceptAsync(newResult -> {
                if (newResult.getCode() == OperationResult.Code.OK) {
                    this.clean(result, x);
                } else if (newResult.getAffectedRow() == ERROR.REACH_MAX_SLOT.ordinal() || newResult.getAffectedRow() == ERROR.VALIDATION_ERROR.ordinal()) {
                    this.fatal(result, newResult.getAffectedRow());
                }
            }).exceptionally(throwable -> {
                logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR
                        , throwable);
                return null;
            });
        });
    }

    @Override
    public Map<String, String> showFatalModule() {

        Map<String, String> map = new HashMap<>();
        fatalList.entrySet().stream().forEach(entry -> {
            ModuleUpResult moduleUpResult = entry.getValue()._1;
            map.put(moduleUpResult.getCode(), moduleUpResult.getVersion() + ":" + entry.getValue()._2());
        });

        return map;
    }

    @Override
    public Map<String, Object> showCurrentModule() {
        Map<String, Object> map = new HashMap<>();
        mapping.entrySet().stream().forEach(entry -> {
            Tuple2<String, Integer> key = entry.getKey();
            ModuleUpResult value = entry.getValue();
            List<Tuple2<EntityUp, OperationResult>> tuple2s = lists.get(key);

            map.put(value.getCode(), tuple2s.stream().map(x -> x._2().getCode()));
        });
        return map;
    }

    private Tuple2<String, Integer> getTuple(ModuleUpResult result) {
        String code = result.getCode();
        Integer versionNum = VersionUtils.toVersionInt(result.getVersion());
        return Tuple.of(code, versionNum);
    }
}
