package com.xforceplus.ultraman.app.jcunilever.metadata.validator;

import com.xforceplus.ultraman.app.jcunilever.metadata.dict.*;
import com.xforceplus.ultraman.app.jcunilever.metadata.validator.annotation.*;

import org.hibernate.validator.HibernateValidator;
import org.hibernate.validator.HibernateValidatorConfiguration;
import org.hibernate.validator.cfg.ConstraintMapping;
import org.hibernate.validator.cfg.GenericConstraintDef;
import org.hibernate.validator.cfg.defs.*;

import javax.validation.ConstraintViolation;
import javax.validation.Validation;
import javax.validation.Validator;
import javax.validation.executable.ExecutableValidator;
import javax.validation.metadata.BeanDescriptor;
import java.util.Arrays;
import java.util.Collection;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * <p>
 * UltramanMetadataValidator
 * </p>
 *
 * @author ultraman
 * @since 2023-03-06
 */
public class EntityMetaValidator implements Validator {

    private Validator validator;

    public EntityMetaValidator() {
        HibernateValidatorConfiguration configuration = Validation
                .byProvider(HibernateValidator.class)
                .configure();

        ConstraintMapping constraintMapping = configuration.createConstraintMapping();

        buildConstraintMapping(constraintMapping);

        this.validator = configuration.addMapping(constraintMapping)
                .buildValidatorFactory()
                .getValidator();
    }

    public <T> Set<ConstraintViolation<T>> validateProperties(T object, String ...propertyNames) {
        return Arrays.asList(propertyNames).stream()
                .map(propertyName -> validator.validateProperty(object, propertyName))
                .flatMap(Collection::stream)
                .collect(Collectors.toSet());
    }

    @Override
    public <T> Set<ConstraintViolation<T>> validate(T object, Class<?>... groups) {
        return validator.validate(object, groups);
    }

    @Override
    public <T> Set<ConstraintViolation<T>> validateProperty(T object, String propertyName, Class<?>... groups) {
        return validator.validateProperty(object, propertyName, groups);
    }

    @Override
    public <T> Set<ConstraintViolation<T>> validateValue(Class<T> beanType, String propertyName, Object value, Class<?>... groups) {
        return validator.validateValue(beanType, propertyName, value, groups);
    }

    @Override
    public BeanDescriptor getConstraintsForClass(Class<?> clazz) {
        return validator.getConstraintsForClass(clazz);
    }

    @Override
    public <T> T unwrap(Class<T> type) {
        return validator.unwrap(type);
    }

    @Override
    public ExecutableValidator forExecutables() {
        return validator.forExecutables();
    }

    private void buildConstraintMapping(ConstraintMapping constraintMapping) {

        // oqsengine_sdk_om_audit_log
        constraintMapping
            .type(com.xforceplus.ultraman.app.jcunilever.metadata.entity.OqsengineSdkOmAuditLog.class)
            .field("operatorId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("operatorCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("operatorName")
            .constraint(new LengthDef().min(0).max(200))
            .field("operateType")
            .constraint(new LengthDef().min(0).max(200))
            .field("operateTime")
            .field("appId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("appCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("boId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("boCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("boName")
            .constraint(new LengthDef().min(0).max(200))
            .field("entityId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("requestData")
            .constraint(new LengthDef().min(0).max(5000))
            .field("responseData")
            .constraint(new LengthDef().min(0).max(5000))
            .field("remark")
            .constraint(new LengthDef().min(0).max(500))
            .field("id")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("createTime")
            .field("updateTime")
            .field("createUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("updateUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("createUserName")
            .constraint(new LengthDef().min(0).max(100))
            .field("updateUserName")
            .constraint(new LengthDef().min(0).max(100))
            .field("deleteFlag")
            .constraint(new LengthDef().min(0).max(2))
            ;
        // order_detail
        constraintMapping
            .type(com.xforceplus.ultraman.app.jcunilever.metadata.entity.OrderDetail.class)
            .field("orderNo")
            .constraint(new LengthDef().min(0).max(50))
            .field("goodsId")
            .constraint(new LengthDef().min(0).max(50))
            .field("goodsName")
            .constraint(new LengthDef().min(0).max(200))
            .field("skuCode")
            .constraint(new LengthDef().min(0).max(50))
            .field("id")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("createTime")
            .field("updateTime")
            .field("createUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("updateUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("createUserName")
            .constraint(new LengthDef().min(0).max(100))
            .field("updateUserName")
            .constraint(new LengthDef().min(0).max(100))
            .field("deleteFlag")
            .constraint(new LengthDef().min(0).max(2))
            .field("goodsQuantity")
            .constraint(new DigitsDef().integer(12).fraction(6))
            .field("goodsUnitPrice")
            .constraint(new DigitsDef().integer(12).fraction(15))
            .field("goodsAmount")
            .constraint(new DigitsDef().integer(16).fraction(2))
            .field("mainOrderId")
            .constraint(new LengthDef().min(0).max(50))
            ;
        // settlement_pool
        constraintMapping
            .type(com.xforceplus.ultraman.app.jcunilever.metadata.entity.SettlementPool.class)
            .field("requestNo")
            .constraint(new LengthDef().min(0).max(50))
            .field("shopOrderId")
            .constraint(new LengthDef().min(0).max(50))
            .field("orderId")
            .constraint(new LengthDef().min(0).max(50))
            .field("settleAmount")
            .constraint(new DigitsDef().integer(14).fraction(6))
            .field("payTypeDesc")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", PayTypeDesc.class))
            .field("tradeType")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", TradeType.class))
            .field("isContainsRefundBeforeSettle")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", IsContainsRefundBeforeSettle.class))
            .field("productId")
            .constraint(new LengthDef().min(0).max(50))
            .field("goodsCount")
            .constraint(new DigitsDef().integer(14).fraction(6))
            .field("flowTypeDesc")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", FlowTypeDesc.class))
            .field("orderType")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", OrderType.class))
            .field("totalAmount")
            .constraint(new DigitsDef().integer(14).fraction(6))
            .field("totalGoodsAmount")
            .constraint(new DigitsDef().integer(14).fraction(6))
            .field("id")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("createTime")
            .field("updateTime")
            .field("createUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("updateUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("createUserName")
            .constraint(new LengthDef().min(0).max(100))
            .field("updateUserName")
            .constraint(new LengthDef().min(0).max(100))
            .field("deleteFlag")
            .constraint(new LengthDef().min(0).max(2))
            .field("postAmount")
            .constraint(new DigitsDef().integer(14).fraction(6))
            .field("shopCoupon")
            .constraint(new DigitsDef().integer(14).fraction(6))
            .field("refundBeforeSettle")
            .constraint(new DigitsDef().integer(14).fraction(6))
            .field("platformCoupon")
            .constraint(new DigitsDef().integer(14).fraction(6))
            .field("authorCoupon")
            .constraint(new DigitsDef().integer(14).fraction(6))
            .field("ztPayPromotion")
            .constraint(new DigitsDef().integer(14).fraction(6))
            .field("zrPayPromotion")
            .constraint(new DigitsDef().integer(14).fraction(6))
            .field("realPayAmount")
            .constraint(new DigitsDef().integer(14).fraction(6))
            .field("totalIncome")
            .constraint(new DigitsDef().integer(14).fraction(6))
            .field("platformServiceFee")
            .constraint(new DigitsDef().integer(14).fraction(6))
            .field("commission")
            .constraint(new DigitsDef().integer(14).fraction(6))
            .field("goodLearnChannelFee")
            .constraint(new DigitsDef().integer(14).fraction(6))
            .field("colonelServiceFee")
            .constraint(new DigitsDef().integer(14).fraction(6))
            .field("channelPromotionFee")
            .constraint(new DigitsDef().integer(14).fraction(6))
            .field("otherSharingAmount")
            .constraint(new DigitsDef().integer(14).fraction(6))
            .field("remark")
            .constraint(new LengthDef().min(0).max(500))
            .field("settleTime")
            .field("orderTime")
            .field("dataOrig")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", DataOrig.class))
            .field("totalAutcome")
            .constraint(new DigitsDef().integer(14).fraction(6))
            .field("settlePoolNo")
            .constraint(new LengthDef().min(0).max(200))
            ;
        // order_pool
        constraintMapping
            .type(com.xforceplus.ultraman.app.jcunilever.metadata.entity.OrderPool.class)
            .field("applyNo")
            .constraint(new LengthDef().min(0).max(50))
            .field("errorInfo")
            .constraint(new LengthDef().min(0).max(1000))
            .field("storeId")
            .constraint(new LengthDef().min(0).max(50))
            .field("mainOrderNo")
            .constraint(new LengthDef().min(0).max(50))
            .field("businessType")
            .constraint(new LengthDef().min(0).max(20))
            .field("orderType")
            .constraint(new LengthDef().min(0).max(20))
            .field("purchaserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("purchaserTaxNo")
            .constraint(new LengthDef().min(0).max(30))
            .field("purchaserAddress")
            .constraint(new LengthDef().min(0).max(200))
            .field("purchaserTel")
            .constraint(new LengthDef().min(0).max(30))
            .field("purchaserBankName")
            .constraint(new LengthDef().min(0).max(200))
            .field("purchaserBankAccount")
            .constraint(new LengthDef().min(0).max(200))
            .field("applyType")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", ApplyType.class))
            .field("applyTime")
            .field("settlementStatus")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", SettlementStatus.class))
            .field("makeInvoiceStatus")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", MakeInvoiceStatus.class))
            .field("submitInvoiceStatus")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", SubmitInvoiceStatus.class))
            .field("invoiceType")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", InvoiceType.class))
            .field("headerType")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", HeaderType.class))
            .field("orderTotal")
            .constraint(new DigitsDef().integer(16).fraction(2))
            .field("goodsTotal")
            .constraint(new DigitsDef().integer(16).fraction(2))
            .field("dataOrig")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", DataOrig.class))
            .field("settlementTotal")
            .constraint(new DigitsDef().integer(18).fraction(2))
            .field("invoiceAmount")
            .constraint(new DigitsDef().integer(18).fraction(2))
            .field("isConsistent")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", IsConsistent.class))
            .field("verificationVariance")
            .constraint(new DigitsDef().integer(18).fraction(2))
            .field("cancelTime")
            .field("cancelReason")
            .constraint(new LengthDef().min(0).max(500))
            .field("id")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("createTime")
            .field("updateTime")
            .field("createUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("updateUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("createUserName")
            .constraint(new LengthDef().min(0).max(100))
            .field("updateUserName")
            .constraint(new LengthDef().min(0).max(100))
            .field("deleteFlag")
            .constraint(new LengthDef().min(0).max(2))
            .field("invoiceNo")
            .constraint(new LengthDef().min(0).max(200))
            .field("invoiceCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("expressNumber")
            .constraint(new LengthDef().min(0).max(50))
            .field("invoiceOrig")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", InvoiceOrig.class))
            .field("forceInvoice")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", ForceInvoice.class))
            .field("makeInvoiceTime")
            .field("salesbillNo")
            .constraint(new LengthDef().min(0).max(100))
            .field("storeName")
            .constraint(new LengthDef().min(0).max(30))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", StoreName.class))
            .field("assessmentCountDown")
            .constraint(new DigitsDef().integer(18).fraction(2))
            ;
        // order_pool_raw
        constraintMapping
            .type(com.xforceplus.ultraman.app.jcunilever.metadata.entity.OrderPoolRaw.class)
            .field("applyNo")
            .constraint(new LengthDef().min(0).max(50))
            .field("errorInfo")
            .constraint(new LengthDef().min(0).max(200))
            .field("storeId")
            .constraint(new LengthDef().min(0).max(50))
            .field("mainOrderNo")
            .constraint(new LengthDef().min(0).max(50))
            .field("businessType")
            .constraint(new LengthDef().min(0).max(20))
            .field("orderType")
            .constraint(new LengthDef().min(0).max(20))
            .field("purchaserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("purchaserTaxNo")
            .constraint(new LengthDef().min(0).max(30))
            .field("purchaserAddress")
            .constraint(new LengthDef().min(0).max(200))
            .field("purchaserTel")
            .constraint(new LengthDef().min(0).max(30))
            .field("purchaserBankName")
            .constraint(new LengthDef().min(0).max(200))
            .field("purchaserBankAccount")
            .constraint(new LengthDef().min(0).max(200))
            .field("applyType")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", ApplyType.class))
            .field("applyTime")
            .field("settlementStatus")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", SettlementStatus.class))
            .field("makeInvoiceStatus")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", MakeInvoiceStatus.class))
            .field("submitInvoiceStatus")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", SubmitInvoiceStatus.class))
            .field("invoiceType")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", InvoiceType.class))
            .field("headerType")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", HeaderType.class))
            .field("orderTotal")
            .constraint(new DigitsDef().integer(16).fraction(2))
            .field("goodsTotal")
            .constraint(new DigitsDef().integer(16).fraction(2))
            .field("dataOrig")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", DataOrig.class))
            .field("settlementTotal")
            .constraint(new DigitsDef().integer(18).fraction(2))
            .field("invoiceAmount")
            .constraint(new DigitsDef().integer(18).fraction(2))
            .field("isConsistent")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", IsConsistent.class))
            .field("verificationVariance")
            .constraint(new DigitsDef().integer(18).fraction(2))
            .field("cancelTime")
            .field("cancelReason")
            .constraint(new LengthDef().min(0).max(500))
            .field("invoiceNo")
            .constraint(new LengthDef().min(0).max(50))
            .field("invoiceCode")
            .constraint(new LengthDef().min(0).max(20))
            .field("expressNumber")
            .constraint(new LengthDef().min(0).max(50))
            .field("invoiceOrig")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", InvoiceOrig.class))
            .field("forceInvoice")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", ForceInvoice.class))
            .field("makeInvoiceTime")
            .field("salesbillNo")
            .constraint(new LengthDef().min(0).max(100))
            .field("storeName")
            .constraint(new LengthDef().min(0).max(30))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", StoreName.class))
            .field("invoiceDetails")
            .constraint(new LengthDef().min(0).max(2000))
            .field("id")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("createTime")
            .field("updateTime")
            .field("createUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("updateUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("createUserName")
            .constraint(new LengthDef().min(0).max(100))
            .field("updateUserName")
            .constraint(new LengthDef().min(0).max(100))
            .field("deleteFlag")
            .constraint(new LengthDef().min(0).max(2))
            .field("assessmentCountDown")
            .constraint(new DigitsDef().integer(18).fraction(2))
            ;
        // blob_files
        constraintMapping
            .type(com.xforceplus.ultraman.app.jcunilever.metadata.entity.BlobFiles.class)
            .field("fileName")
            .constraint(new LengthDef().min(0).max(200))
            .field("fileStatus")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", FileStatus.class))
            .field("fileType")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", FileType.class))
            .field("fileFormat")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", FileFormat.class))
            .field("id")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("createTime")
            .field("updateTime")
            .field("createUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("updateUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("createUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("updateUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("deleteFlag")
            .constraint(new LengthDef().min(0).max(2))
            .field("retryCount")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("fileDirectory")
            .constraint(new LengthDef().min(0).max(200))
            .field("fileArchiveDirectory")
            .constraint(new LengthDef().min(0).max(200))
            .field("errorInfo")
            .constraint(new LengthDef().min(0).max(1000))
            ;
        // billing_head
        constraintMapping
            .type(com.xforceplus.ultraman.app.jcunilever.metadata.entity.BillingHead.class)
            .field("id")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("createTime")
            .field("updateTime")
            .field("createUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("updateUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("createUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("updateUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("deleteFlag")
            .constraint(new LengthDef().min(0).max(2))
            .field("billingdocument")
            .constraint(new LengthDef().min(0).max(50))
            .field("companycode")
            .constraint(new LengthDef().min(0).max(50))
            .field("salesorganization")
            .constraint(new LengthDef().min(0).max(20))
            .field("payer")
            .constraint(new LengthDef().min(0).max(50))
            .field("nameofpayer")
            .constraint(new LengthDef().min(0).max(50))
            .field("soldtoparty")
            .constraint(new LengthDef().min(0).max(50))
            .field("nameofsoldtoparty")
            .constraint(new LengthDef().min(0).max(50))
            .field("keycustomer4name")
            .constraint(new LengthDef().min(0).max(50))
            .field("referencedocumentn")
            .constraint(new LengthDef().min(0).max(50))
            .field("billingdate")
            .constraint(new LengthDef().min(0).max(50))
            .field("billingtype")
            .constraint(new LengthDef().min(0).max(50))
            .field("netvalue")
            .constraint(new LengthDef().min(0).max(50))
            .field("taxamount")
            .constraint(new LengthDef().min(0).max(50))
            .field("ponumber")
            .constraint(new LengthDef().min(0).max(50))
            .field("redivnoticeno")
            .constraint(new LengthDef().min(0).max(50))
            .field("status")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", Status.class))
            ;
        // billing_details
        constraintMapping
            .type(com.xforceplus.ultraman.app.jcunilever.metadata.entity.BillingDetails.class)
            .field("id")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("createTime")
            .field("updateTime")
            .field("createUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("updateUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("createUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("updateUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("deleteFlag")
            .constraint(new LengthDef().min(0).max(2))
            .field("billingdocument")
            .constraint(new LengthDef().min(0).max(50))
            .field("billingitem")
            .constraint(new LengthDef().min(0).max(50))
            .field("companycode")
            .constraint(new LengthDef().min(0).max(50))
            .field("salesorganization")
            .constraint(new LengthDef().min(0).max(50))
            .field("billtoparty")
            .constraint(new LengthDef().min(0).max(50))
            .field("nameofbilltopart")
            .constraint(new LengthDef().min(0).max(50))
            .field("localsalesworkforcel4")
            .constraint(new LengthDef().min(0).max(50))
            .field("emscode")
            .constraint(new LengthDef().min(0).max(50))
            .field("keycustomer4name")
            .constraint(new LengthDef().min(0).max(50))
            .field("referenceobd")
            .constraint(new LengthDef().min(0).max(50))
            .field("billingdate")
            .constraint(new LengthDef().min(0).max(50))
            .field("billingtype")
            .constraint(new LengthDef().min(0).max(50))
            .field("ponumber")
            .constraint(new LengthDef().min(0).max(50))
            .field("salesdoc")
            .constraint(new LengthDef().min(0).max(50))
            .field("matnumusedcustomer")
            .constraint(new LengthDef().min(0).max(50))
            .field("status")
            .constraint(new LengthDef().min(0).max(50))
            .field("netvalue")
            .constraint(new LengthDef().min(0).max(50))
            .field("tax")
            .constraint(new LengthDef().min(0).max(50))
            .field("netvaluetax")
            .constraint(new LengthDef().min(0).max(50))
            .field("originalboxprice")
            .constraint(new LengthDef().min(0).max(50))
            .field("publishprice")
            .constraint(new LengthDef().min(0).max(50))
            .field("salesdeal")
            .constraint(new LengthDef().min(0).max(50))
            .field("externaldescription")
            .constraint(new LengthDef().min(0).max(50))
            .field("otdiscount")
            .constraint(new LengthDef().min(0).max(50))
            .field("eotdiscount")
            .constraint(new LengthDef().min(0).max(50))
            .field("cppdiscount")
            .constraint(new LengthDef().min(0).max(50))
            .field("aftereotot")
            .constraint(new LengthDef().min(0).max(50))
            .field("internalorder")
            .constraint(new LengthDef().min(0).max(50))
            .field("material")
            .constraint(new LengthDef().min(0).max(50))
            .field("materialdescription")
            .constraint(new LengthDef().min(0).max(50))
            .field("invoicedqty")
            .constraint(new LengthDef().min(0).max(50))
            .field("salesunit")
            .constraint(new LengthDef().min(0).max(50))
            .field("plant")
            .constraint(new LengthDef().min(0).max(50))
            .field("vatregistrationno")
            .constraint(new LengthDef().min(0).max(50))
            .field("taxrate")
            .constraint(new LengthDef().min(0).max(50))
            .field("netprice1")
            .constraint(new LengthDef().min(0).max(50))
            .field("netprice2")
            .constraint(new LengthDef().min(0).max(50))
            .field("discount")
            .constraint(new LengthDef().min(0).max(50))
            .field("netinvoice")
            .constraint(new LengthDef().min(0).max(50))
            .field("notes")
            .constraint(new LengthDef().min(0).max(500))
            .field("commoditycode")
            .constraint(new LengthDef().min(0).max(50))
            .field("customertelno")
            .constraint(new LengthDef().min(0).max(50))
            .field("customeraddress")
            .constraint(new LengthDef().min(0).max(50))
            .field("bankname")
            .constraint(new LengthDef().min(0).max(50))
            .field("bankaccountno")
            .constraint(new LengthDef().min(0).max(50))
            .field("statusfortransfer")
            .constraint(new LengthDef().min(0).max(50))
            .field("gtxinvoicecode")
            .constraint(new LengthDef().min(0).max(50))
            .field("goldentaxissuedate")
            .constraint(new LengthDef().min(0).max(50))
            ;
        // billing_discount_details
        constraintMapping
            .type(com.xforceplus.ultraman.app.jcunilever.metadata.entity.BillingDiscountDetails.class)
            .field("id")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("createTime")
            .field("updateTime")
            .field("createUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("updateUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("createUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("updateUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("deleteFlag")
            .constraint(new LengthDef().min(0).max(2))
            .field("billingdoc")
            .constraint(new LengthDef().min(0).max(50))
            .field("billingitem")
            .constraint(new LengthDef().min(0).max(20))
            .field("conditiontype")
            .constraint(new LengthDef().min(0).max(30))
            .field("conditionunit")
            .constraint(new LengthDef().min(0).max(30))
            .field("conditionvalue")
            .constraint(new LengthDef().min(0).max(30))
            ;
        // producer
        constraintMapping
            .type(com.xforceplus.ultraman.app.jcunilever.metadata.entity.Producer.class)
            .field("produceObj")
            .constraint(new LengthDef().min(0).max(50))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", ProduceObj.class))
            .field("produceStatus")
            .constraint(new LengthDef().min(0).max(50))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", ProduceStatus.class))
            .field("produceType")
            .constraint(new LengthDef().min(0).max(50))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", ProduceType.class))
            .field("produceSource")
            .constraint(new LengthDef().min(0).max(50))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", ProduceSource.class))
            .field("produceMessage")
            .constraint(new LengthDef().min(0).max(10000))
            .field("serialNo")
            .constraint(new LengthDef().min(0).max(100))
            .field("businessKey")
            .constraint(new LengthDef().min(0).max(100))
            .field("id")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("createTime")
            .field("updateTime")
            .field("createUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("updateUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("createUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("updateUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("deleteFlag")
            .constraint(new LengthDef().min(0).max(2))
            ;
        // customer
        constraintMapping
            .type(com.xforceplus.ultraman.app.jcunilever.metadata.entity.Customer.class)
            .field("customerMessage")
            .constraint(new LengthDef().min(0).max(10000))
            .field("serialNo")
            .constraint(new LengthDef().min(0).max(50))
            .field("customerObj")
            .constraint(new LengthDef().min(0).max(50))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", ProduceObj.class))
            .field("customerStatus")
            .constraint(new LengthDef().min(0).max(50))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", ProduceStatus.class))
            .field("customerType")
            .constraint(new LengthDef().min(0).max(50))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", ProduceType.class))
            .field("customerSource")
            .constraint(new LengthDef().min(0).max(50))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", ProduceSource.class))
            .field("businessKey")
            .constraint(new LengthDef().min(0).max(50))
            .field("id")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("createTime")
            .field("updateTime")
            .field("createUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("updateUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("createUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("updateUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("deleteFlag")
            .constraint(new LengthDef().min(0).max(2))
            .field("retryCount")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("errorInfo")
            .constraint(new LengthDef().min(0).max(10000))
            .field("retryTime")
            ;
        // dispatcher_rule_config
        constraintMapping
            .type(com.xforceplus.ultraman.app.jcunilever.metadata.entity.DispatcherRuleConfig.class)
            .field("produceObj")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", ProduceObj.class))
            .field("produceType")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", ProduceType.class))
            .field("produceSource")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", ProduceSource.class))
            .field("isOpen")
            .field("consumerObj")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", ProduceObj.class))
            .field("consumerType")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", ProduceType.class))
            .field("consumerSource")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", ProduceSource.class))
            .field("nextRetryTime")
            .constraint(new LengthDef().min(0).max(50))
            .field("id")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("createTime")
            .field("updateTime")
            .field("createUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("updateUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("createUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("updateUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("deleteFlag")
            .constraint(new LengthDef().min(0).max(2))
            ;
        // business_unit_config
        constraintMapping
            .type(com.xforceplus.ultraman.app.jcunilever.metadata.entity.BusinessUnitConfig.class)
            .field("id")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("createTime")
            .field("updateTime")
            .field("createUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("updateUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("createUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("updateUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("deleteFlag")
            .constraint(new LengthDef().min(0).max(2))
            .field("businessUnit")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", BusinessUnit.class))
            .field("salesOrg")
            .constraint(new LengthDef().min(0).max(300))
            ;
        // discount_reason_config
        constraintMapping
            .type(com.xforceplus.ultraman.app.jcunilever.metadata.entity.DiscountReasonConfig.class)
            .field("discountName")
            .constraint(new LengthDef().min(0).max(50))
            .field("discountDescription")
            .constraint(new LengthDef().min(0).max(200))
            .field("id")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("createTime")
            .field("updateTime")
            .field("createUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("updateUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("createUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("updateUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("deleteFlag")
            .constraint(new LengthDef().min(0).max(2))
            .field("discountType")
            .constraint(new LengthDef().min(0).max(50))
            ;
        // no_invoiced_reason_config
        constraintMapping
            .type(com.xforceplus.ultraman.app.jcunilever.metadata.entity.NoInvoicedReasonConfig.class)
            .field("customerCode")
            .constraint(new LengthDef().min(0).max(20))
            .field("noInvoiceStatus")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", NoInvoiceStatus.class))
            .field("noInvoiceReason")
            .constraint(new LengthDef().min(0).max(200))
            .field("noInvoiceReasonDesc")
            .constraint(new LengthDef().min(0).max(200))
            .field("id")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("createTime")
            .field("updateTime")
            .field("createUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("updateUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("createUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("updateUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("deleteFlag")
            .constraint(new LengthDef().min(0).max(2))
            .field("businessUnit")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnums.class).param("value", BusinessUnit.class))
            ;
        // mdm_init
        constraintMapping
            .type(com.xforceplus.ultraman.app.jcunilever.metadata.entity.MdmInit.class)
            .field("status")
            .constraint(new LengthDef().min(0).max(500))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", Status.class))
            .field("retryCount")
            .constraint(new DigitsDef().integer(200).fraction(0))
            .field("lastFaultReason")
            .constraint(new LengthDef().min(0).max(200))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", HandleFaultReason.class))
            .field("nextProcessTime")
            .field("remark")
            .constraint(new LengthDef().min(0).max(500))
            .field("custCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("saleOrgCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("partnerFunc")
            .constraint(new LengthDef().min(0).max(200))
            .field("custAddr")
            .constraint(new LengthDef().min(0).max(1000))
            .field("toCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("toName")
            .constraint(new LengthDef().min(0).max(500))
            .field("tree1Level4Name")
            .constraint(new LengthDef().min(0).max(500))
            .field("toIsDefault")
            .constraint(new LengthDef().min(0).max(200))
            .field("docNum")
            .constraint(new LengthDef().min(0).max(200))
            .field("bankCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("bankName")
            .constraint(new LengthDef().min(0).max(500))
            .field("bankAccountNo")
            .constraint(new LengthDef().min(0).max(500))
            .field("bankAccountType")
            .constraint(new LengthDef().min(0).max(200))
            .field("phone")
            .constraint(new LengthDef().min(0).max(200))
            .field("vatRegistrationNo")
            .constraint(new LengthDef().min(0).max(500))
            .field("invoiceType")
            .constraint(new LengthDef().min(0).max(200))
            .field("billingMail")
            .constraint(new LengthDef().min(0).max(500))
            .field("guiCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("operationType")
            .constraint(new LengthDef().min(0).max(200))
            .field("combinationReference")
            .constraint(new LengthDef().min(0).max(1000))
            .field("dt")
            .constraint(new LengthDef().min(0).max(200))
            .field("ext1")
            .constraint(new LengthDef().min(0).max(1000))
            .field("ext2")
            .constraint(new LengthDef().min(0).max(200))
            .field("ext3")
            .constraint(new LengthDef().min(0).max(1000))
            .field("ext4")
            .constraint(new LengthDef().min(0).max(1000))
            .field("ext5")
            .constraint(new LengthDef().min(0).max(1000))
            .field("ext6")
            .constraint(new LengthDef().min(0).max(1000))
            .field("ext7")
            .constraint(new LengthDef().min(0).max(1000))
            .field("ext8")
            .constraint(new LengthDef().min(0).max(1000))
            .field("ext9")
            .constraint(new LengthDef().min(0).max(1000))
            .field("ext10")
            .constraint(new LengthDef().min(0).max(1000))
            .field("id")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("createTime")
            .field("updateTime")
            .field("createUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("updateUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("createUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("updateUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("deleteFlag")
            .constraint(new LengthDef().min(0).max(2))
            .field("isElectronic")
            .constraint(new LengthDef().min(0).max(50))
            .field("isDiscountDetails")
            .constraint(new LengthDef().min(0).max(20))
            ;
        // billing_init
        constraintMapping
            .type(com.xforceplus.ultraman.app.jcunilever.metadata.entity.BillingInit.class)
            .field("billingInit")
            .constraint(new LengthDef().min(0).max(20))
            .field("id")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("createTime")
            .field("updateTime")
            .field("createUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("updateUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("createUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("updateUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("deleteFlag")
            .constraint(new LengthDef().min(0).max(2))
            ;
        // open_a_r_init
        constraintMapping
            .type(com.xforceplus.ultraman.app.jcunilever.metadata.entity.OpenARInit.class)
            .field("openARInit")
            .constraint(new LengthDef().min(0).max(20))
            .field("id")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("createTime")
            .field("updateTime")
            .field("createUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("updateUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("createUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("updateUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("deleteFlag")
            .constraint(new LengthDef().min(0).max(2))
            ;
        // statement_init
        constraintMapping
            .type(com.xforceplus.ultraman.app.jcunilever.metadata.entity.StatementInit.class)
            .field("statementInit")
            .constraint(new LengthDef().min(0).max(20))
            .field("id")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("createTime")
            .field("updateTime")
            .field("createUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("updateUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("createUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("updateUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("deleteFlag")
            .constraint(new LengthDef().min(0).max(2))
            ;
        // invoice_match_init
        constraintMapping
            .type(com.xforceplus.ultraman.app.jcunilever.metadata.entity.InvoiceMatchInit.class)
            .field("invoiceMatchInit")
            .constraint(new LengthDef().min(0).max(20))
            .field("id")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("createTime")
            .field("updateTime")
            .field("createUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("updateUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("createUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("updateUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("deleteFlag")
            .constraint(new LengthDef().min(0).max(2))
            ;
        // mdm
        constraintMapping
            .type(com.xforceplus.ultraman.app.jcunilever.metadata.entity.Mdm.class)
            .field("custCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("saleOrgCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("tree1Level4Name")
            .constraint(new LengthDef().min(0).max(500))
            .field("toCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("toName")
            .constraint(new LengthDef().min(0).max(500))
            .field("isElectronic")
            .constraint(new LengthDef().min(0).max(200))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", Whether.class))
            .field("isDiscountDetails")
            .constraint(new LengthDef().min(0).max(200))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", Whether.class))
            .field("customerType")
            .constraint(new LengthDef().min(0).max(200))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", CustomerType.class))
            .field("isAccountCheck")
            .constraint(new LengthDef().min(0).max(200))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", Whether.class))
            .field("retailerCode")
            .constraint(new LengthDef().min(0).max(500))
            .field("retailerName")
            .constraint(new LengthDef().min(0).max(500))
            .field("id")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("createTime")
            .field("updateTime")
            .field("createUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("updateUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("createUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("updateUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("deleteFlag")
            .constraint(new LengthDef().min(0).max(2))
            .field("invoiceType")
            .constraint(new LengthDef().min(0).max(200))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", InvoiceTypeMDM.class))
            .field("partners")
            .constraint(new LengthDef().min(0).max(100))
            .field("isDefault")
            .constraint(new LengthDef().min(0).max(50))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", Whether.class))
            .field("partnerFuc")
            .constraint(new LengthDef().min(0).max(200))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", PartnerFuc.class))
            ;
        // mdm_sync
        constraintMapping
            .type(com.xforceplus.ultraman.app.jcunilever.metadata.entity.MdmSync.class)
            .field("status")
            .constraint(new LengthDef().min(0).max(500))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", Status.class))
            .field("retryCount")
            .constraint(new DigitsDef().integer(200).fraction(0))
            .field("lastFaultReason")
            .constraint(new LengthDef().min(0).max(500))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", HandleFaultReason.class))
            .field("remark")
            .constraint(new LengthDef().min(0).max(200))
            .field("queryDate")
            .field("lastQueryPage")
            .constraint(new DigitsDef().integer(200).fraction(0))
            .field("id")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("createTime")
            .field("updateTime")
            .field("createUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("updateUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("createUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("updateUserName")
            .constraint(new LengthDef().min(0).max(200))
            .field("deleteFlag")
            .constraint(new LengthDef().min(0).max(2))
            .field("totalCount")
            .constraint(new DigitsDef().integer(200).fraction(0))
            .field("pageCount")
            .constraint(new DigitsDef().integer(200).fraction(0))
            .field("pageSize")
            .constraint(new DigitsDef().integer(200).fraction(0))
            ;
        // flow_instance_log
        constraintMapping
            .type(com.xforceplus.ultraman.app.jcunilever.metadata.entity.FlowInstanceLog.class)
            .field("id")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("createTime")
            .field("updateTime")
            .field("createUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("updateUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("createUserName")
            .constraint(new LengthDef().min(0).max(100))
            .field("updateUserName")
            .constraint(new LengthDef().min(0).max(100))
            .field("deleteFlag")
            .constraint(new LengthDef().min(0).max(2))
            .field("instanceId")
            .constraint(new LengthDef().min(0).max(36))
            .field("flowCode")
            .constraint(new LengthDef().min(0).max(64))
            .field("errorMsg")
            .constraint(new LengthDef().min(0).max(512))
            .field("exceptionTrace")
            .constraint(new LengthDef().min(0).max(4096))
            .field("errorNodeId")
            .constraint(new LengthDef().min(0).max(64))
            .field("rollback")
            .field("snapshot")
            .constraint(new LengthDef().min(0).max(4096))
            .field("snapshotFormat")
            .constraint(new LengthDef().min(0).max(128))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", SnaphostFormat.class))
            .field("rollbackNodeId")
            .constraint(new LengthDef().min(0).max(64))
            .field("flowName")
            .constraint(new LengthDef().min(0).max(32))
            .field("status")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", FlowStatus.class))
            .field("startSnapshot")
            .constraint(new LengthDef().min(0).max(4096))
            .field("startSnapshotFormat")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", SnaphostFormat.class))
            .field("userContext")
            .constraint(new LengthDef().min(0).max(4098))
            .field("nodeHistory")
            .constraint(new LengthDef().min(0).max(4096))
            ;
        // flow_replay_log
        constraintMapping
            .type(com.xforceplus.ultraman.app.jcunilever.metadata.entity.FlowReplayLog.class)
            .field("flowInstanceId")
            .constraint(new LengthDef().min(0).max(36))
            .field("id")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("tenantCode")
            .constraint(new LengthDef().min(0).max(200))
            .field("createTime")
            .field("updateTime")
            .field("createUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("updateUserId")
            .constraint(new DigitsDef().integer(20).fraction(0))
            .field("createUserName")
            .constraint(new LengthDef().min(0).max(100))
            .field("updateUserName")
            .constraint(new LengthDef().min(0).max(100))
            .field("deleteFlag")
            .constraint(new LengthDef().min(0).max(2))
            .field("flowCode")
            .constraint(new LengthDef().min(0).max(64))
            .field("errorMsg")
            .constraint(new LengthDef().min(0).max(512))
            .field("exceptionTrace")
            .constraint(new LengthDef().min(0).max(4096))
            .field("errorNodeId")
            .constraint(new LengthDef().min(0).max(64))
            .field("rollback")
            .field("rollbackNodeId")
            .constraint(new LengthDef().min(0).max(64))
            .field("status")
            .constraint(new LengthDef().min(0).max(20))
            .constraint(new GenericConstraintDef<>(CheckUltramanEnum.class).param("value", FlowStatus.class))
            ;
    }
}
