package com.xforceplus.ultraman.billing.client.aspect;


import com.xforceplus.ultraman.billing.client.UsageLifecycle;
import com.xforceplus.ultraman.billing.client.impl.UsageMatchedContext;
import com.xforceplus.ultraman.billing.client.remote.UsageApi;
import com.xforceplus.ultraman.billing.domain.*;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import io.vavr.Tuple5;
import io.vavr.control.Either;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.expression.Expression;
import org.springframework.expression.ExpressionParser;
import org.springframework.expression.spel.standard.SpelExpressionParser;
import org.springframework.expression.spel.support.StandardEvaluationContext;

import java.util.*;
import java.util.concurrent.CompletionStage;
import java.util.function.Supplier;
import java.util.stream.Collectors;

@Slf4j
@Aspect
public class BillingAspect {

    private TeamCodeSupplier teamCodeSupplier;

    private List<String> serviceTokens;

    private UsageApi usageApi;

    @Autowired
    private UsageLifecycle usageLifecycle;

    @Autowired
    private Supplier<String> appCodeSupplier;

    public BillingAspect(TeamCodeSupplier teamCodeSupplier, Supplier<String> appCodeSupplier, UsageApi usageApi, String serviceTokens) {
        this.teamCodeSupplier = teamCodeSupplier;
        this.usageApi = usageApi;
        this.serviceTokens = Arrays.asList(serviceTokens.split(","));
        this.appCodeSupplier = appCodeSupplier;
    }
    
    private void rollback(Map<String, String> preFetchIdMapping) {
        try {
            if(!preFetchIdMapping.isEmpty()) {
                Collection<String> values = preFetchIdMapping.values();
                RollbackRequest rollbackRequest = new RollbackRequest();
                rollbackRequest.setPrefetchIds(new ArrayList<>(values));
                usageApi.rollback(rollbackRequest);
            }
        } catch(Throwable throwable) {
            log.warn("", throwable);
        }
    }

    @Around("(@within(com.xforceplus.ultraman.billing.client.aspect.BillingScope) && execution(public * *(..))) || @annotation(com.xforceplus.ultraman.billing.client.aspect.BillingScope)")
    public Object transactionExecution(ProceedingJoinPoint pjp) throws Throwable {
        UsageMatchedContext usageMatchedContext = null;
        List<Tuple5<Tuple2<Usage, UsageRule>, String, String, Response, Integer>> passedUsageList = new ArrayList<>();
        Map<String, String> preFetchIdMapping = new HashMap<>();
        try {
            usageMatchedContext = usageLifecycle
                    .matchedUsage(teamCodeSupplier, x -> x.getExternalType().equalsIgnoreCase("METHOD"));
        } catch (Throwable throwable) {
            log.error("", throwable);
        }

        String teamCode = teamCodeSupplier.get();
        
        if(usageMatchedContext != null) {

            List<Either<Usage, UsageRule>> targetUsages = matchUsage(usageMatchedContext, pjp);
  
            for (Either<Usage, UsageRule> usage : targetUsages) {
                Tuple5<Tuple2<Usage, UsageRule>, String, String, Response, Integer> tupleResponse = usageProcess(pjp, usageMatchedContext, usage, teamCode, preFetchIdMapping);
                if (tupleResponse != null) {
                    if (tupleResponse._5 < 0) {
                        rollback(preFetchIdMapping);
                        throw new RuntimeException(tupleResponse._4.getMessage());
                    } else {
                        passedUsageList.add(tupleResponse);
                    }
                }
            }
        }

        Object proceedResult;

        List<Tuple5<Usage, String, String, Response, Integer>> callbacks = new ArrayList<>();

        //TODO 两个时点
        //usage -> key && size
        try {
            proceedResult = pjp.proceed();
            for (Tuple5<Tuple2<Usage, UsageRule>, String, String, Response, Integer> passedUsage : passedUsageList) {
                if(passedUsage != null) {
                    try {
                        Map<String, String> keySizeMap = new HashMap<>();
                        String targetKey = passedUsage._2;
                        String targetSize = passedUsage._3;

                        boolean inCB = false;

                        keySizeMap.put("key", targetKey);
                        keySizeMap.put("size", targetSize);

                        UsageRequest usageRequest = new UsageRequest();
                        usageRequest.setUsage(passedUsage._1._1.getName());
                        usageRequest.setPreFetchId(preFetchIdMapping.get(passedUsage._1._1.getId()));
                        String keyExpr = null;
                        String sizeExpr = null;
                        //TODO
                        if (keySizeMap.get("key") == null) {
                            
                            keyExpr = passedUsage._1._1.getKeyExpr() == null ? passedUsage._1._2.getKeyExpr() : passedUsage._1._1.getKeyExpr();
                            if (keyExpr.startsWith("CALLBACK:") && proceedResult instanceof CompletionStage) {
                                keyExpr = keyExpr.substring(9);
                                inCB = true;
                            } else if (keyExpr.startsWith("RETURN:")) {
                                Map<String, Object> contextMap = new HashMap<>();
                                contextMap.put("root", proceedResult);
                                keyExpr = keyExpr.substring(7);
                                String evaluatedKey = evaluate(keyExpr, String.class, contextMap);
                                keySizeMap.put("key", evaluatedKey);
                            }
                        }

                        if (keySizeMap.get("size") == null) {
                            sizeExpr = passedUsage._1._1.getSizeExpr() == null ? passedUsage._1._2.getSizeExpr() : passedUsage._1._1.getSizeExpr();
                            if (sizeExpr.startsWith("CALLBACK:") && proceedResult instanceof CompletionStage) {
                                sizeExpr = sizeExpr.substring(9);
                                inCB = true;
                            } else if (sizeExpr.startsWith("RETURN:")) {
                                Map<String, Object> contextMap = new HashMap<>();
                                contextMap.put("root", proceedResult);
                                sizeExpr = sizeExpr.substring(7);
                                String evaluatedSize = evaluate(sizeExpr, String.class, contextMap);
                                keySizeMap.put("size", evaluatedSize);
                            }
                        }
                        if (!inCB) {
                            usageRequest.setKey(keySizeMap.get("key"));
                            usageRequest.setSize(keySizeMap.get("size"));
                            usageRequest.setTenant(teamCode);
                            Response record = usageApi.record(usageRequest);
                        } else {
                            String finalKeyExpr = keyExpr;
                            String finalSizeExpr = sizeExpr;
                            ((CompletionStage<?>) proceedResult).thenApply(x -> {
                                try {
                                    String targetKeyInCB = keySizeMap.get("key");
                                    String targetSizeInCB = keySizeMap.get("size");
                                    if (finalKeyExpr != null && targetKeyInCB == null) {
                                        Map<String, Object> context = new HashMap<>();
                                        context.put("root", x);
                                        targetKeyInCB = evaluate(finalKeyExpr, String.class, context);
                                    }

                                    if (finalSizeExpr != null && targetSizeInCB == null) {
                                        Map<String, Object> context = new HashMap<>();
                                        context.put("root", x);
                                        targetSizeInCB = evaluate(finalSizeExpr, String.class, context);
                                    }

                                    usageRequest.setKey(targetKeyInCB);
                                    usageRequest.setSize(targetSizeInCB);
                                    usageRequest.setTenant(teamCode);
                                    Response record = usageApi.record(usageRequest);
                                } catch (Throwable throwable) {
                                    log.error("", throwable);
                                }
                                return x;
                            });
                        }
                    } catch (Throwable throwable) {
                        log.error("", throwable);
                    }
                }
            }
        } catch (Throwable throwable) {
            throw throwable;
        }

        return proceedResult;
    }


    private String renderExpr(String expression, Map<String, Object> templateBinding) {
        for (Map.Entry<String, Object> entry : templateBinding.entrySet()) {
            String k = entry.getKey();
            Object v = entry.getValue();
            if (expression.contains("{{".concat(k).concat("}}"))) {
                expression = expression.replaceAll("\\{\\{".concat(k).concat("\\}\\}"), v.toString());
            }
        }

        return expression;
    }


    private Tuple5<Tuple2<Usage, UsageRule>, String, String, Response, Integer> usageProcess(ProceedingJoinPoint pjp
            , UsageMatchedContext usageMatchedContext, Either<Usage, UsageRule> targetUsage, String teamCode, Map<String, String> mapping) {

        String limitationExpr = null;
        String dryRunSizeExpr = null;
        String validateExpr = null;
        String keyExpr = null;
        String sizeExpr = null;
        String filterExpr = null;

        String targetKey = null;
        String size = null;
        Usage retUsage = null;
        UsageRule retRule = null;

        if (targetUsage.isLeft()) {
            Usage targetUsageLeft = targetUsage.getLeft();
            limitationExpr = targetUsageLeft.getLimitation();
            dryRunSizeExpr = targetUsageLeft.getDryRunSizeExpr();
            validateExpr = targetUsageLeft.getLimitationExpr();
            sizeExpr = targetUsageLeft.getSizeExpr();
            keyExpr = targetUsageLeft.getKeyExpr();
            retUsage = targetUsageLeft;
        } else {
            UsageRule usageRule = targetUsage.get();
            Usage usageByRuleId = usageMatchedContext.getUsageByRuleId(usageRule.getId().toString());
            Map<String, Object> templateBinding = Optional.ofNullable(usageByRuleId.getTemplateBinding()).orElseGet(Collections::emptyMap);
            Map<String, Object> finalTemplateBinding = new HashMap<>(templateBinding);
            finalTemplateBinding.put("appCode", appCodeSupplier.get());
            limitationExpr = usageByRuleId.getLimitation();
            dryRunSizeExpr = usageRule.getDryRunSizeExpr() == null ? null : renderExpr(usageRule.getDryRunSizeExpr(), finalTemplateBinding);
            validateExpr = usageRule.getValidateExpr() == null ? null : renderExpr(usageRule.getValidateExpr(), finalTemplateBinding);
            sizeExpr = usageRule.getSizeExpr() == null ? null : renderExpr(usageRule.getSizeExpr(), finalTemplateBinding);
            keyExpr = usageRule.getKeyExpr() == null ? null : renderExpr(usageRule.getKeyExpr(), finalTemplateBinding);
            filterExpr = usageRule.getFilterExpr() == null ? null : renderExpr(usageRule.getFilterExpr(), finalTemplateBinding);
            retUsage = usageByRuleId;
            retRule = usageRule;
        }

        if (!StringUtils.isEmpty(filterExpr)) {
            Map<String, Object> vars = new HashMap<>();
            if (filterExpr.startsWith("PARAMS:")) {
                filterExpr = filterExpr.substring(7);
                Map<String, Object> params = new HashMap<>();
                Object[] args = pjp.getArgs();
                for (int i = 0; i < args.length; i++) {
                    params.put(Integer.toString(i), args[i]);
                }
                vars.putAll(params);
            }

            Boolean matched = evaluate(filterExpr, Boolean.class, vars);
            if (!matched) {
                //skip this
                return null;
            }
        }

        if (StringUtils.isEmpty(keyExpr)) {
            targetKey = "default";
        } else {
            Map<String, Object> vars = new HashMap<>();
            boolean skipEvaluation = false;
            if (keyExpr.startsWith("PARAMS:")) {
                keyExpr = keyExpr.substring(7);
                Map<String, Object> params = new HashMap<>();
                Object[] args = pjp.getArgs();
                for (int i = 0; i < args.length; i++) {
                    params.put(Integer.toString(i), args[i]);
                }
                vars.putAll(params);
            } else {
                skipEvaluation = true;
            }

            if (!skipEvaluation) {
                targetKey = evaluate(keyExpr, String.class, vars);
            }
        }

        if (!StringUtils.isEmpty(validateExpr)) {
            Map<String, Object> vars = new HashMap<>();
            //do with expr
            if (validateExpr.startsWith("PARAMS:")) {
                validateExpr = validateExpr.substring(7);
                Boolean isPassed = true;
                Map<String, Object> params = new HashMap<>();
                Object[] args = pjp.getArgs();
                for (int i = 0; i < args.length; i++) {
                    params.put(Integer.toString(i), args[i]);
                }

                vars.putAll(params);
                isPassed = evaluate(validateExpr, Boolean.class, vars);
                if (!isPassed) {
                    Response overResp = new Response();
                    overResp.setCode("-1");
                    overResp.setMessage(targetUsage.getLeft().getErrorMsg());
                    return Tuple.of(Tuple.of(retUsage, retRule), null, null, overResp, -1);
                }
            }
        }

        //record check
        if (!StringUtils.isEmpty(limitationExpr)) {
            CheckUsageRequest checkUsageRequest = new CheckUsageRequest();
            if (!StringUtils.isEmpty(dryRunSizeExpr)) {
                if (dryRunSizeExpr.startsWith("PARAMS:")) {
                    dryRunSizeExpr = dryRunSizeExpr.substring(7);
                    Map<String, Object> params = new HashMap<>();
                    Object[] args = pjp.getArgs();
                    for (int i = 0; i < args.length; i++) {
                        params.put(Integer.toString(i), args[i]);
                    }
                    String dryRunSize = evaluate(dryRunSizeExpr, String.class, params);
                    checkUsageRequest.setDryRunValue(dryRunSize);
                }
            }

            if (checkUsageRequest.getDryRunValue() == null) {
                checkUsageRequest.setDryRunValue("1");
            }
            checkUsageRequest.setUsage(retUsage);
            checkUsageRequest.setTenant(teamCode);
            checkUsageRequest.setRecordKey(targetKey);
            checkUsageRequest.setPreFetch(true);
            Response checkUsage = usageApi.checkUsage(checkUsageRequest);
            if (!"1".equalsIgnoreCase(checkUsage.getCode())) {
                return Tuple.of(Tuple.of(retUsage, retRule), null, null, checkUsage, -1);
            } else {
                if(checkUsage.getResult() != null) {
                    mapping.put(retUsage.getId(), checkUsage.getResult().toString());
                }
            }
        }

        String targetSize;
        if (StringUtils.isEmpty(sizeExpr)) {
            targetSize = "1";
        } else {
            Map<String, Object> vars = new HashMap<>();
            boolean skipEvaluation = false;
            if (sizeExpr.startsWith("PARAMS:")) {
                sizeExpr = sizeExpr.substring(7);
                Map<String, Object> params = new HashMap<>();
                Object[] args = pjp.getArgs();
                for (int i = 0; i < args.length; i++) {
                    params.put(Integer.toString(i), args[i]);
                }
                vars.put("root", params);
                //do with body
                //TODO
            } else {
                skipEvaluation = true;
            }

            if (!skipEvaluation) {
                targetSize = evaluate(sizeExpr, String.class, vars);
            } else {
                if (sizeExpr.startsWith("CALLBACK:")) {
                    targetSize = null;
                } else {
                    targetSize = sizeExpr;
                }
            }
        }

        return Tuple.of(Tuple.of(retUsage, retRule), targetKey, targetSize, null, 1);
    }

    private List<Either<Usage, UsageRule>> matchUsage(UsageMatchedContext usageMatchedContext, ProceedingJoinPoint pjp) {
        Signature signature = pjp.getSignature();
        String targetResource = signature.getDeclaringTypeName().concat("#").concat(signature.getName());
        List<Either<Usage, UsageRule>> retList = new ArrayList<>();
        usageMatchedContext.getUsageIdMapping().values()
                .forEach(v -> {
                    if (v.getResourceRef() != null) {
                        if (v.getResourceRef().getExternalResource().equalsIgnoreCase(targetResource)) {
                            retList.add(Either.left(v));
                        }
                    } else {
                        List<UsageRule> rules = v.getRules();
                        if (rules != null && !rules.isEmpty()) {
                            rules.forEach(r -> {
                                if (r.getResourceRef() != null) {
                                    if (r.getResourceRef().getExternalResource().equalsIgnoreCase(targetResource)) {
                                        retList.add(Either.right(r));
                                    }
                                }
                            });
                        }
                    }
                });

        return retList;
    }

    private <T> T evaluate(String keyExpr, Class<T> clz, Map<String, Object> varMapping) {
        ExpressionParser parser = new SpelExpressionParser();
        StandardEvaluationContext ec = new StandardEvaluationContext(varMapping);
        Expression expression = parser.parseExpression(keyExpr);
        T result = expression.getValue(ec, clz);
        return result;
    }
}
