package com.xforceplus.ultraman.billing.client.filter;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.collect.Sets;
import com.usthe.sureness.matcher.util.TirePathTree;
import com.usthe.sureness.mgt.SurenessSecurityManager;
import com.usthe.sureness.subject.Subject;
import com.xforceplus.ultraman.billing.client.UsageLifecycle;
import com.xforceplus.ultraman.billing.client.dto.UsageResponse;
import com.xforceplus.ultraman.billing.client.impl.UsageMatchedContext;
import com.xforceplus.ultraman.billing.client.remote.UsageApi;
import com.xforceplus.ultraman.billing.client.utils.CachedBodyHttpServletRequest;
import com.xforceplus.ultraman.billing.client.utils.CachedBodyHttpServletResponse;
import com.xforceplus.ultraman.billing.client.utils.HttpPayloadExtractor;
import com.xforceplus.ultraman.billing.client.utils.PayloadExtractor;
import com.xforceplus.ultraman.billing.domain.*;
import com.xforceplus.ultraman.bocp.metadata.util.JsonUtils;
import com.xforceplus.ultraman.bocp.uc.context.UcUserInfoHolder;
import com.xforceplus.ultraman.bocp.uc.pojo.auth.UcAuthUser;
import io.vavr.Tuple;
import io.vavr.Tuple3;
import io.vavr.Tuple4;
import io.vavr.Tuple5;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.util.Strings;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.expression.Expression;
import org.springframework.expression.ExpressionParser;
import org.springframework.expression.ParserContext;
import org.springframework.expression.spel.standard.SpelExpressionParser;
import org.springframework.expression.spel.support.StandardEvaluationContext;
import org.springframework.web.filter.OncePerRequestFilter;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;

import static com.xforceplus.ultraman.billing.client.utils.ExtractUtils.extractKeyValueMapping;

public class BillingFilter extends OncePerRequestFilter {

    private UsageApi usageApi;

    @Value("${ultraman.billing.service}")
    private List<String> serviceList;
    
    @Autowired
    private UsageLifecycle usageLifecycle;

    public BillingFilter(UsageApi usageApi) {
        this.usageApi = usageApi;
    }

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws ServletException, IOException {

        UsageMatchedContext usageMatchedContext = null;
        String targetResource = null;
        String teamCode = null;
        /**
         * usage id or rule id
         */
        List<String> matchedIds = new ArrayList<>();
        try {
            UcAuthUser ucAuthUser = UcUserInfoHolder.get();
            if (ucAuthUser != null) {
                teamCode = ucAuthUser.getTeamCode();
                if (!StringUtils.isEmpty(teamCode)) {
                    String finalTeamCode = teamCode;
                    usageMatchedContext = usageLifecycle
                            .matchedUsage(() -> finalTeamCode, source -> source
                                    .getExternalType().equalsIgnoreCase("HTTP"));
                }
            }
            
            if(usageMatchedContext != null) {
                TirePathTree tirePathTree = new TirePathTree();
                Set<String> resources = usageMatchedContext.getSourceIdMapping().entrySet().stream().map(e -> {
                    return transformResource(e.getKey(), e.getValue());
                }).collect(Collectors.toSet());

                tirePathTree.buildTree(resources);
                List<Subject> subject = SurenessSecurityManager.getInstance().createSubject(request);
                if (!subject.isEmpty()) {
                    Subject firstSubject = subject.get(0);
                    targetResource = (String) firstSubject.getTargetResource();
                    tirePathTree.searchPathFilterRoles(request.getMethod());
                    String roles = tirePathTree.searchPathFilterRoles(targetResource);
                    if (roles != null) {
                        String[] ids = roles.split(",");
                        matchedIds.addAll(Arrays.asList(ids));
                    }
                }
            }
        } catch (Throwable throwable) {
            logger.error(throwable);
        }

        request = new CachedBodyHttpServletRequest(request);
        List<Tuple5<Usage, String, String, Response, Integer>> passedUsageList = new ArrayList<>();

        for (String idToken : matchedIds) {
            Tuple5<Usage, String, String, Response, Integer> tupleResponse = usageProcess(request, usageMatchedContext, idToken, targetResource, teamCode);
            if (tupleResponse._5 < 0) {
                Response failedResponse = tupleResponse._4;
                response.setContentType("application/json");
                response.setStatus(403);
                response.setCharacterEncoding("UTF8");
                if (failedResponse != null) {
                    response.getWriter().write(JsonUtils.object2Json(failedResponse));
                    response.flushBuffer();
                }
                return;
            } else {
                passedUsageList.add(tupleResponse);
            }
        }

        response = new CachedBodyHttpServletResponse(response);
        filterChain.doFilter(request, response);

        if (response.getStatus() == 200) {

            HttpPayloadExtractor httpPayloadExtractor = new HttpPayloadExtractor();
            String responseBody = httpPayloadExtractor.extract(response);
            try {
                JsonNode jsonNode = JsonUtils.readTree(responseBody);
                JsonNode codeNode = jsonNode.get("code");
                if (codeNode != null) {
                    //TODO
                    if (!codeNode.asText().equals("0")) {
                        return;
                    }
                }
            } catch (Throwable throwable) {
                logger.warn(throwable);
            }

            //TODO
            Map<String, Object> map = JsonUtils.json2Object(responseBody, Map.class);

            for (Tuple5<Usage, String, String, Response, Integer> passedUsage : passedUsageList) {
                try {
                    String targetKey = passedUsage._2;
                    String targetSize = passedUsage._3;
                    UsageRequest usageRequest = new UsageRequest();
                    usageRequest.setUsage(passedUsage._1.getName());
                    if (targetKey == null) {
                        String keyExpr = passedUsage._1.getKeyExpr();
                        if (keyExpr.startsWith("RES_BODY:")) {
                            keyExpr = keyExpr.substring(9);
                            targetKey = evaluate(keyExpr, String.class, map);
                        }
                    }

                    if (targetSize == null) {
                        String sizeExpr = passedUsage._1.getSizeExpr();
                        if (sizeExpr.startsWith("RES_BODY:")) {
                            sizeExpr = sizeExpr.substring(9);
                            targetSize = evaluate(sizeExpr, String.class, map);
                        }
                    }
                    usageRequest.setKey(targetKey);
                    usageRequest.setSize(targetSize);
                    usageRequest.setTenant(teamCode);
                    Response record = usageApi.record(usageRequest);
                } catch (Throwable throwable) {
                    logger.error(throwable);
                }
            }
        }
    }

    private String renderExpr(String expression, Map<String, Object> templateBinding) {
        for (Map.Entry<String, Object> entry : templateBinding.entrySet()) {
            String k = entry.getKey();
            Object v = entry.getValue();
            if (expression.contains("{{".concat(k).concat("}}"))) {
                expression = expression.replaceAll("\\{\\{".concat(k).concat("\\}\\}"), v.toString());
            }
        }
        
        return expression;
    }
    
    private Tuple5<Usage, String, String, Response, Integer> usageProcess(HttpServletRequest request
            , UsageMatchedContext context, String idToken, String targetResource, String teamCode) {

        String targetKey = null;
        String targetSize = null;

        //isPre
        String limitationExpr = null;
        String validationExpr = null;
        String dryRunSizeExpr = null;
        String keyExpr = null;
        String sizeExpr = null;
        
        ResourceCreateRequest relatedResource = null;
        Usage targetUsage = null;
        
        if(idToken.startsWith("rule")) {
            idToken = idToken.substring(5);
            UsageRule usageRule = context.getUsageRuleById(idToken);
            if(usageRule != null) {
                targetUsage = context.getUsageByRuleId(idToken);
                Map<String, Object> templateBinding = targetUsage.getTemplateBinding();
                limitationExpr = targetUsage.getLimitation();
                validationExpr = usageRule.getValidateExpr() == null ? null : renderExpr(usageRule.getValidateExpr(), templateBinding); ;
                dryRunSizeExpr = usageRule.getDryRunSizeExpr() == null ? null : renderExpr(usageRule.getDryRunSizeExpr(), templateBinding);
                keyExpr = usageRule.getKeyExpr() == null ? null: renderExpr(usageRule.getKeyExpr(), templateBinding);;
                sizeExpr = usageRule.getSizeExpr() == null ? null : renderExpr(usageRule.getSizeExpr(), templateBinding);; ;
                relatedResource = usageRule.getResourceRef();
            }
        } else {
            Usage usage = context.getUsageById(idToken);
            limitationExpr = usage.getLimitation();
            validationExpr = usage.getLimitationExpr();
            dryRunSizeExpr = usage.getDryRunSizeExpr();
            keyExpr = usage.getKeyExpr();
            sizeExpr = usage.getSizeExpr();
            relatedResource = usage.getResourceRef();
            targetUsage = usage;
        }
        
        //TODO optimize
        HttpPayloadExtractor httpPayloadExtractor = new HttpPayloadExtractor();

        if (StringUtils.isEmpty(sizeExpr)) {
            targetSize = "1";
        } else {
            boolean skipEvaluation = false;
            Map<String, Object> varMapping = new HashMap<>();
            if (sizeExpr.startsWith("BODY:")) {
                sizeExpr = sizeExpr.substring(5);
                //do with body
                String extractRequest = httpPayloadExtractor.extract(request);
                if (extractRequest.startsWith("{")) {
                    Map map = JsonUtils.json2Object(extractRequest, Map.class);
                    varMapping.putAll(map);
                } else if (extractRequest.startsWith("[")) {
                    List<Map> maps = JsonUtils.json2ObjectList(extractRequest, Map.class);
                    varMapping.put("root", maps);
                }
            } else if (sizeExpr.startsWith("PATH:")) {
                //do with url
                //prepare path env
                sizeExpr = sizeExpr.substring(5);
                if(relatedResource != null) {
                    preparePathEnv(relatedResource.getExternalResource(), targetResource, varMapping);
                }
            } else if (sizeExpr.startsWith("QUERY:")) {
                //do with query
                sizeExpr = sizeExpr.substring(6);
            } else {
                skipEvaluation = true;
            }

            if (!skipEvaluation) {
                targetSize = evaluate(sizeExpr, String.class, varMapping);
            } else {
                targetSize = sizeExpr;
            }
        }

        if (StringUtils.isEmpty(keyExpr)) {
            targetKey = "default";
        } else {
            boolean skipEvaluation = false;
            Map<String, Object> varMapping = new HashMap<>();
            if (keyExpr.startsWith("BODY:")) {
                keyExpr = keyExpr.substring(5);
                //do with body
                //TODO
            } else if (keyExpr.startsWith("PATH:")) {
                //do with url
                //prepare path env
                keyExpr = keyExpr.substring(5);
                if(relatedResource != null) {
                    preparePathEnv(relatedResource.getExternalResource(), targetResource, varMapping);
                }
            } else if (keyExpr.startsWith("QUERY:")) {
                String queryString = request.getQueryString();
                Map<String, Object> mapping = parseQueryString(queryString);
                varMapping.putAll(mapping);
                //do with query
                keyExpr = keyExpr.substring(6);
            } else if (keyExpr.startsWith("HEADER:")) {
                Enumeration<String> headerNames = request.getHeaderNames();
                while (headerNames.hasMoreElements()) {
                    String next = headerNames.nextElement();
                    varMapping.put(next, request.getHeader(next));
                }
                keyExpr = keyExpr.substring(7);
            } else {
                skipEvaluation = true;
            }

            if (!skipEvaluation) {
                targetKey = evaluate(keyExpr, String.class, varMapping);
            }
        }

        if (!StringUtils.isEmpty(validationExpr)) {
            if (validationExpr.startsWith("BODY:")) {
                validationExpr = validationExpr.substring(5);
                String extract = httpPayloadExtractor.extract(request);
                if (!StringUtils.isEmpty(extract)) {
                    Boolean isPassed = true;
                    if (extract.startsWith("{")) {
                        Map<String, Object> maps = JsonUtils.json2Object(extract, Map.class);
                        isPassed = evaluate(validationExpr, Boolean.class, maps);
                    } else if (extract.startsWith("[")) {
                        List<Map> mapList = JsonUtils.json2ObjectList(extract, Map.class);
                        Map<String, Object> maps = new HashMap<>();
                        maps.put("root", mapList);
                        isPassed = evaluate(validationExpr, Boolean.class, maps);
                    }

                    if (!isPassed) {
                        Response overResp = new Response();
                        overResp.setCode("-1");
                        //TODO
                        overResp.setMessage(targetUsage.getErrorMsg());
                        return Tuple.of(targetUsage, null, null, overResp, -1);
                    }
                }
            }
        }

        //record check
        if (!StringUtils.isEmpty(limitationExpr)) {
            CheckUsageRequest checkUsageRequest = new CheckUsageRequest();
            if (!StringUtils.isEmpty(dryRunSizeExpr)) {
                if (dryRunSizeExpr.startsWith("BODY:")) {
                    dryRunSizeExpr = dryRunSizeExpr.substring(5);
                    String extract = httpPayloadExtractor.extract(request);
                    if (!StringUtils.isEmpty(extract)) {
                        String dryRunSize = "1";
                        if (extract.startsWith("{")) {
                            Map<String, Object> maps = JsonUtils.json2Object(extract, Map.class);
                            dryRunSize = evaluate(dryRunSizeExpr, String.class, maps);
                        } else if (extract.startsWith("[")) {
                            List<Map> mapList = JsonUtils.json2ObjectList(extract, Map.class);
                            Map<String, Object> maps = new HashMap<>();
                            maps.put("root", mapList);
                            dryRunSize = evaluate(dryRunSizeExpr, String.class, maps);
                        }
                        checkUsageRequest.setDryRunValue(dryRunSize);
                    }
                } else {
                    checkUsageRequest.setDryRunValue(dryRunSizeExpr);
                }
            }
                
            if(checkUsageRequest.getDryRunValue() == null) {
                checkUsageRequest.setDryRunValue("1");
            }
            checkUsageRequest.setUsage(targetUsage);
            checkUsageRequest.setTenant(teamCode);
            checkUsageRequest.setRecordKey(targetKey);
            Response checkUsage = usageApi.checkUsage(checkUsageRequest);
            if (!"1".equalsIgnoreCase(checkUsage.getCode())) {
                return Tuple.of(targetUsage, null, null, checkUsage, -1);
            }
        }

        return Tuple.of(targetUsage, targetKey, targetSize, null, 1);
    }

    /**
     * get var from request
     *
     * @param rawResource
     * @param targetResource
     * @param varMapping
     */
    private void preparePathEnv(String rawResource, String targetResource, Map<String, Object> varMapping) {
        String[] split = rawResource.split(" ");
        if (split.length == 2) {
            String url = split[1];
            extractKeyValueMapping(url, targetResource).forEach((k, v) -> {
                varMapping.put("path.".concat(k), v);
            });
        }
    }

    private String getValue(String expression, HttpServletRequest request) {
        //TODO 
        HttpPayloadExtractor httpPayloadExtractor = new HttpPayloadExtractor();
        return httpPayloadExtractor.extract(request);
    }

    private String transformResource(String rawResource, List<String> usageIds) {
        String[] split = rawResource.split(" ");
        if (split.length == 2) {
            String method = split[0];
            String url = split[1];
            if (url.contains("{") && url.contains("}")) {
                url = url.replaceAll("\\{[^/]+\\}", "*");
            }
            return url.concat("===").concat(method.toLowerCase()).concat("===").concat(Strings.join(usageIds, ','));
        }
        return null;
    }

    private <T> T evaluate(String keyExpr, Class<T> clz, Map<String, Object> varMapping) {
        ExpressionParser parser = new SpelExpressionParser();
        StandardEvaluationContext ec = new StandardEvaluationContext(varMapping);
        Expression expression = parser.parseExpression(keyExpr);
        T result = expression.getValue(ec, clz);
        return result;
    }
    
    public static Map<String, Object> parseQueryString(String queryString) {
        if (StringUtils.isEmpty(queryString)) {
            return null;
        }
        int index = queryString.indexOf("?");
        if (index >= 0) {
            queryString = queryString.substring(index + 1);
        }

        Map<String, Object> argMap = new HashMap<String, Object>();
        String[] queryArr = queryString.split("&");
        for (int i = 0; i < queryArr.length; i++) {
            String string = queryArr[i];
            String keyAndValue[] = string.split("=", 2);
            if (keyAndValue.length != 2) {
                argMap.put(keyAndValue[0], null);
            } else {
                argMap.put(keyAndValue[0], keyAndValue[1]);
            }
        }
        return argMap;
    }
}
