package com.xforceplus.ultraman.transfer.client.controller;

import com.google.common.collect.Lists;
import com.xforceplus.ultraman.metadata.global.common.enums.AppI18nResourceType;
import com.xforceplus.ultraman.metadata.jsonschema.enums.SchemaMetadataType;
import com.xforceplus.ultraman.metadata.jsonschema.pojo.version.SchemaAppVersion;
import com.xforceplus.ultraman.transfer.client.config.OqsSdkProperties;
import com.xforceplus.ultraman.transfer.client.front.runtime.AliyunOSSConstant;
import com.xforceplus.ultraman.transfer.client.front.runtime.MetadataAliyunOSSClient;
import com.xforceplus.ultraman.transfer.client.front.util.FrontUtil;
import com.xforceplus.ultraman.transfer.common.context.MetadataContextHolder;
import com.xforceplus.ultraman.transfer.common.util.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * 用户维护及查看socket信息
 *
 * @author leo
 * @version 0.1 2022/11/30 09:26
 * @since 1.8
 */
@RequestMapping("/metadatas")
@Slf4j
public class BocpClientOSSMetadataController {

    private OqsSdkProperties oqsSdkProperties;
    private MetadataAliyunOSSClient metadataClient;

    public BocpClientOSSMetadataController(OqsSdkProperties oqsSdkProperties, MetadataAliyunOSSClient metadataAliyunOSSClient) {
        this.oqsSdkProperties = oqsSdkProperties;
        this.metadataClient = metadataAliyunOSSClient;
    }

    /**
     * 获取环境应用版本信息
     * @return
     */
    @GetMapping("/version")
    @ResponseBody
    public ResponseEntity<String> envAppVersion() {
        String envCode = FrontUtil.getEnvCode(oqsSdkProperties.getAuth().getEnv(), oqsSdkProperties.getAuth().getEnvCode());
        if (StringUtils.isBlank(envCode)) {
            return ResponseEntity.badRequest().build();
        }
        return ResponseEntity.ok(
                metadataClient.loadEnvAppVersion(
                        envCode,
                        oqsSdkProperties.getAuth().getAppCode()
//                        MetadataContextHolder.appCode()
                )
        );
    }

    /**
     * 获取应用版本信息
     * @param version
     * @return
     */
    @GetMapping("/versions/{version:.+}")
    @ResponseBody
    public ResponseEntity<String> appVersion(@PathVariable String version) {
        return ResponseEntity.ok(
                metadataClient.loadAppVersionStr(MetadataContextHolder.appId(), version)
        );
    }

    /**
     * 获取列表视图全局配置
     * @param pageId
     * @param pageVersion
     * @return
     */
    @GetMapping("/pages/{pageId}/v/{pageVersion:.+}/_")
    @ResponseBody
    public ResponseEntity<String> pageInfo(@PathVariable Long pageId, @PathVariable String pageVersion) {
        return ResponseEntity.ok(
                metadataClient.loadMetadataSStr(MetadataContextHolder.appId(), SchemaMetadataType.PAGE, pageId, pageVersion, AliyunOSSConstant.UNDERSCORE)
        );
    }

    /**
     * 获取列表视图全局配置-租户定制
     * @param pageId
     * @param pageVersion
     * @param tenantCode
     * @return
     */
    @GetMapping("/pages/{pageId}/v/{pageVersion:.+}/tenants/{tenantCode}/_")
    @ResponseBody
    public ResponseEntity<String> tenantPageInfo(@PathVariable Long pageId, @PathVariable String pageVersion, @PathVariable String tenantCode) {
        return ResponseEntity.ok(
                metadataClient.loadTenantMetadataSStr(MetadataContextHolder.appId(), SchemaMetadataType.PAGE, pageId, pageVersion, tenantCode, AliyunOSSConstant.UNDERSCORE)
        );
    }

    /**
     * 获取列表视图绑定对象配置
     * @param pageId
     * @param pageVersion
     * @param boSettingId
     * @return
     */
    @GetMapping("/pages/{pageId}/v/{pageVersion:.+}/boSettings/{boSettingId}")
    @ResponseBody
    public ResponseEntity<String> pageSetting(@PathVariable Long pageId, @PathVariable String pageVersion, @PathVariable Long boSettingId) {
        return ResponseEntity.ok(
                metadataClient.loadMetadataSItemStr(MetadataContextHolder.appId(), SchemaMetadataType.PAGE, pageId, pageVersion, AliyunOSSConstant.BO_SETTINGS_DIR, boSettingId)
        );
    }

    /**
     * 获取列表视图绑定对象配置-租户定制
     * @param pageId
     * @param pageVersion
     * @param tenantCode
     * @param boSettingId
     * @return
     */
    @GetMapping("/pages/{pageId}/v/{pageVersion:.+}/tenants/{tenantCode}/boSettings/{boSettingId}")
    @ResponseBody
    public ResponseEntity<String> tenantPageSetting(@PathVariable Long pageId, @PathVariable String pageVersion, @PathVariable String tenantCode, @PathVariable Long boSettingId) {
        return ResponseEntity.ok(
                metadataClient.loadTenantMetadataSItemStr(MetadataContextHolder.appId(), SchemaMetadataType.PAGE, pageId, pageVersion, tenantCode, AliyunOSSConstant.BO_SETTINGS_DIR, boSettingId)
        );
    }

    /**
     * 获取表单视图配置
     * @param formId
     * @param formVersion
     * @return
     */
    @GetMapping("/forms/{formId}/v/{formVersion:.+}/_")
    @ResponseBody
    public ResponseEntity<String> formInfo(@PathVariable Long formId, @PathVariable String formVersion) {
        return ResponseEntity.ok(
                metadataClient.loadMetadataSStr(MetadataContextHolder.appId(), SchemaMetadataType.FORM, formId, formVersion, AliyunOSSConstant.UNDERSCORE)
        );
    }

    /**
     * 获取表单视图配置-租户定制
     * @param formId
     * @param formVersion
     * @param tenantCode
     * @return
     */
    @GetMapping("/forms/{formId}/v/{formVersion:.+}/tenants/{tenantCode}/_")
    @ResponseBody
    public ResponseEntity<String> tenantFormInfo(@PathVariable Long formId, @PathVariable String formVersion, @PathVariable String tenantCode) {
        return ResponseEntity.ok(
                metadataClient.loadTenantMetadataSStr(MetadataContextHolder.appId(), SchemaMetadataType.FORM, formId, formVersion, tenantCode, AliyunOSSConstant.UNDERSCORE)
        );
    }

    /**
     * 获取新视图配置
     * @param pageSettingId
     * @param pageSettingVersion
     * @return
     */
    @GetMapping("/pageSettings/{pageSettingId}/v/{pageSettingVersion:.+}/_")
    @ResponseBody
    public ResponseEntity<String> newPageInfo(@PathVariable Long pageSettingId, @PathVariable String pageSettingVersion) {
        return ResponseEntity.ok(
                metadataClient.loadMetadataSStr(MetadataContextHolder.appId(), SchemaMetadataType.PAGE_SETTING, pageSettingId, pageSettingVersion, AliyunOSSConstant.UNDERSCORE)
        );
    }

    /**
     * 获取新视图配置-租户定制
     * @param pageSettingId
     * @param pageSettingVersion
     * @param tenantCode
     * @return
     */
    @GetMapping("/pageSettings/{pageSettingId}/v/{pageSettingVersion:.+}/tenants/{tenantCode}/_")
    @ResponseBody
    public ResponseEntity<String> tenantNewPageInfo(@PathVariable Long pageSettingId, @PathVariable String pageSettingVersion, @PathVariable String tenantCode) {
        return ResponseEntity.ok(
                metadataClient.loadTenantMetadataSStr(MetadataContextHolder.appId(), SchemaMetadataType.PAGE_SETTING, pageSettingId, pageSettingVersion, tenantCode, AliyunOSSConstant.UNDERSCORE)
        );
    }

    /**
     * 获取流配置
     * @param flowId
     * @param flowVersion
     * @return
     */
    @GetMapping("/flows/{flowId}/v/{flowVersion:.+}/_")
    @ResponseBody
    public ResponseEntity<String> flowSetting(@PathVariable Long flowId, @PathVariable String flowVersion) {
        return ResponseEntity.ok(
                metadataClient.loadMetadataSStr(MetadataContextHolder.appId(), SchemaMetadataType.FLOW, flowId, flowVersion, AliyunOSSConstant.UNDERSCORE)
        );
    }

    /**
     * 获取流配置-租户定制
     * @param flowId
     * @param flowVersion
     * @param tenantCode
     * @return
     */
    @GetMapping("/flows/{flowId}/v/{flowVersion:.+}/tenants/{tenantCode}/_")
    @ResponseBody
    public ResponseEntity<String> tenantFlowSetting(@PathVariable Long flowId, @PathVariable String flowVersion, @PathVariable String tenantCode) {
        return ResponseEntity.ok(
                metadataClient.loadTenantMetadataSStr(MetadataContextHolder.appId(), SchemaMetadataType.FLOW, flowId, flowVersion, tenantCode, AliyunOSSConstant.UNDERSCORE)
        );
    }

    /**
     * 获取流前端配置
     * @param flowId
     * @param flowVersion
     * @return
     */
    @GetMapping("/flows/{flowId}/v/{flowVersion:.+}/_front")
    @ResponseBody
    public ResponseEntity<String> flowFrontSetting(@PathVariable Long flowId, @PathVariable String flowVersion) {
        return ResponseEntity.ok(
                metadataClient.loadMetadataSStr(MetadataContextHolder.appId(), SchemaMetadataType.FLOW, flowId, flowVersion, AliyunOSSConstant.FRONT)
        );
    }

    /**
     * 获取流前端配置-租户定制
     * @param flowId
     * @param flowVersion
     * @param tenantCode
     * @return
     */
    @GetMapping("/flows/{flowId}/v/{flowVersion:.+}/tenants/{tenantCode}/_front")
    @ResponseBody
    public ResponseEntity<String> tenantFlowFrontSetting(@PathVariable Long flowId, @PathVariable String flowVersion, @PathVariable String tenantCode) {
        return ResponseEntity.ok(
                metadataClient.loadTenantMetadataSStr(MetadataContextHolder.appId(), SchemaMetadataType.FLOW, flowId, flowVersion, tenantCode, AliyunOSSConstant.FRONT)
        );
    }

    /**
     * 获取国际化语言
     *
     * @param appI18nLocaleVersion
     * @return
     */
    @GetMapping("/appI18nLocales/v/{appI18nLocaleVersion:.+}/_")
    @ResponseBody
    public ResponseEntity<String> appI18nLocale(@PathVariable String appI18nLocaleVersion) {
        return ResponseEntity.ok(
                metadataClient.loadMetadataStr(MetadataContextHolder.appId(), SchemaMetadataType.APP_I18N_LOCALE, appI18nLocaleVersion)
        );
    }

    /**
     * 获取国际化资源
     * @param appI18nResourceId
     * @param appI18nResourceVersion
     * @return
     */
    @GetMapping("/appI18nResources/{appI18nResourceId}/v/{appI18nResourceVersion:.+}/_")
    @ResponseBody
    public ResponseEntity<String> appI18nResource(@PathVariable Long appI18nResourceId, @PathVariable String appI18nResourceVersion) {
        return ResponseEntity.ok(
                metadataClient.loadMetadataSStr(MetadataContextHolder.appId(), SchemaMetadataType.APP_I18N_RESOURCE, appI18nResourceId, appI18nResourceVersion, AliyunOSSConstant.UNDERSCORE)
        );
    }

    /**
     * 获取国际化资源-通用结构
     * @param appI18nResourceId
     * @param appI18nResourceVersion
     * @return
     */
    @GetMapping("/appI18nResources/{appI18nResourceId}/v/{appI18nResourceVersion:.+}/_i18n")
    @ResponseBody
    public ResponseEntity<String> appI18nResourceCommonDataStruct(@PathVariable Long appI18nResourceId, @PathVariable String appI18nResourceVersion) {
        return ResponseEntity.ok(
                metadataClient.loadMetadataSStr(MetadataContextHolder.appId(), SchemaMetadataType.APP_I18N_RESOURCE, appI18nResourceId, appI18nResourceVersion, AliyunOSSConstant.I18N_DATA_STRUCT)
        );
    }

    /**
     * 获取国际化资源-二开逻辑
     * @return
     */
    @GetMapping("/appI18nResources/customs")
    @ResponseBody
    public ResponseEntity<List<String>> getCustomI18nResources() {
        String envCode = FrontUtil.getEnvCode(oqsSdkProperties.getAuth().getEnv(), oqsSdkProperties.getAuth().getEnvCode());
        if (StringUtils.isBlank(envCode)) {
            return ResponseEntity.badRequest().build();
        }
        String envAppVersion = metadataClient.loadEnvAppVersion(envCode, oqsSdkProperties.getAuth().getAppCode());
        SchemaAppVersion schemaAppVersion = JsonUtils.json2Object(envAppVersion, SchemaAppVersion.class);
        if (null == schemaAppVersion) {
            return ResponseEntity.badRequest().build();
        }
        return ResponseEntity.ok(
                Optional.ofNullable(schemaAppVersion.getAppI18nResources()).map(map ->
                        map.values().stream()
                                .filter(o -> AppI18nResourceType.CUSTOM.equals(o.getType()))
                                .map(o -> metadataClient.loadMetadataSStr(MetadataContextHolder.appId(), SchemaMetadataType.APP_I18N_RESOURCE, Long.valueOf(o.getId()), o.getVersion(), AliyunOSSConstant.UNDERSCORE))
                                .collect(Collectors.toList())
                ).orElse(Lists.newArrayList()));
    }
}
